"""Writer for ADCIRC fort.15 control files."""

# 1. Standard Python modules
import datetime
from io import StringIO
import os
import shutil
import subprocess
import sys

# 2. Third party modules
from harmonica.tidal_constituents import Constituents
import numpy as np

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint import read_grid_from_file
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.target_type import TargetType
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import datetime_to_qdatetime, ISO_DATETIME_FORMAT

# 4. Local modules
from xms.adcirc.data.sim_data import REG_KEY_SUPPRESS_HOTSTART, REG_KEY_SUPPRESS_NONPERIODIC
from xms.adcirc.dmi.fort13_data_getter import write_fort13_data_json
from xms.adcirc.dmi.fort15_data_getter import Fort15DataGetter
from xms.adcirc.dmi.fort22_data_getter import write_fort22_data_json
from xms.adcirc.feedback.xmlog import report_error, XmLog


class Fort15Writer:
    """Export an ADCIRC simulation."""
    def __init__(self, filename, query=None, xms_data=None, template=False):
        """Construct the writer, Query SMS for data if not passed in.

        Args:
            filename (:obj:`str`): File location to export the fort.15 to
            query (:obj:`Query`): For communicating with XMS. Will not be used if `xms_data` is provided. `xms_data`
                and `query` should not be provided when this script is called from XMS as a simulation-level export for
                a model run. Tests should provide `xms_data` and not `query`. Other entry points such as component
                dialogs should provide `query` at the ADCIRC simulation component level but not `xms_data`.
            xms_data (:obj:`dict`): All the data required to export. If not provided, will Query XMS for it.
                    {
                        'projection': :obj:`data_objects.parameters.Projection`,

                        'bc_datas': :obj: list[`MappedBcData`],

                        'main_bc': MappedBcData,

                        'sim_data': :obj:`SimData`,

                        'tidal_data': :obj:`[MappedTidalData]`,

                        'domain_name': :obj:`str`,

                        'wind_grid': :obj:`str` (CoGrid file for wind grid),

                        'station_comp': :obj:`StationComponent`,

                        'station_pts': :obj:`{id: data_objects.parameters.Point}`,

                        'flow_data': :obj:`MappedFlowData`
                    }
            template (bool): If True will export a templatized fort.15 for CSTORM
        """
        XmLog().instance.info('Exporting ADCIRC simulation control file (fort.15)...')
        self._write_netcdf_opts = False  # If any output is in NetCDF format, we need to write the NetCDF parameters.
        self._filename = filename
        self._template = template  # This makes things messy, but the code is already ugly here.
        self._proj_sys = 1
        self._nws = 0
        self._ss = StringIO()
        self._xms_data = xms_data
        self._sim_export = xms_data is None and query is None
        if xms_data is not None and 'sim_export' in xms_data:
            self._sim_export = xms_data['sim_export']
        self._data = None
        if self._xms_data is None:
            self._get_xms_data(query)
        self._xms_data['template'] = self._template
        if 'sim_data' in self._xms_data:
            self._data = self._xms_data['sim_data']  # For convenience, use it a lot.
        self._domain = None
        self._wind_grid = None
        self._main_bc = self._xms_data.get('main_bc')
        self._suppress_missing_hotstart_errors = False
        self._suppress_missing_nonperiodic_errors = False
        self._read_registry_settings()

    @property
    def domain(self):
        """Make sure we only read the grid file once."""
        if self._domain is None:
            co_grid = read_grid_from_file(self._xms_data['cogrid_file'])
            self._domain = co_grid.ugrid
        return self._domain

    @property
    def wind_grid(self):
        """Make sure we only read the wind grid file once."""
        if self._wind_grid is None and self._xms_data['wind_grid']:
            co_grid = read_grid_from_file(self._xms_data['wind_grid'])
            self._wind_grid = co_grid
        return self._wind_grid

    def _get_xms_data(self, query):
        """Query XMS for all the data needed to export the simulation.

        Args:
            query (:obj:`Query`): Object for requesting data from XMS
        """
        self._xms_data = {}
        try:
            getter = Fort15DataGetter(query, self._xms_data)
            getter.retrieve_data()
            self._xms_data['template'] = self._template
        except Exception as ex:
            msg = 'Unable to retrieve ADCIRC data from XMS for exporting.'
            XmLog().instance.exception(msg)
            if self._sim_export:
                XmEnv.report_error(msg)
                XmEnv.report_error(ex)

    def _launch_fort13(self):
        """Launch the fort.13 export script in parallel.

        Returns:
            (:obj:`Union[Popen, None]`): The Popen process handle, or None if we did not launch the parallel process
        """
        fort13_proccess = None
        if self._sim_export and self._xms_data.get('att_names'):
            XmLog().instance.info('Starting fort.13 export in parallel process...')
            write_fort13_data_json(self._xms_data)  # Write the XMS data JSON file so script doesn't need Query
            fort13_proccess = subprocess.Popen(
                [
                    sys.executable,  # Python executable
                    os.path.normpath(
                        os.path.join(
                            os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'fort13_parallel.py'
                        )
                    ),  # Path to script
                    os.path.normpath(os.path.join(os.path.dirname(self._filename), 'fort.13')),  # Output location
                ],
                env=os.environ,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE
            )
        return fort13_proccess

    def _launch_fort14(self):
        """Launch the fort.14 export script in parallel.

        Returns:
            (:obj:`Union[Popen, None]`): The Popen process handle, or None if we did not launch the parallel process
        """
        fort14_proccess = None
        if self._sim_export:
            XmLog().instance.info('Starting fort.14 export in parallel process...')
            args = [
                sys.executable,  # Python executable
                os.path.normpath(
                    os.path.join(os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'fort14_parallel.py')
                ),  # Path to script
                os.path.normpath(os.path.join(os.path.dirname(self._filename), 'fort.14')),  # Output location
                os.path.normpath(self._xms_data['cogrid_file']),  # CoGrid file
                self._xms_data['domain_name'],  # Name of the domain mesh
            ]
            # Can be n number of levee BC coverages now.
            mapped_bc_datas = [
                os.path.normpath(mapped_bc_data._filename) for mapped_bc_data in self._xms_data['bc_datas']
            ]
            args.extend(mapped_bc_datas)
            fort14_proccess = subprocess.Popen(
                args=args, env=os.environ, stdout=subprocess.PIPE, stderr=subprocess.PIPE
            )
        return fort14_proccess

    def _launch_fort20(self):
        """Launch the fort.14 export script in parallel.

        Returns:
            (:obj:`Union[Popen, None]`): The Popen process handle, or None if we did not launch the parallel process
        """
        fort20_process = None
        if self._sim_export:
            XmLog().instance.info('Starting fort.20 export in parallel process...')
            args = [
                sys.executable,  # Python executable
                os.path.normpath(
                    os.path.join(os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'fort20_parallel.py')
                ),  # Path to script
                os.path.normpath(os.path.join(os.path.dirname(self._filename), 'fort.20')),  # Output location
            ]
            # Can be n number of levee BC coverages now.
            mapped_bc_datas = [
                os.path.normpath(mapped_bc_data._filename) for mapped_bc_data in self._xms_data['bc_datas']
            ]
            args.extend(mapped_bc_datas)
            fort20_process = subprocess.Popen(
                args=args, env=os.environ, stdout=subprocess.PIPE, stderr=subprocess.PIPE
            )
        return fort20_process

    def _launch_fort22(self):
        """Launch the fort.22 export script in parallel.

        Returns:
            (:obj:`Union[Popen, None]`): The Popen process handle, or None if we did not launch the parallel process
        """
        fort22_proccess = None
        if self._sim_export and self._data.wind.attrs['NWS'] != 0:
            XmLog().instance.info('Starting fort.22 export in parallel process...')
            if not write_fort22_data_json(self._xms_data):  # Write the XMS data JSON file so script doesn't need Query
                return None
            fort22_proccess = subprocess.Popen(
                [
                    sys.executable,  # Python executable
                    os.path.normpath(
                        os.path.join(
                            os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'fort22_parallel.py'
                        )
                    ),  # Path to script
                    os.path.normpath(os.path.join(os.path.dirname(self._filename), 'fort.22')),  # Output location
                ],
                env=os.environ,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE
            )
        return fort22_proccess

    def _read_registry_settings(self):
        """Read settings from registry for suppressing error messages."""
        settings = SettingsManager()
        self._suppress_missing_hotstart_errors = settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_HOTSTART, 0) == 1
        suppress_nonperiodic = settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_NONPERIODIC, 0)
        self._suppress_missing_nonperiodic_errors = suppress_nonperiodic == 1

    def _copy_hotstart_files(self):
        """Copy the hot start files to the export location if they do not already exist from a previous run."""
        # only do this if this is a hot start run
        file_type = self._data.general.attrs['IHOT']
        if file_type and file_type != 0:
            XmLog().instance.info('Copying hot start files to export location...')
            # copy the main hot start file
            hotstart_file = os.path.join(self._data.info.attrs['proj_dir'], self._data.general.attrs['IHOT_file'])
            if not os.path.isfile(hotstart_file):
                if not self._suppress_missing_hotstart_errors:  # Don't report error if suppress in registry setting.
                    hotstart_file = hotstart_file if XmEnv.xms_environ_running_tests(
                    ) != 'TRUE' else os.path.basename(hotstart_file)
                    report_error(f'Unable to copy hot start file to export location: {hotstart_file}', self._sim_export)
            else:
                if file_type == 17:
                    new_file = 'fort.17'
                elif file_type == 67:
                    new_file = 'fort.67'
                elif file_type == 68:
                    new_file = 'fort.68'
                elif file_type == 367 or file_type == 567:
                    new_file = 'fort.67.nc'
                elif file_type == 368 or file_type == 568:
                    new_file = 'fort.68.nc'
                else:
                    return
                new_file = os.path.join(os.path.dirname(self._filename), new_file)
                if not os.path.exists(new_file) or not os.path.samefile(hotstart_file, new_file):
                    shutil.copyfile(hotstart_file, new_file)

                # look for a maxele and a maxvel to copy
                file_dir = os.path.dirname(hotstart_file)
                maxele = os.path.join(file_dir, 'maxele.63')
                if os.path.isfile(maxele):
                    new_file = os.path.join(os.path.dirname(self._filename), 'maxele.63')
                    if not os.path.exists(new_file) or not os.path.samefile(maxele, new_file):
                        shutil.copyfile(maxele, new_file)
                maxvel = os.path.join(file_dir, 'maxvel.63')
                if os.path.isfile(maxvel):
                    new_file = os.path.join(os.path.dirname(self._filename), 'maxvel.63')
                    if not os.path.exists(new_file) or not os.path.samefile(maxvel, new_file):
                        shutil.copyfile(maxvel, new_file)

    def _get_variable_str(self, variable, attrs):
        """Get a variable string to write to the fort.15, templatized for CSTORM if needed.

        Args:
            variable (str): name of the variable
            attrs (dict): dict of the attrs containing the variable

        Returns:
            str: the string that should be written to the fort.15
        """
        if self._template:
            return f'%{variable}%'
        else:
            return attrs[variable]

    def _write_run_options(self):
        """Write the run options section."""
        self._ss.write(
            f"{self._get_variable_str('RUNDES', self._data.general.attrs):<40} "
            f"{'! 32 CHARACTER ALPHANUMERIC RUN DESCRIPTION'}\n"
            f"{self._get_variable_str('RUNID', self._data.general.attrs):<40} "
            f"{'! 24 CHARACTER ALPHANUMERIC RUN IDENTIFICATION'}\n"
            f"{self._data.general.attrs['NFOVER']:<40} {'! NFOVER - NONFATAL ERROR OVERRIDE OPTION'}\n"
            f"{self._data.general.attrs['NABOUT']:<40} {'! NABOUT - ABREVIATED OUTPUT OPTION PARAMETER'}\n"
            f"{self._data.general.attrs['NSCREEN']:<40} {'! NSCREEN - OUTPUT TO UNIT 6 PARAMETER'}\n"
            f"{self._get_variable_str('IHOT', self._data.general.attrs):<40} "
            f"{'! IHOT - HOT START OPTION PARAMETER'}\n"
        )
        coord_sys = self._xms_data['projection'].coordinate_system
        if coord_sys == "GEOGRAPHIC":
            self._proj_sys = 2  # 1=Cartesian, 2=Spherical
        self._ss.write(f"{self._proj_sys:<40} {'! ICS - COORDINATE SYSTEM OPTION PARAMETER'}\n")
        model_type = self._data.formulation.attrs['IM']
        self._ss.write(
            f"{model_type:<40} {'! IM - MODEL RUN TYPE: 0,10,20,30 = 2DDI, 1,11,21,31 = 3D(VS), 2 = 3D(DSS)'}\n"
        )
        if model_type == 21:  # Only write IDEN if IM is set to Baroclinic 3D
            self._ss.write(
                f"{self._data.formulation.attrs['IDEN']:<40} {'! IDEN - FORM OF DENSITY FORCING IN A 3D RUN'}\n"
            )

    def _write_noli_options(self):
        """Write lines from NOLIBF to NOLICAT."""
        nolibf = self._data.general.attrs['NOLIBF']
        if nolibf > 2 or self._data.formulation.attrs['TAU0'] == -3:
            # Friction specified with nodal atttribute dataset
            self._ss.write(f"{'1':<40} {'! NOLIBF - NONLINEAR BOTTOM FRICTION OPTION'}\n")
        else:
            self._ss.write(f"{nolibf:<40} {'! NOLIBF - NONLINEAR BOTTOM FRICTION OPTION'}\n")
        nolifa = self._data.formulation.attrs['NOLIFA']
        nolica = self._data.formulation.attrs['NOLICA']
        self._ss.write(
            f"{nolifa:<40} {'! NOLIFA - OPTION TO INCLUDE FINITE AMPLITUDE TERMS'}\n"
            f"{nolica:<40} {'! NOLICA - OPTION TO INCLUDE CONVECTIVE ACCELERATION TERMS'}\n"
        )
        if nolifa != 0 or nolica != 0:  # nolicat must be 1
            self._ss.write(f"{'1':<40} {'! NOLICAT - OPTION TO CONSIDER TIME DERIVATIVE OF CONV ACC TERMS'}\n")
        else:
            self._ss.write(
                f"{self._data.formulation.attrs['NOLICAT']:<40} "
                f"{'! NOLICAT - OPTION TO CONSIDER TIME DERIVATIVE OF CONV ACC TERMS'}\n"
            )

    def _write_nodal_attributes(self):
        """Write the nodal attribute lines."""
        node_atts = self._data.get_enabled_nodal_atts()
        self._ss.write(f"{len(node_atts):<40} {'! NWP - Number of nodal attributes.'}\n")
        for node_att in node_atts:
            self._ss.write(f'{node_att}\n')

    def _write_nws(self):
        """Write the NCOR, NTIP, and NWS lines. Biggest thing here is computing the true NWS value."""
        self._ss.write(
            f"{self._data.general.attrs['NCOR']:<40} {'! NCOR - VARIABLE CORIOLIS IN SPACE OPTION PARAMETER'}\n"
        )
        # If we have mapped tidal constituents, use tidal potential forcing
        # TODO: Do we want to support NTIP=2 and write the Self Attraction/Earth Load Tide Forcing File (fort.24)?
        ntip = 1 if self._xms_data['tidal_data'] else 0
        self._ss.write(
            f"{ntip:<40} {'! NTIP - TIDAL POTENTIAL OPTION PARAMETER'}\n"
            f"{'%NWS%' if self._template else self._nws:<40} "
            f"{'! NWS - WIND STRESS AND BAROMETRIC PRESSURE OPTION PARAMETER'}\n"
        )

    def _write_ramp_and_timing_options(self):
        """Write NRAMP - REFTIM lines."""
        self._ss.write(f"{self._data.timing.attrs['NRAMP']:<40} {'! NRAMP - RAMP FUNCTION OPTION'}\n")
        v_units = self._xms_data['projection'].vertical_units
        if v_units == 'METERS' or v_units == 'CENTIMETERS':
            self._ss.write(f"{'9.80665':<40} {'! G - ACCELERATION DUE TO GRAVITY - DETERMINES UNITS'}\n")
        else:
            self._ss.write(f"{'32.2000':<40} {'! G - ACCELERATION DUE TO GRAVITY - DETERMINES UNITS'}\n")
        tau0 = self._data.formulation.attrs['TAU0']
        if tau0 == 1:  # tau0 specified
            self._ss.write(
                f"{self._data.formulation.attrs['TAU0_specified']:<40} {'! TAU0 - WEIGHTING FACTOR IN GWCE'}\n"
            )
        else:
            self._ss.write(f"{tau0:<40} {'! TAU0 - WEIGHTING FACTOR IN GWCE'}\n")
            if tau0 == -5:  # export domain for for FullDomainTimeVaryingTau0
                domain = (
                    f"{self._data.formulation.attrs['Tau0FullDomainMin']} "
                    f"{self._data.formulation.attrs['Tau0FullDomainMax']}"
                )
                self._ss.write(f"{domain:<40} {'! Tau0FullDomainMin, Tau0FullDomainMax'}\n")

        dtdp = self._data.timing.attrs['DTDP']
        if self._template:
            dtdpstr = f'%DT={dtdp}%'
            dtdpstr = f'{dtdpstr:<40}'
        else:
            dtdpstr = f'{dtdp:<40}'
        # STATIM and REFTIM are ignored by ADCIRC. They are just relics.
        self._ss.write(
            f"{dtdpstr} {'! DTDP - TIME STEP (IN SECONDS)'}\n"
            f"{0.0:<40} {'! STATIM - STARTING SIMULATION TIME IN DAYS'}\n"
            f"{0.0:<40} {'! REFTIME - REFERENCE TIME (IN DAYS) FOR NODAL FACTORS AND EQUILIBRIUM ARGS'}\n"
        )

    def _write_wtiminc(self):
        """Write the WTIMINC line in all its variants."""
        wtiminc = ''
        cicetiminc = ''
        rstiminc = ''
        nws_simple = abs(self._nws) % 100

        # We don't claim to support NWS=3 anymore. We couldn't find any reliable test cases, so we assume no one is
        # using it. Should export without error, but we can't guarantee the files are correct. See Mantis issue
        # 14382.
        if nws_simple == 3:
            XmLog().instance.warning(
                'NWS=3 is no longer supported in the SMS interface. Please contact technical support to request this '
                'feature.'
            )

        # Don't write the wtiminc card if NWS = 0. Don't write it if NWS = 1, but we still need to write
        #  the fort.22.
        if nws_simple in [0, 1, 11]:
            return

        if self._template:  # Don't need to compute WTIMINC or RSTIMINC if a template for CSTORM
            self._ss.write(f"{'%WTIMINC% %RSTIMINC%':<40} ! WTIMINC and RSTIMINC in (seconds)\n")
            return

        if self._data.wind.attrs['use_ice']:
            cicetiminc = self._data.wind.attrs['CICE_TIMINC']
        if self._data.wind.attrs['wave_radiation'] != 0:
            rstiminc = self._data.wind.attrs['RSTIMINC']
        doc = "! WTIMINC\n"

        if nws_simple in [2, 4, 5, 10, 12, 15, 16]:  # wind data specified as a dataset on the mesh
            wtiminc = f"{self._data.wind.attrs['WTIMINC']}"
        elif nws_simple in [3, 6]:  # WTIMINC - Wind data on a grid
            grid_reftime = datetime_to_qdatetime(
                datetime.datetime.strptime(self._data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT)
            )
            if nws_simple == 3:
                # NWS=3 has an extra line
                doc = "! IREFYR, IREFMO, IREFDAY, IREFHR, IREFMIN, REFSEC"
                forcing_str = ""
                if rstiminc:
                    forcing_str = f' {rstiminc}'
                    doc += ', RSTIMINC'
                if cicetiminc:
                    forcing_str += f' {cicetiminc}'
                    doc += ', CICE_TIMEINC'
                doc += '\n'
                year_str = f'{grid_reftime.date().year()}'
                retime_str = (
                    f"{year_str[-2:]} {grid_reftime.date().month():0>2d} "
                    f"{grid_reftime.date().day():0>2d} {grid_reftime.time().hour():0>2d} "
                    f"{grid_reftime.time().minute():0>2d} {grid_reftime.time().second():0>2d}{forcing_str}"
                )
                self._ss.write(f"{retime_str:<40} {doc}")

            # Write out the wind grid definition.
            isizes = self.wind_grid.locations_x  # These are offsets from the origin
            jsizes = self.wind_grid.locations_y
            nwlat = len(jsizes) - 1
            nwlon = len(isizes) - 1
            wlatinc = jsizes[1] - jsizes[0]
            wloninc = isizes[1] - isizes[0]
            origin = self.wind_grid.origin
            wlonmin = origin[0] + (wloninc * 0.5)  # adjust to the center of the cell
            wlatmax = origin[1] + jsizes[-1]
            wlatmax -= wlatinc * 0.5  # adjust to center of the cell

            wtiminc = f'{nwlat} {nwlon} {wlatmax:.6f} {wlonmin:.6f} {wlatinc:.6f} {wloninc:.6f} ' \
                      f'{self._data.wind.attrs["WTIMINC"]}'
            doc = "! NWLAT, NWLON, WLATMAX, WLONMIN, WLATINC, WLONINC, WTIMINC\n"
        elif nws_simple in [8, 19, 20]:  # WTIMINC - Wind track coverage
            # We need to write the interpolation reference date to the fort.15, not the time of the first node
            # in the storm track.
            storm_start = datetime_to_qdatetime(
                datetime.datetime.strptime(self._data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT)
            )
            wtiminc = f"{storm_start.date().year()} {storm_start.date().month():0>2d} {storm_start.date().day():0>2d}" \
                      f" {storm_start.time().hour():0>2d} 1 {self._data.wind.attrs['bladj']}"
            if nws_simple == 20:
                doc = "! YYYY MM DD HH24 StormNumber BLAdj geofactor\n"
                wtiminc += f" {self._data.wind.attrs['geofactor']}"
            else:
                doc = "! YYYY MM DD HH24 StormNumber BLAdj\n"
        if rstiminc and nws_simple != 3:  # If NWS=3, written on previous line.
            wtiminc = f"{wtiminc} {rstiminc}"
        if cicetiminc and nws_simple != 3:  # If NWS=3, written on previous line.
            wtiminc = f"{wtiminc} {cicetiminc}"
        self._ss.write(f"{wtiminc:<40} {doc}")

    def _write_dramp(self):
        """Write the RUNDAY and DRAMP lines."""
        rnday_doc = '! RNDAY - TOTAL LENGTH OF SIMULATION (IN DAYS)\n'
        if self._template:  # Yet another special case for CSTORM templates
            self._ss.write(f'{"%RNDAY%":<40} {rnday_doc}')
        else:
            self._ss.write(f"{self._get_variable_str('RUNDAY', self._data.timing.attrs):<40} {rnday_doc}")
        params = [0.0]  # Ensure DRAMP is 0.0 if not enabled
        doc = '! DRAMP - DURATION OF RAMP FUNCTION (IN DAYS)'
        nramp = self._data.timing.attrs['NRAMP']
        if nramp > 0:
            params = [self._data.timing.attrs['DRAMP']]  # Get the real DRAMP value
            if nramp > 1:
                params.extend([self._data.timing.attrs['DRAMPExtFlux'], self._data.timing.attrs['FluxSettlingTime']])
                doc += ", DRAMPExtFlux, FluxSettlingTime"
                if nramp > 2:
                    params.append(self._data.timing.attrs['DRAMPIntFlux'])
                    doc += ", DRAMPIntFlux"
                    if nramp > 3:
                        params.append(self._data.timing.attrs['DRAMPElev'])
                        doc += ", DRAMPElev"
                        if nramp > 4:
                            params.append(self._data.timing.attrs['DRAMPTip'])
                            doc += ", DRAMPTip"
                            if nramp > 5:
                                params.append(self._data.timing.attrs['DRAMPMete'])
                                doc += ", DRAMPMete"
                                if nramp > 6:
                                    params.append(self._data.timing.attrs['DRAMPWRad'])
                                    doc += ", DRAMPWRad"
                                    if nramp > 7:
                                        params.append(self._data.timing.attrs['DUnRampMete'])
                                        doc += ", DUnRampMete"
        self._ss.write(f'{" ".join(format(param, ".5f") for param in params):<40} {doc}\n')

    def _write_equation_options(self):
        """Write equation option lines."""
        model_type = self._data.formulation.attrs['IM']
        if model_type > 100000:
            weighting_factors = "0.000000 1.000000 0.000000"
        else:
            weighting_factors = (
                f"{self._data.formulation.attrs['A00']:.6f} "
                f"{self._data.formulation.attrs['B00']:.6f} "
                f"{self._data.formulation.attrs['C00']:.6f} "
            )
        self._ss.write(f"{weighting_factors:<40} {'! TIME WEIGHTING FACTORS FOR THE GWCE EQUATION'}\n")
        nolifa = self._data.formulation.attrs['NOLIFA']
        if nolifa in [0, 1]:
            self._ss.write(
                f"{self._data.formulation.attrs['H0']:<40} {'! H0 - MINIMUM WATER DEPTH AND DRYING/WETTING OPTIONS'}\n"
            )
        else:
            # Two dummy integers needed by ADCIRC code for backwards compatibility, will be ignored.
            doc = '! H0, NODEDRYMIN, NODEWETMIN, VELMIN - MINIMUM WATER DEPTH AND DRYING/WETTING OPTIONS\n'
            card_vals = f"{self._data.formulation.attrs['H0']} 12 12 {self._data.formulation.attrs['VELMIN']}"
            self._ss.write(f"{card_vals:<40} {doc}")

        if self._proj_sys != 2:
            slam0_sfea0 = '0.000000 0.000000'  # not used, but has to be written
        else:
            if self._data.general.attrs['calc_center_coords']:  # Calculate center of the domain (roughly)
                domain_extents = self.domain.extents
                slam0 = (domain_extents[0][0] + domain_extents[1][0]) / 2
                sfea0 = (domain_extents[0][1] + domain_extents[1][1]) / 2
                slam0_sfea0 = f'{slam0:.6f} {sfea0:.6f}'
            else:
                slam0_sfea0 = f"{self._data.general.attrs['SLAM0']:.6f} {self._data.general.attrs['SFEA0']:.6f}"
        doc = '! SLAM0, SFEA0 - LONGITUDE AND LATITUDE ON WHICH THE CPP COORDINATE PROJECTION IS CENTERED\n'
        self._ss.write(f'{slam0_sfea0:<40} {doc}')

        ffactor = self._data.general.attrs['CF']
        nolibf = self._data.general.attrs['NOLIBF']
        if nolibf > 2:
            # Friction specified with nodal atttribute dataset, set CF to 0.0
            ffactor = 0.0
        doc = '! TAU\n' if nolibf == 0 else '! CF\n'
        if nolibf == 2:
            ffactor = f"{ffactor:.6f} {self._data.general.attrs['HBREAK']:.6f} " \
                      f"{self._data.general.attrs['FTHETA']:.6f} {self._data.general.attrs['FGAMMA']:.6f}"

            doc = '! CF, HBREAK, FTHETA, FGAMMA\n'
        self._ss.write(f'{ffactor:<40} {doc}')

        if model_type in [0, 1, 10]:
            doc = "! ESLM - SPATIALLY CONSTANT HORIZONTAL EDDY VISCOSITY FOR THE MOMENTUM EQUATIONS\n"
            val = self._data.formulation.attrs['ESLM']
            if model_type == 10:  # Not in the interface. Only can be set if read in.
                val += f" {self._data.formulation.attrs['ESLC']}"
            self._ss.write(f'{val:<40} {doc}')
        self._ss.write(f"{self._data.general.attrs['CORI']:<40} {'! CORI - CONSTANT CORIOLIS COEFFICIENT'}\n")

    def _write_constituents(self):
        """Write tidal potential, tidal forcing, and flow forcing constituents.

        ANGINN parameter line also happens to be in the middle of here.
        """
        self._write_tidal_potential()
        self._write_tidal_forcing()
        self._ss.write(f"{self._data.general.attrs['ANGINN']:<40} {'! ANGINN - MINIMUM ANGLE FOR TANGENTIAL FLOW'}\n")
        self._write_flow_forcing()

    def _write_tidal_potential(self):
        """Write tidal potential lines.

        Currently, if tidal constituents are applied to a simulation, those constituents will be used for tidal
        potential, even if there are no ocean boundary nodes.

        TODO: Do we want to support NTIP=2 (specified on previous line in fort.15)? If so, we need to write the
          Self Attraction/Earth Load Tide Forcing File (fort.24). Not sure if we set NTIF to be 0 here in
          that case.
        """
        num_cons = sum([data.cons.sizes['con'] for data in self._xms_data['tidal_data']])
        self._ss.write(f"{num_cons:<40} {'! NTIF - NUMBER OF TIDAL POTENTIAL CONSTITUENTS'}\n")
        tipotag_doc = '! TIPOTAG - NAME OF TIDAL POTENTIAL CONSTITUENT'
        props_doc = '! TPK, AMIGT, ETRF, FFT, FACET - CONSTITUENT PROPERTIES'
        first = True
        for tidal_data in self._xms_data['tidal_data']:  # Loop through the applied tidal constituent components.
            for con_name in tidal_data.cons.coords['con']:
                con_dset = tidal_data.cons.sel(con=con_name)
                if self._template:
                    con_props = (
                        f"{con_dset['amplitude'].item()} "
                        f"{con_dset['frequency'].item()} "
                        f"{con_dset['earth_tide_reduction_factor'].item()} "
                        f"%{con_name.item()}_NF% "
                        f"%{con_name.item()}_QARG% "
                    )
                    if first:  # CSTORM templates write the date to a comment on the first line.
                        tipotag_doc = '! ( %TIDE_DATE_STR% )'
                    else:
                        tipotag_doc = '! TIPOTAG - NAME OF TIDAL POTENTIAL CONSTITUENT'
                    first = False
                else:
                    con_props = (
                        f"{con_dset['amplitude'].item()} "
                        f"{con_dset['frequency'].item()} "
                        f"{con_dset['earth_tide_reduction_factor'].item()} "
                        f"{con_dset['nodal_factor'].item()} "
                        f"{con_dset['equilibrium_argument'].item()} "
                    )
                self._ss.write(f'{con_name.item():<40} {tipotag_doc}\n{con_props:<40} {props_doc}\n')

    def _write_tidal_forcing(self):
        """Write tidal forcing lines."""
        num_cons = sum([data.values.sizes['con'] for data in self._xms_data['tidal_data']])
        self._ss.write(
            f"{num_cons:<40} ! NBFR - NUMBER OF PERIODIC FORCING FREQUENCIES ON ELEVATION SPECIFIED BOUNDARIES\n"
        )
        if num_cons > 0:  # We have applied tidal constituents, export periodic tidal forcing.
            # First loop is constituent properties. Two lines per constituent. First line is constituent name.
            # Second line is: frequency nodal_factor equilibrium_argument
            boundtag_doc = '! BOUNTAG - NAME OF TIDAL FORCING CONSTITUENT'
            con_props_doc = '! AMIG, FF, FACE - CONSTITUENT PROPERTIES'
            first = True
            for tidal_data in self._xms_data['tidal_data']:  # Loop through the applied tidal constituent datasets.
                for con_name in tidal_data.values.coords['con']:
                    con_dset = tidal_data.cons.sel(con=con_name)
                    if self._template:
                        con_props = (f"{con_dset['frequency'].item()} %{con_name.item()}_NF% %{con_name.item()}_EQARG%")
                        if first:  # CSTORM writes a comment with the date on the first constituent.
                            boundtag_doc = '! ( %TIDE_DATE_STR% )'
                        else:
                            boundtag_doc = '! BOUNTAG - NAME OF TIDAL FORCING CONSTITUENT'
                        con_props = f'{con_props:<40}   {con_props_doc}'
                        first = False
                    else:
                        con_props = (
                            f"{con_dset['frequency'].item()} "
                            f"{con_dset['nodal_factor'].item()} "
                            f"{con_dset['equilibrium_argument'].item()}   {con_props_doc}"
                        )
                    self._ss.write(f'{con_name.item():<40} {boundtag_doc}\n{con_props:<40}\n')
            # Second loop is nodal amplitudes and phases. For each constituent, one line for constituent name and
            # one line for each ocean boundary node containing: amplitude phase
            alpha_doc = '! ALPHA - NAME OF TIDAL FORCING CONSTITUENT'
            node_props_doc = '! EMO, EFA - NODE CONSTITUENT PROPERTIES'
            for tidal_data in self._xms_data['tidal_data']:  # Loop through the applied tidal constituent datasets.
                for con_name in tidal_data.values.coords['con']:
                    con_dset = tidal_data.values.sel(con=con_name)
                    amps = con_dset['amplitude'].data.tolist()
                    phases = con_dset['phase'].data.tolist()
                    con_props = '\n'.join(
                        f'{amp:.6f} {phase:.3f}   {node_props_doc}' for amp, phase in zip(amps, phases)
                    )
                    self._ss.write(f'{con_name.item():<40} {alpha_doc}\n{con_props}\n')
        elif self._xms_data['main_bc'] and self._xms_data['main_bc'].source_data.info.attrs['periodic_tidal'] == 0:
            # Limited support for non-periodic water elevation forcing. Copy over file if it exists.
            # Stored paths will be relative to project if saved.
            proj_dir = self._xms_data['main_bc'].source_data.info.attrs['proj_dir']
            fort19 = ''
            try:
                fort19 = io_util.resolve_relative_path(
                    proj_dir, self._xms_data['main_bc'].source_data.info.attrs['fort.19']
                )
            except Exception:
                pass
            if os.path.isfile(fort19):
                io_util.copyfile(fort19, os.path.join(os.path.dirname(self._filename), 'fort.19'))
            elif not self._suppress_missing_nonperiodic_errors:
                ocean_nodes, _ = self._xms_data['main_bc'].get_ocean_node_ids()
                if len(ocean_nodes) > 0:
                    report_error(f'Unable to copy fort.19 file to export location: {fort19}', self._sim_export)

    def _write_flow_forcing(self):
        """Write flow forcing lines."""
        river_nodes = []
        if self._main_bc:
            river_nodes = self._xms_data['main_bc'].get_river_node_ids()
        if not river_nodes:
            return  # Do not write NFFR line if no river boundaries

        num_cons = 0
        if self._xms_data['flow_data']:
            num_cons = self._xms_data['flow_data'].cons.sizes['con']
        if num_cons > 0:  # We have applied periodic flow constituents, export periodic flow forcing.
            self._ss.write(  # NFFR for periodic flow forcing
                f"{num_cons:<40} ! NFFR - NUMBER OF FREQUENCIES ON NORMAL FLOW SPECIFIED BOUNDARIES\n"
            )

            # First loop is constituent properties. Two lines per constituent. First line is constituent name.
            # Second line is: frequency nodal_factor equilibrium_argument
            boundtag_doc = '! FBOUNTAG'
            con_props_doc = '! FAMIGT, FFF, FFACE - CONSTITUENT PROPERTIES'
            for con_name in self._xms_data['flow_data'].cons.coords['con']:
                con_dset = self._xms_data['flow_data'].cons.sel(con=con_name)
                con_props = (
                    f"{con_dset['frequency'].item():.15f} "
                    f"{con_dset['nodal_factor'].item():.3f} "
                    f"{con_dset['equilibrium_argument'].item():.3f}   {con_props_doc}"
                )
                self._ss.write(f'{con_name.item():<40} {boundtag_doc}\n{con_props}\n')

            # Second loop is nodal amplitudes and phases. For each constituent, one line for constituent name and
            # one line for each ocean boundary node containing: amplitude phase
            node_props_doc = '! QNAM, QNPH - NODE CONSTITUENT PROPERTIES'
            for con_name in self._xms_data['flow_data'].values.coords['con']:
                if self._template:
                    self._ss.write(f'%{con_name.item():<40}% {boundtag_doc}\n%BEGIN_RIVER%\n%END_RIVER%\n')
                else:
                    con_dset = self._xms_data['flow_data'].values.sel(con=con_name)
                    amps = con_dset['amplitude'].data.tolist()
                    phases = con_dset['phase'].data.tolist()
                    con_props = '\n'.join(
                        f'{amp:.6f} {phase:.3f}   {node_props_doc}' for amp, phase in zip(amps, phases)
                    )
                    self._ss.write(f'{con_name.item():<40} {boundtag_doc}\n{con_props}\n')
        elif self._xms_data['main_bc'].source_data.info.attrs['periodic_flow'] == 0:
            # NFFR for non-periodic flow forcing should be -1 or 0 and we copy a fort.20
            self._ss.write(
                f"{self._xms_data['main_bc'].source_data.info.attrs['hot_start_flow']:<40} "
                f"! NFFR - NUMBER OF FREQUENCIES ON NORMAL FLOW SPECIFIED BOUNDARIES\n"
            )
            # Limited support for non-periodic water elevation forcing. Copy over file if it exists.
            # Stored paths will be relative to project if save
            # proj_dir = self._xms_data['main_bc'].source_data.info.attrs['proj_dir']
            # fort20 = ''
            # try:
            #     fort20 = io_util.resolve_relative_path(
            #         proj_dir, self._xms_data['main_bc'].source_data.info.attrs['fort.20']
            #     )
            # except Exception:
            #     pass
            # if os.path.isfile(fort20):
            #     io_util.copyfile(fort20, os.path.join(os.path.dirname(self._filename), 'fort.20'))
            # elif not self._suppress_missing_nonperiodic_errors:
            #     fort20 = fort20 if XmEnv.xms_environ_running_tests() != 'TRUE' else os.path.basename(fort20)
            #     report_error(f'Unable to copy fort.20 file to export location: {fort20}', self._sim_export)

    def _calc_num_output_ts(self, ts_min):
        """Convert minute GUI input to ADCIRC timesteps."""
        if not ts_min:
            ts_min = self._data.timing.attrs['DTDP']
        ts_sec = ts_min * 60.0
        return round(ts_sec / self._data.timing.attrs['DTDP'])

    def _write_recording_stations(self):
        """Write the recording station output lines."""
        have_stations = self._xms_data['station_comp'] is not None and self._xms_data['station_pts']
        proj_dir = self._data.info.attrs['proj_dir']  # Stored file paths will be relative to project if saved.
        self._write_station_elevations(have_stations, proj_dir)
        self._write_station_velocity(have_stations, proj_dir)
        self._write_station_wind(have_stations, proj_dir)

    def _write_station_elevations(self, have_stations, proj_dir):
        """Write the recording stations for elevation.

        Args:
            have_stations (:obj:`bool`): True if we have any stations at all
            proj_dir (:obj:`str`): Path to the SMS project
        """
        # Elevation stations
        noute_doc = '! NOUTE, TOUTSE, TOUTFE, NSPOOLE - FORT 61 OPTIONS'
        nstae_doc = '! NSTAE - NUMBER OF ELEVATION RECORDING STATIONS, FOLLOWED BY LOCATIONS ON PROCEEDING LINES'
        elev_stations = self._data.output.sel(ParamName='NOUTE')
        noute = elev_stations['Output'].item()
        if self._template:
            self._ss.write(f'{"%NOUTE% %TOUTSE% %TOUTFE% %NSPOOLE%":<40} {noute_doc}\n')
            self._ss.write(f'{"%NSTAE%":<40} ! NSTAE - NUMBER OF ELEVATION RECORDING STATIONS\n')
        elif not have_stations or noute == 0:  # Output is disabled, write dummy values.
            self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noute_doc}\n")
            self._ss.write(f"{'0':<40} {nstae_doc}\n")
        else:  # Elevation station output is enabled
            if noute in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                self._write_netcdf_opts = True  # Have to write NetCDF options if any output formats are NetCDF
            spoole = self._calc_num_output_ts(elev_stations['Increment __new_line__ (min)'].item())
            append_to_hotstart = elev_stations['File'].item() == 0
            if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                noute *= -1
            toutse = elev_stations['Start __new_line__ (days)'].item()
            toutfe = elev_stations['End __new_line__ (days)'].item()
            card_value = f"{noute} {toutse:.6f} {toutfe:.6f} {spoole:d}"
            self._ss.write(f"{card_value:<40} {noute_doc}\n")
            hotstart_file = io_util.resolve_relative_path(proj_dir, elev_stations['Hot Start'].item())
            if append_to_hotstart and os.path.isfile(hotstart_file):
                # Copy the hot start file if specified. Not letting user pick any files in GUI that don't match
                # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

            # Write elevation station locations.
            if have_stations:
                temp_ss = StringIO()
                num_elevs = 0
                # Clear out unused component ids.
                cov_uuid = self._xms_data['station_comp'].cov_uuid
                used_comp_ids = self._xms_data['station_comp'].comp_to_xms[cov_uuid][TargetType.point].keys()
                used_comp_ids = np.array(list(used_comp_ids))
                mask = self._xms_data['station_comp'].data.stations.comp_id.isin(used_comp_ids)
                self._xms_data['station_comp'].data.stations = self._xms_data['station_comp'].data.stations.where(
                    mask, drop=True
                )
                elev_stations = self._xms_data['station_comp'].data.stations.where(
                    self._xms_data['station_comp'].data.stations.elevation == 1, drop=True
                )
                elev_comp_ids = elev_stations.coords['comp_id'].data.astype('i4').tolist()
                for elev_station in elev_comp_ids:
                    # Get the XMS att ids for this component id.
                    xms_ids = self._xms_data['station_comp'].get_xms_ids(TargetType.point, elev_station)
                    if xms_ids == -1:
                        msg = f'Could not export elevation station point with component id {elev_station}. ' \
                              f'The fort.15 is invalid and ADCIRC will not run.'
                        report_error(msg, self._sim_export)
                        continue
                    num_elevs += len(xms_ids)
                    for xms_id in xms_ids:
                        try:
                            do_point = self._xms_data['station_pts'][xms_id]
                            card_value = f"{do_point.x:.6f} {do_point.y:.6f}"
                            temp_ss.write(f"{card_value:<40} ! ELEVATION STATION LOCATION (ID = {xms_id})\n")
                        except KeyError:
                            msg = f'Could not export elevation station point with id {xms_id}. The fort.15 is ' \
                                  f'invalid and ADCIRC will not run.'
                            report_error(msg, self._sim_export)
                self._ss.write(f"{num_elevs:<40} {nstae_doc}\n")
                temp_ss.seek(0)
                self._ss.write(temp_ss.read())

    def _write_station_velocity(self, have_stations, proj_dir):
        """Write the recording stations for velocity.

        Args:
            have_stations (:obj:`bool`): True if we have any stations at all
            proj_dir (:obj:`str`): Path to the SMS project
        """
        # Velocity stations
        noutv_doc = '! NOUTV, TOUTSV, TOUTFV, NSPOOLV - FORT 62 OPTIONS'
        nstav_doc = '! NSTAV - NUMBER OF VELOCITY RECORDING STATIONS, FOLLOWED BY LOCATIONS ON PROCEEDING LINES'
        vel_stations = self._data.output.sel(ParamName='NOUTV')
        noutv = vel_stations['Output'].item()
        if self._template:  # Write a template file for CSTORM
            self._ss.write(f'{"%NOUTV% %TOUTSV% %TOUTFV% %NSPOOLV%":<40} {noutv_doc}\n')
            self._ss.write(f'{"%NSTAV%":<40} ! NSTAV - NUMBER OF VELOCITY RECORDING STATIONS\n')
        elif not have_stations or noutv == 0:  # Output is disabled, write dummy values.
            self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noutv_doc}\n")
            self._ss.write(f"{'0':<40} {nstav_doc}\n")
        else:  # Elevation station output is enabled
            if noutv in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                self._write_netcdf_opts = True  # Have to write NetCDF options if any output formats are NetCDF
            spoolv = self._calc_num_output_ts(vel_stations['Increment __new_line__ (min)'].item())
            append_to_hotstart = vel_stations['File'].item() == 0
            if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                noutv *= -1
            toutsv = vel_stations['Start __new_line__ (days)'].item()
            toutfv = vel_stations['End __new_line__ (days)'].item()
            card_value = f"{noutv} {toutsv:.6f} {toutfv:.6f} {spoolv:d}"
            self._ss.write(f"{card_value:<40} {noutv_doc}\n")
            hotstart_file = io_util.resolve_relative_path(proj_dir, vel_stations['Hot Start'].item())
            if append_to_hotstart and os.path.isfile(hotstart_file):
                # Copy the hot start file if specified. Not letting user pick any files in GUI that don't match
                # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

            # Write velocity station locations.
            if have_stations:
                vel_stations = self._xms_data['station_comp'].data.stations.where(
                    self._xms_data['station_comp'].data.stations.velocity == 1, drop=True
                )
                vel_comp_ids = vel_stations.coords['comp_id'].data.astype('i4').tolist()

                temp_ss = StringIO()
                num_vels = 0
                for vel_station in vel_comp_ids:
                    # Get the XMS att ids for this component id.
                    xms_ids = self._xms_data['station_comp'].get_xms_ids(TargetType.point, vel_station)
                    if xms_ids == -1:
                        msg = f'Could not export velocity station point with component id {vel_station}. ' \
                              f'The fort.15 is invalid and ADCIRC will not run.'
                        report_error(msg, self._sim_export)
                        continue
                    num_vels += len(xms_ids)
                    for xms_id in xms_ids:
                        try:
                            do_point = self._xms_data['station_pts'][xms_id]
                            card_value = f"{do_point.x:.6f} {do_point.y:.6f}"
                            temp_ss.write(f"{card_value:<40} ! VELOCITY STATION LOCATION (ID = {xms_id})\n")
                        except KeyError:
                            msg = f'Could not export velocity station point with id {xms_id}. The fort.15 is ' \
                                  f'invalid and ADCIRC will not run.'
                            report_error(msg, self._sim_export)
                self._ss.write(f"{num_vels:<40} {nstav_doc}\n")
                temp_ss.seek(0)
                self._ss.write(temp_ss.read())

    def _write_station_wind(self, have_stations, proj_dir):
        """Write the recording stations for wind.

        Args:
            have_stations (:obj:`bool`): True if we have any stations at all
            proj_dir (:obj:`str`): Path to the SMS project
        """
        # Meteorological stations
        if self._data.wind.attrs['NWS'] != 0:  # Do not write if no wind
            noutw_doc = '! NOUTM, TOUTSM, TOUTFM, NSPOOLM - FORT 71/72 OPTIONS'
            nstaw_doc = '! NSTAM - NUMBER OF MET RECORDING STATIONS, FOLLOWED BY LOCATIONS ON PROCEEDING LINES'
            wind_stations = self._data.output.sel(ParamName='NOUTW')
            noutw = wind_stations['Output'].item()
            if self._template:  # Write a template file for CSTORM
                self._ss.write(f'{"%NOUTM% %TOUTSM% %TOUTFM% %NSPOOLM%":<40} {noutw_doc}\n')
                self._ss.write(f'{"%NSTAM%":<40} ! NSTAV - NUMBER OF WIND RECORDING STATIONS\n')
            elif not have_stations or noutw == 0:  # Output is disabled, write dummy values.
                self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noutw_doc}\n")
                self._ss.write(f"{'0':<40} {nstaw_doc}\n")
            else:  # Elevation station output is enabled
                if noutw in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                    self._write_netcdf_opts = True  # Have to write NetCDF options if any output formats are NetCDF
                spoolw = self._calc_num_output_ts(wind_stations['Increment __new_line__ (min)'].item())
                append_to_hotstart = wind_stations['File'].item() == 0
                if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                    noutw *= -1
                toutsw = wind_stations['Start __new_line__ (days)'].item()
                toutfw = wind_stations['End __new_line__ (days)'].item()
                card_value = f"{noutw} {toutsw:.6f} {toutfw:.6f} {spoolw:d}"
                self._ss.write(f"{card_value:<40} {noutw_doc}\n")
                if append_to_hotstart:
                    # Copy the hot start file if specified. Not letting user pick any files in GUI that don't match
                    # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                    hotstart_file = io_util.resolve_relative_path(proj_dir, wind_stations['Hot Start'].item())
                    if os.path.isfile(hotstart_file):
                        io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))
                    hotstart_file = io_util.resolve_relative_path(
                        proj_dir, wind_stations['Hot Start (Wind Only)'].item()
                    )
                    if os.path.isfile(hotstart_file):
                        io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

                # Write wind station locations.
                if have_stations:
                    wind_stations = self._xms_data['station_comp'].data.stations.where(
                        self._xms_data['station_comp'].data.stations.wind == 1, drop=True
                    )
                    wind_comp_ids = wind_stations.coords['comp_id'].data.astype('i4').tolist()

                    temp_ss = StringIO()
                    num_winds = 0
                    for wind_station in wind_comp_ids:
                        # Get the XMS att ids for this component id.
                        xms_ids = self._xms_data['station_comp'].get_xms_ids(TargetType.point, wind_station)
                        if xms_ids == -1:
                            msg = f'Could not export wind station point with component id {wind_station}. ' \
                                  f'The fort.15 is invalid and ADCIRC will not run.'
                            report_error(msg, self._sim_export)
                            continue
                        num_winds += len(xms_ids)
                        for xms_id in xms_ids:
                            try:
                                do_point = self._xms_data['station_pts'][xms_id]
                                card_value = f"{do_point.x:.6f} {do_point.y:.6f}"
                                temp_ss.write(f"{card_value:<40} ! MET STATION LOCATION (ID = {xms_id})\n")
                            except KeyError:
                                msg = f'Could not export wind station point with id {xms_id}. The fort.15 is ' \
                                      f'invalid and ADCIRC will not run.'
                                report_error(msg, self._sim_export)
                    self._ss.write(f"{num_winds:<40} {nstaw_doc}\n")
                    temp_ss.seek(0)
                    self._ss.write(temp_ss.read())

    def _write_output_options(self):
        """Write the global output lines."""
        proj_dir = self._data.info.attrs['proj_dir']  # Stored file paths will be relative to project if saved.

        # Global elevation output
        elev_format = self._data.output.sel(ParamName='NOUTGE')
        noutge_doc = '! NOUTGE, TOUTSGE, TOUTFGE, NSPOOLGE - GLOBAL ELEVATION OUTPUT INFO (UNIT 63)'
        noutge = elev_format['Output'].item()
        if noutge == 0:  # Output is disabled, write dummy values.
            self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noutge_doc}\n")
        else:  # Global elevation output is enabled
            if self._template:
                self._ss.write(f'{"%NOUTGE% %TOUTSGE% %TOUTFGE% %NSPOOLGE%":<40} {noutge_doc}\n')
            else:
                if noutge in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                    self._write_netcdf_opts = True
                spoolge = self._calc_num_output_ts(elev_format['Increment __new_line__ (min)'].item())
                append_to_hotstart = elev_format['File'].item() == 0
                if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                    noutge *= -1
                toutsge = elev_format['Start __new_line__ (days)'].item()
                toutfge = elev_format['End __new_line__ (days)'].item()
                card_value = f"{noutge} {toutsge:.6f} {toutfge:.6f} {spoolge:d}"
                self._ss.write(f"{card_value:<40} {noutge_doc}\n")
                hotstart_file = io_util.resolve_relative_path(proj_dir, elev_format['Hot Start'].item())
                if append_to_hotstart and os.path.isfile(hotstart_file):
                    # Copy the hot start file if specified. Not letting user pick any files in GUI that don't match
                    # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                    io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

        # Global velocity output
        vel_format = self._data.output.sel(ParamName='NOUTGV')
        noutgv_doc = '! NOUTGV, TOUTSGV, TOUTFGV, NSPOOLGV - GLOBAL VELOCITY OUTPUT INFO (UNIT 64)'
        noutgv = vel_format['Output'].item()
        if noutgv == 0:  # Output is disabled, write dummy values.
            self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noutgv_doc}\n")
        else:  # Global velocity output is enabled
            if self._template:  # Create a template for CSTORM
                self._ss.write(f'{"%NOUTGV% %TOUTSGV% %TOUTFGV% %NSPOOLGV%":<40} {noutgv_doc}\n')
            else:
                if noutgv in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                    self._write_netcdf_opts = True
                spoolgv = self._calc_num_output_ts(vel_format['Increment __new_line__ (min)'].item())
                append_to_hotstart = vel_format['File'].item() == 0
                if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                    noutgv *= -1
                toutsgv = vel_format['Start __new_line__ (days)'].item()
                toutfgv = vel_format['End __new_line__ (days)'].item()
                card_value = f"{noutgv} {toutsgv:.6f} {toutfgv:.6f} {spoolgv:d}"
                self._ss.write(f"{card_value:<40} {noutgv_doc}\n")
                hotstart_file = io_util.resolve_relative_path(proj_dir, vel_format['Hot Start'].item())
                if append_to_hotstart and os.path.isfile(hotstart_file):
                    # Copy the hot start file if specified. Not letting user pick any files in GUI that don't match
                    # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                    io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

        # Global meteorological output
        if self._nws != 0:  # Do not write if no wind
            wind_format = self._data.output.sel(ParamName='NOUTGW')
            noutgw_doc = '! NOUTGW, TOUTSGW, TOUTFGW, NSPOOLGW - GLOBAL VELOCITY OUTPUT INFO (UNIT 73 & 74)'
            noutgw = wind_format['Output'].item()
            if noutgw == 0:  # Output is disabled, write dummy values.
                self._ss.write(f"{'0 0.000000 0.000000 0':<40} {noutgw_doc}\n")
            else:  # Global velocity output is enabled
                if self._template:  # Create a template for CSTORM
                    self._ss.write(f'{"%NOUTGM% %TOUTSGM% %TOUTFGM% %NSPOOLGM%":<40} {noutgw_doc}\n')
                else:
                    if noutgw in [3, 5]:  # 3=NetCDF, 5=NetCDF4
                        self._write_netcdf_opts = True
                    spoolgw = self._calc_num_output_ts(wind_format['Increment __new_line__ (min)'].item())
                    append_to_hotstart = wind_format['File'].item() == 0
                    if not append_to_hotstart:  # Adjust sign of value based on cold start vs. hot start
                        noutgw *= -1
                    toutsgw = wind_format['Start __new_line__ (days)'].item()
                    toutfgw = wind_format['End __new_line__ (days)'].item()
                    card_value = f"{noutgw} {toutsgw:.6f} {toutfgw:.6f} {spoolgw:d}"
                    self._ss.write(f"{card_value:<40} {noutgw_doc}\n")
                    if append_to_hotstart:
                        # Copy the hot start files if specified. Not letting user pick any files in GUI that don't match
                        # the ADCIRC hardcoded patterns, so don't need to rename here but could.
                        hotstart_file = io_util.resolve_relative_path(proj_dir, wind_format['Hot Start'].item())
                        if os.path.isfile(hotstart_file):
                            io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))
                        hotstart_file = io_util.resolve_relative_path(
                            proj_dir, wind_format['Hot Start (Wind Only)'].item()
                        )
                        if os.path.isfile(hotstart_file):
                            io_util.copyfile(hotstart_file, os.path.join(os.getcwd(), os.path.basename(hotstart_file)))

    def _write_harmonic_analysis(self):
        """Write the harmonic analysis lines."""
        # Get the user-defined constituent properties for harmonic analysis
        user_cons = self._data.harmonics.where(self._data.harmonics.Constituent == 'User defined', drop=True)
        con_names = user_cons['Constituent Name'].data.tolist()
        frequencies = user_cons['Frequency (HAREQ)'].data.tolist()
        nodal_factors = user_cons['Nodal Factor (HAFF)'].data.tolist()
        eq_args = user_cons['Equilibrium (HAFACE)'].data.tolist()

        # THAS is start time in days for harmonic analysis. Relative to ADCIRC reference date.
        thas = self._data.harmonics.attrs['THAS']

        # Get the standard constituent properties for harmonic analysis by querying harmonica
        std_cons = self._data.harmonics.where(self._data.harmonics.Constituent != 'User defined', drop=True)
        std_con_names = std_cons['Constituent Name'].data.tolist()
        if std_con_names:
            ref_date = datetime.datetime.strptime(self._data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT)
            ref_date += datetime.timedelta(hours=thas)
            # Specific tidal model doesn't matter. LeProvost is always available.
            std_con_props = Constituents('leprovost').get_nodal_factor(std_con_names, ref_date)
            con_names.extend(std_con_names)
            frequencies.extend(std_con_props['frequency'].to_list())
            nodal_factors.extend(std_con_props['nodal_factor'].to_list())
            eq_args.extend(std_con_props['equilibrium_argument'].to_list())

        self._ss.write(f"{len(con_names):<40} {'! NFREQ - NUMBER OF FREQENCIES IN HARMONIC ANALYSIS'}\n")
        for idx, con in enumerate(con_names):
            props = f'{frequencies[idx]} {nodal_factors[idx]} {eq_args[idx]}'
            self._ss.write(
                f"{con:<40} {'! NAMEFR - NAME OF HARMONIC ANALYSIS CONSTITUENT'}\n"
                f"{props:<40} {'! HAFREQ, HAFF, HAFACE - CONSTITUENT PROPERTIES'}\n"
            )

        card_value = (
            f"{thas:.6f} {self._data.harmonics.attrs['THAF']:.6f} {int(self._data.harmonics.attrs['NHAINC'])} "
            f"{self._data.harmonics.attrs['FMV']:.6f}"
        )
        self._ss.write(f"{card_value:<40} {'! THAS, THAF, NHAINC, FMV - HARMONIC ANALYSIS PARAMETERS'}\n")
        card_value = (
            f"{self._data.harmonics.attrs['NHASE']} {self._data.harmonics.attrs['NHASV']} "
            f"{self._data.harmonics.attrs['NHAGE']} {self._data.harmonics.attrs['NHAGV']}"
        )
        doc = '! NHASE, NHASV, NHAGE, NHAGV - CONTROL HARMONIC ANALYSIS AND OUTPUT TO UNITS 51,52,53,54'
        self._ss.write(f'{card_value:<40} {doc}\n')

    def _write_hot_start_output_options(self):
        """Write the hot start lines along with the other miscellaneous parameters at the end of the fort.15."""
        nhsinc = self._calc_num_output_ts(self._data.output.attrs['NHSINC'])
        nhstar = self._data.output.attrs['NHSTAR']
        if nhstar in [3, 5]:  # 5 = NetCDF4 which is unsupported until we get a Windows build of ADCIRC with NetCDF4
            self._write_netcdf_opts = True
        card_value = f"{nhstar} {nhsinc:d}"
        self._ss.write(f"{card_value:<40} {'! NHSTAR, NHSINC - HOT START FILE GENERATION PARAMETERS'}\n")
        card_value = (
            f"{self._data.formulation.attrs['ITITER']} {self._data.formulation.attrs['ISLDIA']} "
            f"{self._data.formulation.attrs['CONVCR']:.6g} {self._data.formulation.attrs['ITMAX']}"
        )
        self._ss.write(f"{card_value:<40} {'! ITITER, ISLDIA, CONVCR, ITMAX - ALGEBRAIC SOLUTION PARAMETERS'}\n")

    def _write_netcdf_options(self):
        """Write the optional NetCDF parameters if any of the output file formats are NetCDF."""
        self._ss.write(
            f"{self._data.output.attrs['NCPROJ']:<40} ! NCPROJ - NetCDF Title\n"
            f"{self._data.output.attrs['NCINST']:<40} ! NCINST - NetCDF Institution\n"
            f"{self._data.output.attrs['NCSOUR']:<40} ! NCSOUR - NetCDF Source\n"
            f"{self._data.output.attrs['NCHIST']:<40} ! NCHIST - NetCDF History\n"
            f"{self._data.output.attrs['NCREF']:<40} ! NCREF - NetCDF References\n"
            f"{self._data.output.attrs['NCCOM']:<40} ! NCCOM - NetCDF Comments\n"
            f"{self._data.output.attrs['NCHOST']:<40} ! NCHOST - NetCDF Host\n"
            f"{self._data.output.attrs['NCCONV']:<40} ! NCCONV - NetCDF Conventions\n"
            f"{self._data.output.attrs['NCCONT']:<40} ! NCCONT - NetCDF Contact information\n"
        )
        # Date must be in yyyy-MM-dd hh:mm:_ss tz format (e.g. 2010-05-01 00:00:00 UTC)
        user_date = self._data.timing.attrs['ref_date']
        if user_date:
            user_date = datetime.datetime.strptime(self._data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT)
            year = user_date.year
            month = user_date.month
            day = user_date.day
            hour = user_date.hour
            minute = user_date.minute
            second = user_date.second
        else:  # Use current time if not specified
            now = datetime.datetime.now()
            year = now.year
            month = now.month
            day = now.day
            hour = now.hour
            minute = now.minute
            second = now.second
        date_str = f'{year}-{month:0>2d}-{day:0>2d} {hour:0>2d}:{minute:0>2d}:{second:0>2d} UTC'
        self._ss.write(f'{date_str:<40} ! NCDATE - NetCDF Date\n')

    def _write_namelists(self):
        """Write the optional Fortran namelists at the end of the fort.15.

        Currently only the time-varying bathymetry namelist is supported.
        """
        doc = "! NDDT, BTIMINC, BCHGTIMINC -- BATHYMETRY TIME RECORDS (IN SECONDS) AND TRANSITION TIME\n"
        nddt = self._data.advanced.attrs['NDDT'] * self._data.advanced.attrs['NDDT_hot_start']
        btiminc = self._data.advanced.attrs['BTIMINC']
        bchgtiminc = self._data.advanced.attrs['BCHGTIMINC']
        self._ss.write(f"{'&timeBathyControl':<40} {doc}")
        self._ss.write(f"    NDDT = {nddt}\n")
        self._ss.write(f"    BTIMINC = {btiminc}\n")
        self._ss.write(f"    BCHGTIMINC = {bchgtiminc}\n")
        self._ss.write("/\n")

    def write(self):
        """Top-level function that triggers export of an ADCIRC simulation."""
        if 'sim_data' not in self._xms_data or not self._xms_data['sim_data']:
            report_error('Unable to find ADCIRC simulation data. Files were not exported', self._sim_export)
            return

        parallel_processes = []
        try:
            # Launch the other export scripts in separate processes
            parallel_processes = [self._launch_fort13(), self._launch_fort14(), self._launch_fort20(),
                                  self._launch_fort22()]

            self._nws = self._data.get_nws_type()

            # Copy hot start files if needed
            self._copy_hotstart_files()

            # Start building up lines of the fort.15
            XmLog().instance.info('Exporting run options...')
            self._write_run_options()
            XmLog().instance.info('Exporting friction options...')
            self._write_noli_options()
            XmLog().instance.info('Exporting nodal attribute options...')
            self._write_nodal_attributes()
            XmLog().instance.info('Exporting NWS options...')
            self._write_nws()
            XmLog().instance.info('Exporting hyperbolic ramp and timing options...')
            self._write_ramp_and_timing_options()
            XmLog().instance.info('Computing WTIMINC...')
            self._write_wtiminc()
            self._write_dramp()
            XmLog().instance.info('Exporting computational equation options...')
            self._write_equation_options()
            XmLog().instance.info('Exporting potential and forcing constituents...')
            self._write_constituents()
            XmLog().instance.info('Exporting recording stations...')
            self._write_recording_stations()
            XmLog().instance.info('Exporting global output options...')
            self._write_output_options()
            XmLog().instance.info('Exporting harmonic analysis options...')
            self._write_harmonic_analysis()
            self._write_hot_start_output_options()
            # Several extra lines need to be written before the Fortran namelists if any of the output format is NetCDF
            if self._write_netcdf_opts:
                XmLog().instance.info('Exporting NetCDF output file metadata...')
                self._write_netcdf_options()
            self._write_namelists()

            # Flush lines to disk.
            XmLog().instance.info('Flushing exported data to disk...')
            with open(self._filename, 'w') as f:
                self._ss.seek(0)
                shutil.copyfileobj(self._ss, f, 100000)
            XmLog().instance.info('Successfully exported fort.15 control file.')
        except Exception as ex:
            msg = 'Error(s) encountered while exporting ADCIRC simulation.'
            XmLog().instance.exception(msg)
            if self._sim_export:
                XmEnv.report_error(msg)
                XmEnv.report_error(ex)
        finally:
            XmLog().instance.info('Waiting for parallel export scripts to complete...')
            process_names = ['fort.13', 'fort.14', 'fort.20', 'fort.22']
            sys.stdout.write('')  # Give us some separation from the fort.15 logging output
            for idx, parallel_process in enumerate(parallel_processes):
                if parallel_process is not None:
                    outs, errs = parallel_process.communicate()
                    XmLog().info(f'{process_names[idx]} info and warning log:\n{outs.decode()}')
                    if errs:
                        XmLog().error(f'{process_names[idx]} error log:\n{errs.decode()}')
            XmLog().instance.info('Export finished.')
