"""Writer for ADCIRC fort.20 flow file."""

# 1. Standard Python modules
from io import StringIO
import shutil

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules

# 4. Local modules
from xms.adcirc.data.mapped_bc_data import MappedBcData
from xms.adcirc.feedback.xmlog import XmLog


def export_flow_to_fort20(filename, mapped_bc_datas):
    """Export grid geometry to the fort.14.

    Args:
        filename (:obj:`str`): Output file location
        mapped_bc_datas (list): Absolute paths to the simulation's mapped boundary conditions datas.
    """
    mapped_bc_datas = [MappedBcData(main_file.strip("'\"")) for main_file in mapped_bc_datas]
    river_flows = None
    for mapped_bc in mapped_bc_datas:
        if len(mapped_bc.river_flows['Flow']) == 0:
            continue
        if river_flows is None:
            river_flows = mapped_bc.river_flows.copy()
        else:
            river_flows = xr.concat([mapped_bc.river_flows, river_flows], dim='comp_id')

    if river_flows is None:
        return

    ss = StringIO()
    XmLog().instance.info('Exporting hydrographs.')

    # get the timestep
    ts_list = river_flows['TS'].values.tolist()
    dt = ts_list[1] - ts_list[0]
    num_ts = river_flows.sizes['TS']

    # get node ids in ascending order
    node_ids = sorted(river_flows['Node Id'].values.tolist())
    node_id_indices = [int(np.where(river_flows['Node Id'].values == nid)[0][0]) for nid in node_ids]

    # get the flows in ascending node order for each TS
    ss.write(f'{dt:.6f}             ! time interval\n')

    for ts_idx in range(num_ts):
        flows = [river_flows['Flow'].isel(**{'Node Id': idx, 'TS': ts_idx}).item() for idx in node_id_indices]
        ss.write(f'{flows[0]:.6f}             ! TS {ts_idx + 1}\n')
        for flow in flows[1:]:
            ss.write(f'{flow}\n')

    # Write in-memory stream to file
    XmLog().instance.info('Saving in-memory stream to file.')
    with open(filename, 'w') as f:
        ss.seek(0)
        shutil.copyfileobj(ss, f, 1000000)
    XmLog().instance.info('Successfully exported fort.20 file.')
