"""Qt dialog for assigning attributes of a Recording Stations coverage point."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QCheckBox, QDialogButtonBox, QGroupBox, QLabel, QSizePolicy, QSpacerItem, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class AssignStationDlg(XmsDlg):
    """A dialog showing the Assign Station dialog."""
    def __init__(self, win_cont, station_data, msg):
        """Initializes the Assign Station dialog.

        Args:
            win_cont (:obj:`QWidget`): Parent dialog
            station_data (:obj:`xarray.Dataset`): Dataset containing the station parameters
            msg (:obj:`str`): Warning/multi-select message, if any
        """
        super().__init__(win_cont, 'xmsadcirc.gui.assign_station_dlg')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC_Coverages#Recording_Stations'
        self._data = station_data
        self._msg = msg
        self._widgets = {}

        # Setup the dialog
        self.setWindowTitle('ADCIRC Recording Stations')
        self._setup_ui()

    @property
    def data(self):
        """Read-only accessor for the station attribute dataset."""
        return self._data

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Add the message label if not empty
        self._widgets['group_box'] = QGroupBox('Recording station options')
        self._widgets['tog_elevation'] = QCheckBox('Water Elevation')
        self._widgets['tog_velocity'] = QCheckBox('Velocity')
        self._widgets['tog_wind'] = QCheckBox('Meteorological')
        self._widgets['vert_layout'] = QVBoxLayout()
        if self._msg:
            label = QLabel(self._msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self._widgets['vert_layout'].addWidget(label)
        self._widgets['vert_layout'].addWidget(self._widgets['tog_elevation'])
        self._widgets['vert_layout'].addWidget(self._widgets['tog_velocity'])
        self._widgets['vert_layout'].addWidget(self._widgets['tog_wind'])
        self._widgets['group_box'].setLayout(self._widgets['vert_layout'])
        self._widgets['vert_layout_main'] = QVBoxLayout()
        self._widgets['vert_layout_main'].addWidget(self._widgets['group_box'])

        # Add a vertical spacer
        self._widgets['vert_spacer'] = QSpacerItem(40, 20, QSizePolicy.Minimum, QSizePolicy.Expanding)
        self._widgets['vert_layout_main'].addSpacerItem(self._widgets['vert_spacer'])

        # Add the buttonbox
        self._widgets['btn_box'] = QDialogButtonBox()
        self._widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        )
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['vert_layout_main'].addWidget(self._widgets['btn_box'])
        self.setLayout(self._widgets['vert_layout_main'])

        # Populate widgets from data
        self._load_data()

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def _load_data(self):
        """Populate widgets from persistent data."""
        self._widgets['tog_elevation'].setCheckState(
            Qt.Checked if int(self._data['elevation'].data[0].item()) == 1 else Qt.Unchecked
        )
        self._widgets['tog_velocity'].setCheckState(
            Qt.Checked if int(self._data['velocity'].data[0].item()) == 1 else Qt.Unchecked
        )
        self._widgets['tog_wind'].setCheckState(
            Qt.Checked if int(self._data['wind'].data[0].item()) == 1 else Qt.Unchecked
        )

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        # Will reassign a new component id outside the dialog
        self._data['elevation'] = 1 if self._widgets['tog_elevation'].checkState() == Qt.Checked else 0
        self._data['velocity'] = 1 if self._widgets['tog_velocity'].checkState() == Qt.Checked else 0
        self._data['wind'] = 1 if self._widgets['tog_wind'].checkState() == Qt.Checked else 0
