"""Qt table view for extracted tidal constituent values."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import QAbstractButton, QAbstractItemView, QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget

TBL_COL_AMPLITUDE = 0
TBL_COL_PHASE = 1


class ConstituentValuesWidget(AdcircTableWidget):
    """The harmonic analysis table widget."""
    def __init__(self, data_frame, parent=None):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
        """
        data_frame = self._rename_columns_for_gui(data_frame)
        super().__init__(parent, data_frame, TBL_COL_AMPLITUDE, {
            'Amplitude': 0.0,
            'Phase': 0.0,
        })
        self.dbl_delegate = None
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_delegate = EditFieldValidator(dbl_validator, self)
        delegates = {
            TBL_COL_AMPLITUDE: self.dbl_delegate,
            TBL_COL_PHASE: self.dbl_delegate,
        }
        super()._setup_ui(delegates, False, True)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        # Set text for the vertical header column
        corner_btn = self.table_view.findChild(QAbstractButton)
        if corner_btn is not None:
            # Create a text label for the vertical header. This could just be a non-editable column in the table
            # I suppose, but this is just how the data gets represented using default base class behavior.
            layout = QVBoxLayout(corner_btn)
            layout.setContentsMargins(0, 0, 0, 0)
            header_lbl = QLabel('Node ID')
            header_lbl.setContentsMargins(0, 0, 0, 0)
            header_lbl.setAlignment(Qt.AlignCenter)
            layout.addWidget(header_lbl, alignment=Qt.AlignCenter)
            # Expand the vertical column so header text is visible.
            font_metrics = QFontMetrics(header_lbl.font())
            self.table_view.verticalHeader().setMinimumWidth(font_metrics.horizontalAdvance(header_lbl.text()) + 5)
            self.table_view.verticalHeader().setDefaultAlignment(Qt.AlignCenter)

    @staticmethod
    def _rename_columns_for_gui(data_frame):
        """Rename the DataFrame columns to user friendly text.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The constituent properties data

        Returns:
            (:obj:`pandas.DataFrame`): The input DataFrame with renamed columns

        """
        data_frame = data_frame.rename(columns={
            'amplitude': 'Amplitude',
            'phase': 'Phase',
        })
        return data_frame
