"""A dialog for displaying feature map lines."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import branca
import folium
from PySide2.QtCore import QUrl
from PySide2.QtWebEngineWidgets import QWebEngineView
from PySide2.QtWidgets import QVBoxLayout
from shapely.geometry import LineString, Point

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.gdal.utilities import gdal_utils as gu
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adcirc.mapping import mapping_util as map_util
from xms.adcirc.tools import levee_check_tool_consts as const


def get_line_extents(line):
    """Find the average, minimum, and maximum coordinates of a line.

    Args:
        line (:obj:`list`): x,y coordinates of the line locations

    Returns:
        (:obj:`tuple`): Average latitude, average longitude, minimum latitude, minimum longitude, maximum latitude,
        maximum longitude
    """
    min_lat = float('inf')
    min_lon = float('inf')
    max_lat = float('-inf')
    max_lon = float('-inf')
    for point in line:
        min_lat = min(point[0], min_lat)
        min_lon = min(point[1], min_lon)
        max_lat = max(point[0], max_lat)
        max_lon = max(point[1], max_lon)
    return min_lat, min_lon, max_lat, max_lon


class LineMapViewerDialog(XmsDlg):
    """A dialog for displaying feature map lines."""
    CHARACTER_WIDTH = 10  # Seems to work well on my machine, hopefully it is consistent.
    NUM_TICK_INTERVALS = 6  # Equidistant intervals along the levee where we place distance tick markers
    CHECK_STATUS_TO_LINE_COLOR = {  # Levee check status to line color
        const.CHECK_STATUS_ADJUSTED: 'blue',
        const.CHECK_STATUS_ADJUSTED_ISSUE: '#FF00FF',  # magenta
        const.CHECK_STATUS_UNADJUSTED: 'green',
        const.CHECK_STATUS_UNADJUSTED_ISSUE: 'red',
    }
    LEGEND_HTML = '''
{% macro html(this, kwargs) %}
<div style="
    position: fixed;
    bottom: 50px;
    left: 50px;
    width: 250px;
    height: 120px;
    z-index:9999;
    font-size:14px;
    ">
    <p><a style="color:#0000FF;font-size:100%;margin-left:20px;">&block;</a>&emsp;Adjusted without issue</p>
    <p><a style="color:#FF00FF;font-size:100%;margin-left:20px;">&block;</a>&emsp;Adjusted with issue</p>
    <p><a style="color:#59A84A;font-size:100%;margin-left:20px;">&block;</a>&emsp;Unadjusted without issue</p>
    <p><a style="color:#FF0000;font-size:100%;margin-left:20px;">&block;</a>&emsp;Unadjusted with issue</p>
</div>
<div style="
    position: fixed;
    bottom: 50px;
    left: 50px;
    width: 200px;
    height: 120px;
    z-index:9998;
    font-size:14px;
    background-color: #ffffff;

    opacity: 0.7;
    ">
</div>
{% endmacro %}
'''

    def __init__(self, parent, do_cov, results_df):
        """Initializes the sediment material list and properties dialog.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            do_cov (:obj:`Coverage`): The data_objects BC coverage geometry
            results_df (:obj:`pandas.DataFrame`): The tool results DataFrame
        """
        super().__init__(parent, 'xmsadcirc.gui.line_map_viewer_dialog')
        self._widgets = {}
        self._lines = []
        self._line_extents = []  # [[(min_lat, min_lon), (max_lat, max_lon)]] - parallel with self._lines
        self._line_idx_to_feature_id = {}  # key = index in self._lines, value = feature arc id
        self._feature_id_to_line_idx = {}  # Reverse lookup of self._line_idx_to_feature_id
        self._feature_id_to_check_status = {}  # {feature_arc_id: check_status}
        self._levee_lookup = {}  # {feature_arc_id: levee_index}
        self._num_levees = 0  # Number of levees being plotted in the results table
        self._markers = []  # [[[label_point_x, label_point_y], popup_text, max_width]]  - Result levee markers
        self._tick_positions = []  # [[(label_point_x0, label_point_y0)...(label_point_x1, label_point_y1)]]
        # Extents of all geometric data - [(min_lat, min_lon), (max_lat, max_lon)]
        self._global_extents = [[float('inf'), float('inf')], [float('-inf'), float('-inf')]]
        # Projection info that we have to pass in because it does not reliably serialize to the Coverage H5 dump file.
        self._wkt = results_df.attrs['wkt']
        self._coord_sys = results_df.attrs['coord_sys']
        self._is_meters = 'METER' in results_df.attrs['horiz_units']
        self._is_local = False  # Don't display ESRI map layers if in local space
        self._url_file = map_util.check_levee_results_html_file()
        self.hidden_rows = set()  # Indices of rows currently being filtered in results table

        # Build a mapping of feature arc id to index of levee pair in results DataFrame.
        self._find_levee_pairs(results_df)
        # Unpack the coverage data_objects Arc locations and convert to lat/lon if needed
        self._transform_points(do_cov)
        # Add widgets to dialog
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        self.setWindowTitle('Map Viewer')
        self.setMinimumSize(300, 300)
        self._widgets['vert_layout'] = QVBoxLayout()
        self._widgets['plot_view'] = QWebEngineView(self)
        self._widgets['vert_layout'].addWidget(self._widgets['plot_view'])
        self.setLayout(self._widgets['vert_layout'])

    def _find_levee_pairs(self, results_df):
        """Store associations between feature arc ids and their levee in the results DataFrame.

        Args:
            results_df (:obj:`pandas.DataFrame`): The tool results DataFrame
        """
        arc1_ids = results_df['Arc 1 ID'].values.tolist()
        arc2_ids = results_df['Arc 2 ID'].values.tolist()
        check_statuses = results_df['Status'].values.tolist()
        for levee_idx, (arc1_id, arc2_id, check_status) in enumerate(zip(arc1_ids, arc2_ids, check_statuses)):
            self._num_levees += 1
            self._feature_id_to_check_status[arc1_id] = check_status
            self._feature_id_to_check_status[arc2_id] = check_status
            self._levee_lookup[arc1_id] = levee_idx
            self._levee_lookup[arc2_id] = levee_idx

    def _transform_points(self, do_cov):
        """Extract line locations from the data_objects coverage and transform if necessary.

        Args:
            do_cov (:obj:`Coverage`): The data_objects coverage geometry
        """
        lines = []
        for idx, do_arc in enumerate(do_cov.arcs):
            # Fill some lookup maps and extract the locations of the arc into something less annoying than dat_objects.
            self._line_idx_to_feature_id[idx] = do_arc.id
            self._feature_id_to_line_idx[do_arc.id] = idx
            lines.append([(pt.y, pt.x) for pt in do_arc.get_points(FilterLocation.PT_LOC_ALL)])
        # Convert coordinates to lat/lon for displaying folium
        if self._coord_sys == 'GEOGRAPHIC':  # Already in geographic
            self._lines = lines
        elif self._coord_sys in ['NONE', '']:  # Local space
            converter = map_util.meters_to_decimal_degrees if self._is_meters else map_util.feet_to_decimal_degrees
            self._lines = [[(converter(pt[0], 0.0), converter(pt[1], 0.0)) for pt in line] for line in lines]
            self._is_local = True
        else:  # We are in non-geographic projected space
            self._reproject_points(lines)
        self._find_all_line_extents()  # Precompute the extents of each line

    def _reproject_points(self, lines):
        """Reproject the line locations from a projected system to geographic.

        Args:
            lines (:obj:`list`): List of the line locations [[(x,y),...],...]
        """
        # Set up a transformation from input non-geographic to WGS 84
        t_wkt = gu.wkt_from_epsg(4326)  # This is geographic GCS_WGS_1984
        transform = gu.get_coordinate_transformation(self._wkt, t_wkt)
        for line in lines:
            transform_line = []
            for point in line:
                coords = transform.TransformPoint(point[1], point[0])
                transform_line.append((coords[0], coords[1]))
            self._lines.append(transform_line)

    def _find_all_line_extents(self):
        """Compute and store the extents of each line (as well as the global extents).

        Notes:
            This method should only be called once during initialization.
        """
        self._markers = [None] * self._num_levees
        self._tick_positions = [None] * self._num_levees
        # Compute the extents for each line
        for line_idx, line in enumerate(self._lines):
            min_lat, min_lon, max_lat, max_lon = get_line_extents(line)
            self._add_line_to_extents(min_lat, min_lon, max_lat, max_lon)
            # If the arc is part of a levee pair, add compute a location for a marker.
            self._store_line_marker_info(line_idx)

    def _add_line_to_extents(self, min_lat, min_lon, max_lat, max_lon):
        """Store the extents of a single line and update the global extents.

        Args:
            min_lat (:obj:`float`): Minimum latitude of the line
            min_lon (:obj:`float`): Minimum longitude of the line
            max_lat (:obj:`float`): Maximum latitude of the line
            max_lon (:obj:`float`): Maximum longitude of the line
        """
        # Store the extents of this arc
        self._line_extents.append([(min_lat, min_lon), (max_lat, max_lon)])
        # Update the global extents
        self._global_extents[0][0] = min(min_lat, self._global_extents[0][0])
        self._global_extents[0][1] = min(min_lon, self._global_extents[0][1])
        self._global_extents[1][0] = max(max_lat, self._global_extents[1][0])
        self._global_extents[1][1] = max(max_lon, self._global_extents[1][1])

    def _find_marker_positions(self, line_idx):
        """Find the midpoint and locations along a line to place the 0.0, 0.2, 0.4, 0.6, 0.8, and 1.0 length ticks.

        Args:
            line_idx (:obj:`int`): Index in self._lines of the line to compute label locations for.

        Returns:
            (:obj:`tuple(Point,list[Point])`: The midpoint, points at 20% length intervals
        """
        linestring = LineString([Point(coord[1], coord[0]) for coord in self._lines[line_idx]])
        midpoint_xy = linestring.interpolate(linestring.length / 2)  # Find midpoint for info label
        distance = 0.0
        length_ticks = []
        for _ in range(self.NUM_TICK_INTERVALS):  # Find locations at which to place length ticks
            length_ticks.append(linestring.interpolate(linestring.length * distance))
            distance += 0.2
        return midpoint_xy, length_ticks

    def _store_line_marker_info(self, line_idx):
        """Initialize marker info for a levee if line is first arc in pair, update marker info if second arc in pair.

        Args:
            line_idx (:obj:`int`): 0-based index of the line in self._lines
        """
        feature_id = self._line_idx_to_feature_id[line_idx]
        levee_idx = self._levee_lookup.get(feature_id)
        if levee_idx is not None:  # This is arc is part of a levee pair
            if self._markers[levee_idx] is None:  # First encountered arc of the levee pair, find a good label point.
                status_label = const.CHECK_STATUS_TEXT[self._feature_id_to_check_status[feature_id]]
                status_label = f'<i>{status_label}</i> <br>'
                levee_label = f'<b>Levee row {levee_idx + 1}</b> <br>'
                arc1_label = f'Arc 1 ID = {feature_id} <br>'
                popup_text = f'{levee_label}{status_label}{arc1_label}'
                midpoint_xy, tick_positions = self._find_marker_positions(line_idx)
                self._markers[levee_idx] = [
                    [midpoint_xy.y, midpoint_xy.x], popup_text,
                    max(len(levee_label), len(arc1_label))
                ]
                self._tick_positions[levee_idx] = [[point.y, point.x] for point in tick_positions]
            else:  # Second encountered arc of the levee pair.
                arc2_label = f'Arc 2 ID = {feature_id} <br>'
                marker = self._markers[levee_idx]
                marker[1] += arc2_label
                marker[2] = max(marker[2], len(arc2_label))
                # Move marker positions to the middle of the levee
                midpoint_xy, tick_positions = self._find_marker_positions(line_idx)
                marker[0][0] = (marker[0][0] + midpoint_xy.y) / 2
                marker[0][1] = (marker[0][1] + midpoint_xy.x) / 2
                ticks = self._tick_positions[levee_idx]
                for tick_idx, tick_position in enumerate(tick_positions):
                    ticks[tick_idx][0] = (ticks[tick_idx][0] + tick_position.y) / 2
                    ticks[tick_idx][1] = (ticks[tick_idx][1] + tick_position.x) / 2

    def _find_levee_extents(self, selected_arcs):
        """Find the extents of a levee pair when its row is selected in the results table.

        Args:
            selected_arcs (:obj:`list[int]`): Feature arc ids of the levee arc pair to zoom image to

        Returns:
            (:obj:`list`): Extents of the levee arc pair - [(min_lat, min_lon), (max_lat, max_lon)]
        """
        if not selected_arcs:
            return self._global_extents  # If no arc selection, frame to extents of all geometric data

        extents = [
            [float('inf'), float('inf')],  # (min_lat, min_lon)
            [float('-inf'), float('-inf')],  # (max_lat, max_lon)
        ]
        for feature_id in selected_arcs:
            line_idx = self._feature_id_to_line_idx[feature_id]
            line_extents = self._line_extents[line_idx]
            extents[0][0] = min(extents[0][0], line_extents[0][0])
            extents[0][1] = min(extents[0][1], line_extents[0][1])
            extents[1][0] = max(extents[1][0], line_extents[1][0])
            extents[1][1] = max(extents[1][1], line_extents[1][1])
        return extents

    def _add_esri_map_layers(self, folium_map):
        """Add the online ESRI map raster layers if data is not in local space.

        Args:
            folium_map (:obj:`folium.Map`): The folium plot to add the raster layers to
        """
        if self._is_local:
            return  # Only add the layers if we have a non-local projection
        url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}'
        tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI Street Map', attr='ESRI')
        tl.add_to(folium_map)
        url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
        tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI World Imagery', attr='ESRI')
        tl.add_to(folium_map)

    def _add_line_markers(self, folium_map, selected_levee_idx, selected_color):
        """Add line markers to the levees that are being plotted in the results table.

        Args:
            folium_map (:obj:`folium.Map`): The folium plot to add line markers to
            selected_levee_idx (:obj:`int`): Index of the selected levee row in the results table, -1 if none selected
            selected_color (:obj:`str`): Color of the selected levee's lines, empty string if none selected
        """
        # Add label points to the levees that are being plotted in the results table.
        for idx, (label_point, popup_text, text_width) in enumerate(self._markers):
            if idx in self.hidden_rows:
                continue  # This levee is currently hidden
            width = text_width * self.CHARACTER_WIDTH
            iframe = folium.IFrame(popup_text)
            popup = folium.Popup(iframe, max_width=width, min_width=width)
            folium.Marker(label_point, popup=popup).add_to(folium_map)
        # Add circle markers at 20% length intervals along the levee if it is currently selected in the results table.
        if selected_levee_idx > -1:
            tick_positions = self._tick_positions[selected_levee_idx]
            distance = 0.0
            for tick_position in tick_positions:
                text = str(f'{distance:.1f}')  # Dumb that we have to do this but get 0.600000...1 if we don't
                tick = folium.CircleMarker(
                    location=tick_position,
                    popup=text,
                    tooltip=text,
                    radius=3,
                    color=selected_color,
                    fill=True,
                    fillOpacity=0.5
                )
                tick.add_to(folium_map)
                distance += 0.2

    def draw_feature_lines(self, selected_arcs):
        """Draw the levee BC coverage in the map.

        Args:
            selected_arcs (:obj:`list[int]`): Feature arc ids of the levee arc pair to zoom image to
        """
        folium_map = folium.Map(tiles='')
        self._add_esri_map_layers(folium_map)
        selected_levee_idx = -1
        selected_levee_idx_color = ''
        for idx, line in enumerate(self._lines):
            arc_id = self._line_idx_to_feature_id[idx]
            levee_idx = self._levee_lookup.get(arc_id, -1)
            if levee_idx < 0 or levee_idx in self.hidden_rows:
                # If arc is part of a levee that is currently filtered out of results table, or it is a non-levee arc,
                # don't draw it.
                continue
            color = self.CHECK_STATUS_TO_LINE_COLOR.get(self._feature_id_to_check_status.get(arc_id, -1), 'grey')
            if arc_id in selected_arcs:  # Draw levee arcs currently selected in the results table thicker and solid
                weight = 2
                dash_array = None
                selected_levee_idx = levee_idx
                selected_levee_idx_color = color
            else:  # Draw visible, non-selected levee arcs and all non-levee arcs thinner and dashed
                weight = 1
                dash_array = '10'
            tooltip = folium.Tooltip(f'Arc ID {arc_id}', sticky=True)
            # Draw adjusted levees purple, unadjusted green, skipped due to warning yellow, skipped due to error red,
            # and non-levee arcs grey.
            polyline = folium.vector_layers.PolyLine(
                locations=line, weight=weight, color=color, tooltip=tooltip, dash_array=dash_array
            )
            polyline.add_to(folium_map)
        # Add label points to the levees that are being plotted in the results table.
        self._add_line_markers(folium_map, selected_levee_idx, selected_levee_idx_color)
        # Zoom to the selected levee pair if specified, otherwise frame to the global extents of the data.
        folium_map.fit_bounds(self._find_levee_extents(selected_arcs))
        folium.map.LayerControl().add_to(folium_map)
        legend = branca.element.MacroElement()
        legend._template = branca.element.Template(self.LEGEND_HTML)
        folium_map.get_root().add_child(legend)
        # Use io, rather than a file on disk, to store the map as html in memory
        folium_map.save(self._url_file)
        self._widgets['plot_view'].setUrl(QUrl.fromLocalFile(self._url_file))
        self._widgets['plot_view'].show()
