"""Convert mapped boundary conditions to a Boundary Conditions coverage."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
from distutils.dir_util import copy_tree
import os
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.constraint.ugrid_builder import UGridBuilder
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Arc, Component, Coverage, Dataset, Point, Projection, UGrid as DoUGrid
from xms.datasets.dataset_writer import DatasetWriter
from xms.grid.ugrid import UGrid
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.adcirc.components.bc_component_display import (
    BC_JSON, BC_POINT_ID_FILE, BC_POINT_JSON, BcComponentDisplay, DEFAULT_POINT_BC_JSON
)
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.data.adcirc_data import UNINITIALIZED_COMP_ID
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.mapping_util import coordinate_hash, get_parametric_lengths

ADCIRC_NULL_VALUE = -99999.0


class BcUnMapper:
    """Class for converting mapped BC components to map module coverages."""
    def __init__(self, mapped_bc_comp, temp_dir, cov_name, co_grid, flow_data):
        """Construct the unmapper.

        Args:
            mapped_bc_comp (:obj:`MappedBcComponent`): The mapped BC component to convert
            temp_dir (:obj:`str`): Path to the component folder in the XMS temporary directory.
            cov_name (:obj:`str`): Name to give to the new Boundary Conditions coverage.
            co_grid (:obj:`CoGrid`): The domain mesh
            flow_data (:obj:`MappedFlowData`): The mapped periodic flow component data associated with the mapped BC
                component being converted to a coverage (None if not applicable).
        """
        self._xmgrid = co_grid.ugrid
        self._mapped_bc_comp = mapped_bc_comp
        self._flow_data = flow_data
        self._bc_cov_uuid = str(uuid.uuid4())  # Generate a new UUID for the BC map module coverage
        self._bc_comp_uuid = str(uuid.uuid4())  # Generate a new UUID for the hidden coverage component
        self._bc_comp_dir = os.path.join(temp_dir, self._bc_comp_uuid)
        self._do_cov = None
        self._next_point_id = 1
        self._node_idx_to_do_point = {}  # {XmUGrid node index: data_objects.parameters.Point}
        self._unmapped_levee_nodestrings = set()  # Only need to unmap one nodestring per unique levee pair
        self._unmapped_levee_comp_ids = set()  # Need to generate new comp_id for levee nodestrings that shared atts.
        self._levee_node_idxs_to_pipe_pt_id = {}  # {(Node1 XmUGrid index, Node2 XmUGrid index): XMS feature point id}
        self._bc_comp = None
        self._flow_forcing_mesh = None
        self._flow_forcing_dsets = []
        self._loc_hashes = {}  # 0-based nodestring to hashed x,y,z coordinates
        self._current_nodestring_id = 0  # For error reporting
        self._temp_bc_comp = None

        # This is only used for testing stupid data_objects Dataset won't give me the filename
        self._flow_forcing_dset_filenames_and_paths = []

        # Strip of " (applied)" from the end of the coverage name if it is there. We append this text to the source
        # coverage name when creating mapped BC components.
        if cov_name.endswith(' (applied)'):
            cov_name = cov_name[:-10]
        self._cov_name = cov_name

        # For initializing coverage component maps in SMS.
        self._pipe_att_to_comp_id = {}
        self._arc_att_to_comp_id = [
            {},  # bcd.UNASSIGNED_INDEX
            {},  # bcd.OCEAN_INDEX
            {},  # bcd.MAINLAND_INDEX
            {},  # bcd.ISLAND_INDEX
            {},  # bcd.RIVER_INDEX
            {},  # bcd.LEVEE_OUTFLOW_INDEX
            {},  # bcd.LEVEE_INDEX
            {},  # bcd.RADIATION_INDEX
            {},  # bcd.ZERO_NORMAL_INDEX
            {},  # bcd.FLOW_AND_RADIATION_INDEX
        ]

    def unmap_data(self):
        """Create a map mmodule coverage from a mapped BC component.

        Returns:
            (:obj:`tuple`): Tuple containing the :obj:`xms.data_objects.parameters.Coverage` geometry, and the
            coverage's hidden :obj:`xms.data_objects.parameters.Component` with the unmapped BC data.
        """
        XmLog().instance.info('Copying applied Boundary Condition attributes to coverage data folder...')
        # Copy the mapped component's mainfile and display options files to the mapped component directory.
        # This assumes we are in the component temp directory.
        os.makedirs(self._bc_comp_dir)  # Create a folder to put the new component's data in
        copy_tree(os.path.dirname(self._mapped_bc_comp.main_file), self._bc_comp_dir, verbose=False)

        # Set the UUID of the parent coverage geometry in the new hidden BC component data.
        bc_main_file = os.path.join(self._bc_comp_dir, bcd.BC_MAIN_FILE)
        from xms.adcirc.components.bc_component import BcComponent
        self._bc_comp = BcComponent(bc_main_file)
        self._bc_comp.data.info.attrs['cov_uuid'] = self._bc_cov_uuid
        self._bc_comp.cov_uuid = self._bc_cov_uuid
        self._bc_comp.data.levees = bcd.default_levee_atts()
        self._bc_comp.data.levee_flags = bcd.default_levee_flags()
        self._bc_comp.data.pipes = bcd.default_pipe_atts()

        # Generate new UUIDs for the arc and point display lists.
        self._update_display_uuids()

        # Recreate the Boundary Conditions coverage's geometry from snapped node locations in mapped component.
        XmLog().instance.info(
            'Creating boundary condition coverage geometry from snapped mesh node locations in the applied BC item...'
        )
        self._create_cov_geometry_and_data()

        # Unmap periodic flow if there is a mapped flow component linked to the simulation.
        self._unmap_periodic_flow()

        # Create the Boundary Conditions coverage's hidden component.
        XmLog().instance.info('Writing applied boundary condition data files.')
        self._bc_comp.data.commit()

        # Write id-based display list files.
        self._write_component_id_files()

        # Create the data_objects component
        do_comp = Component(
            comp_uuid=self._bc_comp_uuid,
            main_file=bc_main_file,
            model_name='ADCIRC',
            unique_name='Bc_Component',
            locked=False
        )
        return self._do_cov, do_comp, self._bc_comp

    def _update_display_uuids(self):
        """Create unique UUIDs for the display lists copied from the mapped BC component."""
        # Set display list and component UUIDs in the new BC coverage's arc display options json file.
        bc_disp = os.path.join(self._bc_comp_dir, os.path.basename(self._mapped_bc_comp.disp_opts_files[0]))
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(bc_disp)
        categories.from_dict(json_dict)
        # Chop off the last category in the list. It is levee pair pipes. In the mapped object, we represent this as
        # a line between the two levee pair mesh nodes the pipe is assigned to. In the BC coverage we represent pipes
        # as disjoint coverage points in the mesh void between levee pair arcs.
        pipe_line_category = categories.categories.pop()
        categories.is_ids = True  # Switch from specifying free locations to coverage component id files.
        categories.uuid = str(uuid.uuid4())
        categories.comp_uuid = self._bc_comp_uuid
        self._bc_comp.data.info.attrs['display_uuid'] = categories.uuid  # Store display UUID in component data.
        write_display_options_to_json(bc_disp, categories)

        # Set display list and component UUIDs in the new BC coverage's point display options json file.
        bc_point_disp = os.path.join(self._bc_comp_dir, BC_POINT_JSON)
        # Copy over some default point display options since the mapped component doesn't draw pipes as points.
        default_opts = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data', DEFAULT_POINT_BC_JSON
        )
        io_util.copyfile(default_opts, bc_point_disp)
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(bc_point_disp)
        categories.from_dict(json_dict)
        categories.is_ids = True  # Switch from specifying free locations to coverage component id files.
        categories.uuid = str(uuid.uuid4())
        categories.comp_uuid = self._bc_comp_uuid
        # Set the color based on line display options.
        categories.categories[0].options.color = pipe_line_category.options.color
        self._bc_comp.data.info.attrs['point_display_uuid'] = categories.uuid  # Store display UUID in component data.
        write_display_options_to_json(bc_point_disp, categories)

    def _create_cov_geometry_and_data(self):
        """Create coverage arcs and pipe points from the mapped BC component.

        Note - I am accessing the underlying numpy arrays in this method instead of using iloc. I found that this
        has much better performance when you need to loop through an entire xarray or pandas object, which is not
        recommended if it can be avoided.
        """
        # Process pipe points before the arcs so we can unmap levee data as we loop through the nodestrings.
        pipe_pts = self._create_pipe_pts_geometry()

        # Build up the coverage arcs from snapped nodestrings.
        num_arcs = self._mapped_bc_comp.data.nodestrings.sizes['dim_0']
        bc_arcs = []
        for arc in range(num_arcs):  # Nodestring id is 0-based index of the row in the dataset (nodestring_id=arc).
            # Get the XmUGrid node indices for the mesh nodes that define this nodestring.
            comp_id = self._mapped_bc_comp.data.nodestrings.comp_id[arc].item()
            is_levee = comp_id in self._mapped_bc_comp.data.levees.comp_id
            node_idx = int(self._mapped_bc_comp.data.nodestrings.nodes_start_idx.data[arc].item())
            num_nodes = int(self._mapped_bc_comp.data.nodestrings.node_count.data[arc].item())

            new_boundary = False
            if arc not in self._unmapped_levee_nodestrings:
                self._current_nodestring_id += 1
                new_boundary = True

            # Convert 1-based node ids to 0-based for XmUGrid call. Keep around the 1-based array for attribute
            # unmapping in case this is a levee boundary.
            one_based_node_ids = self._mapped_bc_comp.data.nodes.id[node_idx:node_idx + num_nodes].astype('i4').data
            node_idxs = (one_based_node_ids - 1)
            node_locs = self._xmgrid.get_points_locations(node_idxs)  # Get node locations of this nodestring definition
            do_points = [  # Only point assign ids to the arc nodes. Vertices should be unique among all nodestrings.
                self._get_do_point(node_idxs[idx], coords, (idx != 0 and idx != len(node_idxs) - 1), is_levee)
                for idx, coords in enumerate(node_locs)
            ]

            if comp_id in self._mapped_bc_comp.data.levees.comp_id:  # This is a levee, use ZCrest for arc elevations
                self._bc_comp.data._arcs['use_elevs'].loc[comp_id] = 1.0
                zcrests = self._mapped_bc_comp.data.levees['Zcrest (m)'].loc[comp_id]
                if len(do_points) != zcrests.size:
                    XmLog().instance.error(f'Inconsistent number of nodes and ZCrest values in nodestring {arc + 1}.')
                else:
                    for i in range(len(zcrests)):
                        do_points[i].z = zcrests[i]

            # Build the data_objects Arc. Convert 0-based nodestring id to 1-based coverage arc id.
            if len(do_points) > 1:
                do_arc = Arc(feature_id=arc + 1, start_node=do_points[0], end_node=do_points[-1])
                if len(do_points) > 2:  # Has vertices
                    do_arc.vertices = do_points[1:len(do_points) - 1]
                bc_arcs.append(do_arc)
            # Create the unmapped arc attributes unless this is an arc that is part of a levee pair and we have already
            # unmapped its partner (only need one set of attributes per pair).
            if new_boundary:
                self._build_bc_arc_dataset(arc, one_based_node_ids, node_locs)

        # Create the data_objects Coverage.
        self._do_cov = Coverage()
        self._do_cov.name = self._cov_name
        self._do_cov.uuid = self._bc_cov_uuid
        self._do_cov.arcs = bc_arcs
        self._do_cov.set_points(pipe_pts)

        # Set the projection of the Coverage to be the projection that the BC locations were mapped in.
        do_proj = Projection(wkt=self._mapped_bc_comp.data.info.attrs['wkt'])
        self._do_cov.projection = do_proj

        self._do_cov.complete()

    def _create_pipe_pts_geometry(self):
        """Create disjoint coverage points in the middle of levee arc pairs with pipes.

        Will also fill in coverage component id maps for later use.

        Returns:
            (:obj:`list`): List of the :obj:`xms.data_objects.parameters.Point` objects for all levee pair pipes in the
            mapped BC component.
        """
        # TODO: This is not ideal because it would be possible to convert a mapped BC object, immediately reapply
        #       the new coverage to the simulation, and get a different mapped object. The two closest nodes to the
        #       point in between two nodes of a levee pair may not necessarily be the two original levee pair nodes.
        #       This is how we used to do it, and we don't think this is a common enough edge case to worry about right
        #       now. So, away we go!
        pipe_levees = self._mapped_bc_comp.data.levees.where(self._mapped_bc_comp.data.levees.Pipe == 1, drop=True)
        num_pipes = pipe_levees.sizes['comp_id']
        # Coverage pipe point locations are not at mesh node locations. Ensure that we generate a new coverage point
        # for each pipe by giving the mesh node indices starting at the number of mesh nodes.
        max_node_idx = self._xmgrid.point_count
        pipe_pts = []
        for pipe_pt in range(num_pipes):
            # Get the mesh node ids of the two levee pair nodes this pipe is assigned to. Convert from 1-based to
            # 0-based.
            node1 = int(pipe_levees['Node1 Id'].data[pipe_pt].item() - 1)
            node2 = int(pipe_levees['Node2 Id'].data[pipe_pt].item() - 1)
            node_locs = self._xmgrid.get_points_locations([node1, node2])
            mid_point = np.mean(node_locs, axis=0)  # Get the midpoint of the segment between the two nodes.

            # Preserve the association between the mapped levee atts and this feature point. Do this before the next
            # line because it will increment self._next_point_id.
            self._levee_node_idxs_to_pipe_pt_id[(node1, node2)] = self._next_point_id
            pipe_pts.append(self._get_do_point(max_node_idx, mid_point, False, False))
            max_node_idx += 1  # Always generate a new data_objects.parameters.Point for each pipe.
        return pipe_pts

    def _get_do_point(self, node_idx, coords, vertex, is_levee):
        """Get the data_objects Point object for a given mesh node. Creates if does not exist.

        Args:
            node_idx (:obj:`int`): 0-based id of the mesh node
            coords (:obj:`Tuple(float)`) x,y,z coordinates for the mesh point to build
            vertex (:obj:`bool`): True if this is a vertex of an arc. In this case the point will have an id of -1,
                otherwise it will get the next available point/node coverage att id.
            is_levee (:obj:`bool`): True if levee

        Returns:
            (:obj:`xms.data_objects.parameters.Point`): See description
        """
        node_pt_exists = node_idx in self._node_idx_to_do_point
        create_dup = (is_levee and not vertex)
        if node_pt_exists is False or create_dup:
            do_point = Point(coords[0], coords[1], coords[2])
            if not vertex:  # Assign the point the next available coverage att id and increment if not an arc vertex.
                do_point.id = self._next_point_id
                self._next_point_id += 1
            self._node_idx_to_do_point[node_idx] = do_point
            return do_point
        do_point = self._node_idx_to_do_point[node_idx]

        # Check for overlapping nodestrings, but don't quit if we find one so that we can report all the problems we
        # find to the user.
        if vertex and do_point.id > 0:
            XmLog().instance.warning(
                f'\nPossible invalid boundary definition detected. Mesh node {node_idx + 1} is a vertex in boundary '
                f'{self._current_nodestring_id} but is a start/end node in another boundary. Please check boundary\n'
                'definitions in the fort.14.\n'
                'Clean feature arcs after coverage is imported to SMS.\n'
            )
        elif not vertex and do_point.id < 1:
            XmLog().instance.warning(
                f'\nPossible invalid boundary definition detected. Mesh node {node_idx + 1} is a start/end node in\n'
                f'boundary {self._current_nodestring_id} but is a vertex in another boundary. Please check boundary '
                'definitions in the fort.14.\n'
                'Clean feature arcs after coverage is imported to SMS.\n'
            )
            # Turn the vertex into a node. The user should clean arcs after importing.
            do_point.id = self._next_point_id
            self._next_point_id += 1
        return do_point

    def _build_bc_arc_dataset(self, nodestring_id, node_ids, node_locs):
        """Build and append arc BC attributes to the source BC coverage component.

        Args:
            nodestring_id (:obj:`int`): 0-based id of the nodestring to build coverage arc data for.
            node_ids (:obj:`list`): List of the 1-based node ids defining the nodestring to convert attributes for.
            node_locs (:obj:`list`): 2-D list where inner dimension contains node coordinates and the outer dimension is
                parallel with node_ids.
        """
        XmLog().instance.info(f'Creating feature arc attributes for nodestring boundary {nodestring_id + 1}...')
        arc_id = nodestring_id + 1  # Convert 0-based nodestring id to 1-based coverage arc id.
        arc_comp_id = int(self._mapped_bc_comp.data.nodestrings.comp_id.data[nodestring_id].item())
        arc_atts = self._bc_comp.data.arcs.sel(comp_id=[arc_comp_id], drop=False)
        arc_type = int(arc_atts.type.item())
        if arc_type in [bcd.LEVEE_INDEX, bcd.LEVEE_OUTFLOW_INDEX]:
            is_pair = arc_type == bcd.LEVEE_INDEX
            arc_comp_id = self._unmap_levee_data(node_ids, node_locs, arc_atts, nodestring_id)
            if arc_comp_id == UNINITIALIZED_COMP_ID:
                # Unable to find the attributes associated with this nodestring. Log an error.
                XmLog().instance.error(f'Unable to assign applied levee attributes to feature arc with id {arc_id}.')
            else:
                self._unmapped_levee_nodestrings.add(nodestring_id)  # Mark this levee nodestring as unmapped.
                # Create lookup mapping of the arc's XMS feature arc att id (att id=nodestring id + 1) to the
                # unmapped attributes.
                self._arc_att_to_comp_id[arc_type][arc_id] = arc_comp_id
                if is_pair:  # Mark the other nodestring as unmapped if a pair.
                    # Get the 0-base nodestring id of the other nodestring in the levee pair.
                    partner_id = int(self._mapped_bc_comp.data.nodestrings.partner_id.data[nodestring_id].item())
                    # Don't unmap atts again when we get to the other nodestring in the pair.
                    self._unmapped_levee_nodestrings.add(partner_id)
                    # Create lookup mapping of the other arc's XMS feature arc att id (att id=nodestring id + 1)
                    # to the unmapped attributes.
                    self._arc_att_to_comp_id[arc_type][partner_id + 1] = arc_comp_id
        elif arc_type == bcd.RIVER_INDEX:
            riv_flow_mask = self._mapped_bc_comp.data.river_flows['comp_id'] == arc_comp_id
            if len(riv_flow_mask) > 0:
                arc_comp_id = self._bc_comp.data.add_bc_atts(arc_atts)
                node_flows = self._mapped_bc_comp.data.river_flows['Flow'].where(riv_flow_mask, drop=True)
                flow_sum_per_ts = node_flows.sum(dim='Node Id')
                ts = node_flows.coords['TS'].values.tolist()

                flow_ds = xr.Dataset(
                    {
                        'Time': (['comp_id'], np.array(ts)),
                        'Flow': (['comp_id'], np.array(flow_sum_per_ts))
                    },
                    coords={
                        'comp_id': np.full(len(ts), arc_comp_id)
                    }
                )
                self._bc_comp.data.q = xr.concat([self._bc_comp.data.q, flow_ds], dim='comp_id')
            self._arc_att_to_comp_id[arc_type][arc_id] = arc_comp_id
        else:
            # If not a levee type, create a mapping from BC type to arc att id to component id. Since we don't
            # have to unmap any attributes for these types, multiple coverage arcs can share the same one.
            self._arc_att_to_comp_id[arc_type][arc_id] = arc_comp_id

    def _unmap_periodic_flow(self):
        """Unmap a mapped periodic flow component associated with the mapped BC component being converted.

        If periodic flow has been defined, the mapped BC component's simulation should also have a mapped flow
        component linked to it. Will create geometry of the river boundary node locations with amplitude and phase
        datasets for each flow constituent.
        """
        if not self._flow_data:
            return  # No periodic flow component linked to the simulation.
        XmLog().instance.info('Converting applied periodic flow boundary data...')

        # Build a geometry of the flow forcing boundary node locations to put the amplitude and phase datasets on.
        geom_uuid = self._build_flow_forcing_mesh()

        # Create an amplitude and phase dataset for each of the defined periodic flow constituents.
        XmLog().instance.info('Creating amplitude and phase data sets from applied periodic flow data...')
        temp_dir = os.path.normpath(os.path.join(self._bc_comp_dir, '../..'))  # Put H5 files in XMS temp directory
        amp_dset_uuids = []
        phase_dset_uuids = []
        # Loop through the defined periodic flow forcing constituents.
        for con_name in self._flow_data.cons.coords['con']:
            # Create an amplitude dataset for each constituent.
            self._build_forcing_dset(temp_dir, con_name.item(), geom_uuid, amp_dset_uuids, False)
            # Create an phase dataset for each constituent.
            self._build_forcing_dset(temp_dir, con_name.item(), geom_uuid, phase_dset_uuids, True)

        # Update the flow forcing options in the converted BC coverage.
        self._bc_comp.data.info.attrs['periodic_flow'] = 1  # Make sure periodic flow is enabled in converted coverage.
        flow_con_data = {
            'Name': xr.DataArray(data=self._flow_data.cons.coords['con'].data),
            'Frequency': xr.DataArray(data=self._flow_data.cons.frequency.data),
            'Nodal __new_line__ Factor': xr.DataArray(data=self._flow_data.cons.nodal_factor.data),
            'Equilibrium __new_line__ Argument': xr.DataArray(data=self._flow_data.cons.equilibrium_argument.data),
            'Amplitude': xr.DataArray(data=np.array(amp_dset_uuids, dtype=object)),
            'Phase': xr.DataArray(data=np.array(phase_dset_uuids, dtype=object)),
        }
        self._bc_comp.data.flow_cons = xr.Dataset(data_vars=flow_con_data)
        # Make the dimension index 1-based for the GUI.
        self._bc_comp.data.flow_cons['dim_0'] = self._bc_comp.data.flow_cons['dim_0'] + 1  # Don't use += here

    def _unmap_levee_data(self, node_ids, node_locs, arc_atts, nodestring_id):
        """Convert mapped levee atts at mesh node locations to component data for the corresponding feature arc.

        Args:
            node_ids (:obj:`list`): List of the 1-based node ids that define the nodestring to convert attributes for.
            node_locs (:obj:`list`): 2-D list where inner dimension contains node coordinates and the outer dimension is
                parallel with node_ids.
            arc_atts (:obj:`xarray.Dataset`): The source BC coverage attributes for this nodestring/arc.
            nodestring_id (:obj:`int`): 0-based nodestring ID

        Returns:
            (:obj:`int`): The component id of the unmapped levee attributes. Should be assigned to both feature arcs of
            the levee pair if not a levee outflow type.
        """
        # Filter the levee dataset by the comp_id coordinate. The coordinate is not unique in this Dataset because there
        # is a row for each node of each nodestring. Furthermore, multiple levees may be sharing the same component id.
        arc_comp_id = int(arc_atts.comp_id.item())
        levee_nodes = self._mapped_bc_comp.data.levees.where(
            self._mapped_bc_comp.data.levees.comp_id == arc_comp_id, drop=True
        )
        # Create a mask to further filter the dataset to rows whose "first" node id is in the arc definition. Each
        # levee node should only appear in a single nodestring's definition.
        levee_nodes_mask = levee_nodes['Node1 Id'].isin(node_ids)
        levee_nodes = levee_nodes.where(levee_nodes_mask, drop=True)
        is_first_nodestring = True
        if not levee_nodes:
            # Check if this is the definition of the "second" nodestring in the pair. Should never be the case if this
            # is a levee outflow type (not a set).
            levee_nodes_mask = levee_nodes['Node2 Id'].isin(node_ids)
            levee_nodes = levee_nodes.where(levee_nodes_mask, drop=True)
            is_first_nodestring = False

        if not levee_nodes.sizes['comp_id']:
            return UNINITIALIZED_COMP_ID  # Couldn't find levee data for this nodestring, give up.

        # Generate a new BC attribute dataset if a nodestring (or nodestring pair) with this component id has already
        # been unmapped.
        new_comp_id = arc_comp_id
        if new_comp_id in self._unmapped_levee_comp_ids:
            new_comp_id = self._bc_comp.data.add_bc_atts(arc_atts)
        self._unmapped_levee_comp_ids.add(new_comp_id)

        # Convert lengths along the arc to parametric lengths between 0.0 and 1.0
        lengths = get_parametric_lengths(node_locs)
        second_lengths = None
        is_pair = arc_atts.type.item() == bcd.LEVEE_INDEX
        if is_pair:  # Use the average of the parametric lengths along the two nodestrings if a levee pair.
            if is_first_nodestring:  # The other nodestring of the pair is the "second" one.
                # Get 0-based node ids of second nodestring.
                second_idxs = (levee_nodes['Node2 Id'].data - 1).astype('i4')
            else:  # The other nodestring of the pair is the "first" one.
                # Get 0-based node ids of first nodestring.
                second_idxs = (levee_nodes['Node1 Id'].data - 1).astype('i4')
            second_locs = self._xmgrid.get_points_locations(second_idxs)
            second_lengths = get_parametric_lengths(second_locs)

            if is_first_nodestring:
                # Store a hash of the levee locations, so we can check if they changed later.
                self._loc_hashes[nodestring_id] = coordinate_hash(node_locs, second_locs)
        else:  # If levee outflow, create a dummy location hash to keep dimensions nice and happy
            self._loc_hashes[nodestring_id] = ''

        self._build_levee_dataset(
            lengths=lengths,
            mapped_levee_dset=levee_nodes,
            levee_comp_id=new_comp_id,
            second_lengths=second_lengths,
            nodestring_id=nodestring_id,
            is_first_nodestring=is_first_nodestring
        )

        # Map pipe attributes if this is a levee pair with pipes defined.
        if is_pair:
            self._add_pipe_point_atts(levee_nodes)
        return new_comp_id

    def _build_levee_dataset(
        self, lengths, mapped_levee_dset, levee_comp_id, second_lengths, nodestring_id, is_first_nodestring
    ):
        """Build and append levee attributes to the source BC coverage component. If levee pair, only one set of atts.

        Args:
            lengths (:obj:`list`): List of floats between 0.0 and 1.0 representing the parametric length along the
                nodestring (or average of parametric lengths along two nodestrings of a levee pair) for each node in the
                nodestring definition.
            mapped_levee_dset (:obj:`xarray.Dataset`): Dataset containing the mapped levee's data.
            levee_comp_id (:obj:`int`): Coverage component id for the levee in the BC coverage component.
            second_lengths (:obj:`list`): The parametric lengths of the second side of a levee pair. None if levee
                outflow.
            nodestring_id (:obj:`int`): 0-based nodestring id
            is_first_nodestring (:obj:`bool`): False if this is the second nodestring in a levee pair, should always be
                True for levee outflows.
        """
        crest_data, len_data, sub_coef_data, super_coef_data = self._get_levee_data(lengths, mapped_levee_dset)
        num_lengths = len(len_data)
        if second_lengths:  # If this is a levee pair, get the attributes of side 2.
            _, len_data2, _, _ = self._get_levee_data(second_lengths, mapped_levee_dset)
        else:  # If a levee outflow, just fill in some dummy values for side 2.
            len_data2 = np.array([0.0] * num_lengths)

        # Generate coordinates for the levee dataset using the new component id and build the dataset.
        comp_ids = {'comp_id': [float(levee_comp_id) for _ in range(num_lengths)]}
        levee_atts = {
            'Parametric __new_line__ Length': ('comp_id', len_data),
            'Parametric __new_line__ Length 2': ('comp_id', len_data2),
            'Zcrest (m)': ('comp_id', crest_data),
            'Subcritical __new_line__ Flow Coef': ('comp_id', sub_coef_data),
            'Supercritical __new_line__ Flow Coef': ('comp_id', super_coef_data),
        }
        levee_dset = xr.Dataset(data_vars=levee_atts, coords=comp_ids)
        self._bc_comp.data.add_levee_atts(levee_dset)

        # There is one entry in the flag dataset per levee.
        if is_first_nodestring:
            comp_ids = {'comp_id': [float(levee_comp_id)]}
            levee_flags = {
                'use_second_side': ('comp_id', [1]),  # Always use side two when unmapping
                'locs': ('comp_id', [self._loc_hashes[nodestring_id]]),
            }
            levee_flag_dset = xr.Dataset(data_vars=levee_flags, coords=comp_ids)
            self._bc_comp.data.add_levee_flags(levee_flag_dset)

    @staticmethod
    def _get_levee_data(lengths, mapped_levee_dset):
        """Get the levee data for a single side.

        Args:
            lengths (:obj:`list`): Parametric lengths of the side
            mapped_levee_dset (:obj:`MappedBcData`): The levee attributes

        Returns:
            (:obj:`tuple`): crest elevations, parametric lengths, subcritical coefficients, supercritical coefficients
        """
        # Create the feature arc levee Dataset and append to source BC data levees. Doing the way we did in 13.0.
        # Seems unnecessarily complex, but I guess it did what we wanted.
        len_data = []  # Add the first row with parametric length of zero.
        crest_data = []
        sub_coef_data = []
        super_coef_data = []
        # We had users that did not want us to collapse duplicate rows. We don't really like it, so if we have the
        # option to collapse in the future, uncomment this junk.
        # Add a row to the levee table anytime the values change from the previous node.
        # last_add_idx = 0
        # last_crest = crest_data[0]
        # last_sub = sub_coef_data[0]
        # last_sup = super_coef_data[0]
        for idx, length in enumerate(lengths):
            crest = mapped_levee_dset['Zcrest (m)'].data[idx].item()
            sub_coef = mapped_levee_dset['Subcritical __new_line__ Flow Coef'].data[idx].item()
            super_coef = mapped_levee_dset['Supercritical __new_line__ Flow Coef'].data[idx].item()
            # Don't add a row for end of previous values if there was only one node.
            #  if crest != last_crest or sub_coef != last_sub or super_coef != last_sup:
            #     len_data.append(lengths[idx - 1])
            #     crest_data.append(mapped_levee_dset['Zcrest (m)'].data[idx - 1].item())
            #     sub_coef_data.append(mapped_levee_dset['Subcritical __new_line__ Flow Coef'].data[idx - 1].item())
            #     super_coef_data.append(
            #         mapped_levee_dset['Supercritical __new_line__ Flow Coef'].data[idx - 1].item()
            #     )
            len_data.append(length)
            crest_data.append(crest)
            sub_coef_data.append(sub_coef)
            super_coef_data.append(super_coef)
            # last_crest = crest
            # last_sub = sub_coef
            # last_sup = super_coef
            # last_add_idx = idx

            # # Add the last row of the table with parametric length of 1.0.
            # if idx == len(lengths) - 1 and last_add_idx != idx and lengths[-1] != 1.0:
            #     len_data.append(1.0)
            #     crest_data.append(crest)
            #     sub_coef_data.append(sub_coef)
            #     super_coef_data.append(super_coef)
        return crest_data, len_data, sub_coef_data, super_coef_data

    def _add_pipe_point_atts(self, mapped_levee_dset):
        """Add all the pipes defined in a mapped levee pair to the source coverage component data.

        Args:
            mapped_levee_dset (:obj:`xarray.Dataset`): Dataset containing the mapped levee pair attributes of the levee
                pair containing the pipe point to add attributes for.
        """
        pipes = mapped_levee_dset.where(mapped_levee_dset.Pipe == 1, drop=True)
        if pipes.sizes['comp_id'] > 0:
            node1s = (pipes['Node1 Id'].data - 1).astype('i4').tolist()  # Convert node ids to 0-based for map lookup.
            node2s = (pipes['Node2 Id'].data - 1).astype('i4').tolist()
            heights = pipes['Zpipe (m)'].data.tolist()
            diameters = pipes['Pipe __new_line__ Diameter (m)'].data.tolist()
            coefs = pipes['Bulk __new_line__ Coefficient'].data.tolist()
            for node1, node2, height, diam, coef in zip(node1s, node2s, heights, diameters, coefs):
                # Create a new pipe dataset.
                pipe_data = {
                    'Height': ('comp_id', [height]),
                    'Coefficient': ('comp_id', [coef]),
                    'Diameter': ('comp_id', [diam]),
                }
                coords = {
                    'comp_id': [UNINITIALIZED_COMP_ID]  # Will generate a valid comp_id when adding to the BcData
                }
                pipe_dset = xr.Dataset(data_vars=pipe_data, coords=coords)
                pipe_comp_id = self._bc_comp.data.add_pipe_atts(pipe_dset)
                # Create mapping from XMS att id to component id for this point. Need to initialize coverage
                # component maps later.
                self._pipe_att_to_comp_id[self._levee_node_idxs_to_pipe_pt_id[(node1, node2)]] = pipe_comp_id

    def _build_flow_forcing_mesh(self):
        """Create a mesh of the flow boundary condition nodes.

        We will add an amplitude and phase dataset on the geometry for each periodic flow constituent defined.

        Returns:
            (:obj:`str`): UUID of the flow forcing mesh that was created
        """
        XmLog().instance.info('Building a geometry of the flow boundary node locations...')
        # Get the flow boundary node locations.
        river_node_ids = [node_id - 1 for node_id in self._mapped_bc_comp.data.get_river_node_ids()]
        flow_forcing_pts = self._xmgrid.get_points_locations(river_node_ids)

        # Build the constrained UGrid
        mesh_uuid = str(uuid.uuid4())
        constraint_file = os.path.join(XmEnv.xms_environ_process_temp_directory(), f'{mesh_uuid}.xmc')
        xmugrid = UGrid(flow_forcing_pts, [])
        co_builder = UGridBuilder()
        co_builder.set_is_2d()
        co_builder.set_ugrid(xmugrid)
        co_grid = co_builder.build_grid()
        co_grid.write_to_file(constraint_file, True)

        # Build the data_objects UGrid
        self._flow_forcing_mesh = DoUGrid(constraint_file, name='Periodic Flow Forcing Nodes', uuid=mesh_uuid)
        do_proj = Projection(wkt=self._mapped_bc_comp.data.info.attrs['wkt'])
        self._flow_forcing_mesh.projection = do_proj

        return mesh_uuid  # Need this to associate datasets with this geometry

    def _build_forcing_dset(self, temp_dir, con_name, geom_uuid, dset_uuids, is_phase):
        """Write a periodic flow forcing dataset to an XMDF formatted file and build the data_object to send to SMS.

        Args:
            temp_dir (:obj:`str`): Path to the XMS temp directory. Where we will write the H5 files
            con_name (:obj:`str`): Name of the periodic flow constituent to create datasets for
            geom_uuid (:obj:`str`): UUID of the flow forcing mesh
            dset_uuids (:obj:`list`): List to append the new dataset's UUID to
            is_phase (:obj:`bool`): True if building the periodic flow phase dataset
        """
        # Select the nodal amplitude or phase dataset for given constituent name.
        if is_phase:
            data_vals = self._flow_data.values.phase.sel(con=con_name).data
            dset_name = f'{con_name} - Phase'
        else:
            data_vals = self._flow_data.values.amplitude.sel(con=con_name).data
            dset_name = f'{con_name} - Amplitude'
        dset_uuids.append(str(uuid.uuid4()))  # Generate a UUID for the dataset. Need to link up to dataset selectors.

        # Set up the dataset builder
        xmdf_filename = os.path.join(temp_dir, f'{dset_uuids[-1]}.h5')
        builder = DatasetWriter(
            h5_filename=xmdf_filename,
            name=dset_name,
            dset_uuid=dset_uuids[-1],
            geom_uuid=geom_uuid,
            time_units='Seconds'
        )
        builder.append_timestep(0.0, data_vals)
        builder.appending_finished()

        # Store the data_objects Dataset to send to XMS later.
        self._flow_forcing_dsets.append(Dataset(xmdf_filename, f'Datasets/{dset_name}', 'NODE', 'NODE'))
        self._flow_forcing_dsets[-1].geom_uuid = geom_uuid  # Link dataset to its geometry

        # Store dataset filename and path for testing purposes.
        self._flow_forcing_dset_filenames_and_paths.append((xmdf_filename, f'Datasets/{dset_name}'))

    def _write_component_id_files(self):
        """Write the component id files for each BC arc type as well as pipe points."""
        XmLog().instance.info('Creating display lists for converted feature arcs and points...')
        # Create display lists for the BC arcs.
        for bc_type in range(bcd.FLOW_AND_RADIATION_INDEX + 1):
            if self._arc_att_to_comp_id[bc_type]:
                loc_file = BcComponentDisplay.get_display_id_file(bc_type, self._bc_comp_dir)
                write_display_option_ids(loc_file, list(self._arc_att_to_comp_id[bc_type].values()))
        # Create the pipe point display list.
        if self._pipe_att_to_comp_id:
            loc_file = os.path.join(self._bc_comp_dir, BC_POINT_ID_FILE)
            write_display_option_ids(loc_file, list(self._pipe_att_to_comp_id.values()))

    def initialize_coverage_comp_ids(self):
        """Initialize the id-based coverage component id maps of the new source BC coverage."""
        # Set the pipe point component ids.
        for att_id, comp_id in self._pipe_att_to_comp_id.items():
            self._bc_comp.update_component_id(TargetType.point, att_id, comp_id)
        # Set the BC arc component ids.
        for id_map in self._arc_att_to_comp_id:
            for att_id, comp_id in id_map.items():
                self._bc_comp.update_component_id(TargetType.arc, att_id, comp_id)
        # Add some display lists for SMS to read for the new BC coverage.
        arc_json = os.path.join(self._bc_comp_dir, BC_JSON)
        pt_json = os.path.join(self._bc_comp_dir, BC_POINT_JSON)
        self._bc_comp.display_option_list = [
            XmsDisplayMessage(file=arc_json, edit_uuid=self._bc_cov_uuid),  # arcs
            XmsDisplayMessage(file=pt_json, edit_uuid=self._bc_cov_uuid),  # points
        ]

    def get_unmapped_flow_data(self):
        """Get the unmapped periodic flow geometry and its datasets.

        Will be an amplitude and phase dataset for every defined flow constituent in the mapped flow component under
        the same simulation as the mapped BC coverage being converted.

        Returns:
            (:obj:`tuple`):
            If periodic flow defined - :obj:`(xms.data_objects.parameters.UGrid, [xms.data_objects.parameters.Dataset]`

            If periodic flow not defined - :obj:`(None, [])`
        """
        return self._flow_forcing_mesh, self._flow_forcing_dsets
