"""Base class for ADCIRC levee tools."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import pickle

# 2. Third party modules
import numpy as np
import pandas as pd
from rtree import index

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.guipy.data.target_type import TargetType
from xms.snap.snap_exterior_arc import SnapExteriorArc
from xms.tool_core import Tool

# 4. Local modules
from xms.adcirc.components.bc_component import BcComponent
from xms.adcirc.mapping import mapping_util as map_util
from xms.adcirc.tools import levee_check_tool_consts as const


class LeveeCheckToolBase(Tool):
    """Base class for the ADCIRC levee check/fix tools."""
    ARG_INPUT_BC_COVERAGE = 0
    ARG_INPUT_GRID = 1

    def __init__(self, tool_type):
        """Initializes the class."""
        super().__init__(name=const.TOOL_NAMES[tool_type])
        self.query = None
        self.results_dialog_module = 'xms.adcirc.gui.check_levee_tool_results_dialog'
        self.results_dialog_class = 'CheckLeveeToolResultsDialog'
        self._tool_type = tool_type
        self._bc_coverage = None
        self._bc_comp = None
        self._threshold = 0.0
        self._cogrid = None
        self._selected_arcs = set()  # Arcs that were selected when tool was brought up, if none check all levees
        self._error_stack = []  # [(level, msg)] - Warnings/errors for a single levee. level in ['Error', 'Warning']
        self._error_dict = dict()
        self._error_levels = dict()
        self._global_error_stack = []  # Warnings/errors that will be reported in the global log section of the results
        self._global_error_dict = dict()
        self._global_error_levels = dict()

        # Keep track of the nodes in each levee definition so we can adjust and warn if a node is multiple levees.
        self._levee_nodes = {}  # {node_idx: [(levee_comp_id, pair_idx, arc1_id, arc2_id, zcrest)]}

        # Snap data we need to store up. We actually have to make two passes on the levees. Once to snap all the arcs
        # and map their attribute curves to the mesh. We then do some checking for nodes defined in multiple levees.
        # Then we loop through all the levees again, performing the tool implementation-specific checks.
        # {
        #     comp_id: {
        #       'arc_id': [], 'snap1': {}, 'snap2': {}, 't_length': [], 'zcrest': [], 'sub_coef': [], 'super_coef': []
        #     }
        # }
        self._snap_data = {}
        self._preprocess_errors = {}  # Errors/warnings encountered on first pass. Report in second pass.

        # Result/report data
        self._num_adjusted = 0
        self._num_adjusted_issue = 0
        self._num_unadjusted = 0
        self._num_unadjusted_issue = 0
        self._results = {    # DataFrame of results data
            # First five columns are 1-D lists with a row for each levee pair reported in the results.
            'Arc 1 ID': [],
            'Arc 2 ID': [],
            'Status': [],  # one of CHECK_STATUS_* const strings
            'log_output': [],
            'status_string': [],
            # Rest of the columns are curves. 2D-lists, with inner list per 'Arc 1 ID'/'Arc 2 ID' column combination.
            'Parametric Length': [],  # X-column of all curves
        }  # Rest of the curves defined by the tool implementation
        self._candidate_levee = {}  # Temp storage used to store a levee's results data during processing
        self._levee_issues = {}  # {comp_id: [issue enum codes]}

    def _add_levee_issue(self, comp_id, enum_code):
        """Flag a check issue type for a levee.

        Args:
            comp_id (:obj:`int`): Component id of the levee
            enum_code (:obj:`int`): One of the const.LEVEE_ISSUE_* enums
        """
        levee_issues = self._levee_issues.setdefault(comp_id, set())
        levee_issues.add(enum_code)
        if enum_code == const.LEVEE_ISSUE_NO_INTERSECT:  # If no intersection at all remove the partial intersect flag
            levee_issues.discard(const.LEVEE_ISSUE_PARTIAL_INTERSECT)

    def _remove_levee_issue(self, comp_id, enum_code):
        """Unflag a check issue type for a levee.

        Args:
            comp_id (:obj:`int`): Component id of the levee
            enum_code (:obj:`int`): One of the const.LEVEE_ISSUE_* enums
        """
        levee_issues = self._levee_issues.get(comp_id)
        if levee_issues:  # If no intersection remove the partial intersect flag
            levee_issues.discard(enum_code)

    def _get_inputs(self, arguments):
        """Get the inputs to the tool from XMS.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`tuple(xarray.Dataset,list)`): The levee dataset and the levee comp_ids.
        """
        self.logger.info('Retrieving input data from SMS...')
        self._bc_coverage = self.get_input_coverage(arguments[self.ARG_INPUT_BC_COVERAGE].value)
        # Copy the H5 coverge geometry dump file so we can read it when displaying the results dialog.
        xfs.copyfile(self._bc_coverage.attrs['filename'], map_util.check_levee_bc_file())
        self._cogrid = self.get_input_grid(arguments[self.ARG_INPUT_GRID].value)
        levee_data, levee_comp_ids = self._retrieve_model_data(self._bc_coverage.attrs['uuid'])
        levee_comp_ids = self._filter_input_to_selection(levee_comp_ids)
        if not levee_comp_ids:  # Either no levee arcs and no selection or no levee arcs selected but other types are
            self.logger.error(
                'Unable to find any levee pair boundaries in the input Boundary Conditions coverage. To check all '
                'levees in the coverage, ensure the feature arc selection is cleared and rerun the tool. To check a '
                'subset of the levee arcs in the coverage, select at least one feature arc in each levee pair of '
                'interest and rerun the tool.'
            )
        return levee_data, levee_comp_ids

    def _filter_input_to_selection(self, levee_comp_ids):
        """Filter list of levee component ids we will process to the currently selected arcs, if any.

        Args:
            levee_comp_ids (:obj:`numpy.ndarray`): The levee component ids

        Returns:
            (:obj:`list[int]`): The levee component ids filtered to the currently selected arcs, if there are any.
            Convert from an ndarray to a native Python list because next step is iterating over the levee component ids.
        """
        if not self._selected_arcs:  # If no arc selection, check all levees in the coverage
            return levee_comp_ids.tolist()

        selected_comp_ids = set()
        for feature_id in self._selected_arcs:
            comp_id = self._bc_comp.get_comp_id(TargetType.arc, feature_id)
            if comp_id > -1:
                selected_comp_ids.add(comp_id)
        return levee_comp_ids[np.isin(levee_comp_ids, list(selected_comp_ids))].tolist()

    def _retrieve_model_data(self, cov_uuid):
        """Retrieve data from the coverage's component (currently only supports ADCIRC BC coverages).

        Args:
            cov_uuid (:obj:`str`): UUID of the coverage geometry

        Returns:
            (:obj:`tuple(xarray.Dataset,numpy.array)`): The levee dataset and the levee comp_ids
        """
        self.logger.info('Loading levee attributes from model data...')
        # Get the ADCIRC BC component
        do_comp = self.query.item_with_uuid(cov_uuid, model_name='ADCIRC', unique_name='Bc_Component')
        self._bc_comp = BcComponent(do_comp.main_file)
        # First ask SMS for a the selected arc's component ids, if any.
        self.query.load_component_ids(self._bc_comp, arcs=True, only_selected=True)
        selections = self._bc_comp.comp_to_xms.get(self._bc_comp.cov_uuid, {}).get(TargetType.arc, {}).values()
        for selection in selections:
            self._selected_arcs.update(selection)
        # Now clear the component id maps and ask SMS for all the currently assigned component ids.
        self._bc_comp.comp_to_xms[self._bc_comp.cov_uuid] = {}
        self.query.load_component_ids(self._bc_comp, arcs=True, points=True)
        self._bc_comp.clean_unused_comp_ids()
        return self._bc_comp.data.levee_pair_data()

    def _get_run_data(self, arguments):
        """Retrieves the data needed to run the tool.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`tuple(dict,list, xarray.Dataset,SnapExteriorArc)`): Mapping of levee feature arc id to data_objects
            Arc, the levee comp ids, the levee attributes, the arc snapper. None to indicate there were no levees in the
            input BC coverage or no levee types selected and checks should not continue.
        """
        levee_data, levee_comp_ids = self._get_inputs(arguments)
        if not levee_comp_ids:
            return None  # No levees to check, don't bother with setting up the snapper
        self.logger.info('Setting up snapper for snapping levee arcs to mesh...')
        snapper = SnapExteriorArc()
        snapper.set_grid(grid=self._cogrid, target_cells=False)
        arcs = self._bc_coverage[self._bc_coverage['geometry_types'] == 'Arc']
        do_arcs = {arc.id: arc for arc in arcs.itertuples()}
        return do_arcs, levee_comp_ids, levee_data, snapper

    def _get_levee_arcs(self, levee_comp_id, do_arcs, snapper):
        """Get the feature arc ids and snap locations of a levee pair.

        Args:
            levee_comp_id (:obj:`int`): Comp id of the levee pair
            do_arcs (:obj:`dict`): Mapping of feature arc id to data_objects Arc
            snapper (:obj:`SnapExteriorArc`): The arc snapper

        Returns:
            (:obj:`tuple(list,dict,dict)`): The feature arc ids of the two levee arcs, the snap result for the first
            levee arc, the snap result for the second levee arc.

            Returns (None,None,None) if levee doesn't exist anymore.

            Returns ([],None,None) if error snapping the arcs.
        """
        # Find the arcs associated with this component id and snap them to the grid.
        arc_ids = self._bc_comp.get_xms_ids(TargetType.arc, levee_comp_id)
        if arc_ids == -1:  # Levee doesn't exist anymore, not an error
            return None, None, None
        snap1, snap2 = self._snap_arcs(levee_comp_id, arc_ids, do_arcs, snapper)
        return arc_ids, snap1, snap2

    def _snap_arcs(self, levee_comp_id, arc_ids, do_arcs, snapper):
        """Snap levee arcs to the domain mesh.

        Args:
            levee_comp_id (:obj:`int`): Comp id of the levee pair
            arc_ids (:obj:`list`): The feature arc IDs of the two levee arcs
            do_arcs (:obj:`dict`): Mapping of feature arc ID to data_objects levee arcs
            snapper (:obj:`SnapExteriorArc`): The arc snapper

        Returns:
            (:obj:`tuple(dict,dict)`): The snap result for each levee arc. None, None if snap invalid
        """
        if len(arc_ids) != 2:  # Should be exactly two arcs in the pair
            msg = 'Invalid levee pair found. Set should contain two arcs, found arcs with following ids in set: ' \
                  f'{arc_ids}'
            self._error_stack.append(('Error', msg))  # Don't log yet, this is a preprocess error
            self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_INVALID_DEFINITION)
            return None, None
        # Snap the coverage arcs to the mesh
        snap1 = snapper.get_snapped_points(list(do_arcs[arc_ids[0]].geometry.coords))
        snap2 = snapper.get_snapped_points(list(do_arcs[arc_ids[1]].geometry.coords))
        if len(snap1['location']) != len(snap2['location']):  # Both arcs should snap to the same number of nodes
            msg = f'Arc {arc_ids[0]} snaps to {len(snap1["location"])} node locations, but arc {arc_ids[1]} of the ' \
                  f'same set snaps to {len(snap2["location"])} node locations. This is invalid for ADCIRC.'
            self._error_stack.append(('Error', msg))  # Don't log yet, this is a preprocess error
            self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_INVALID_DEFINITION)
            return None, None
        # Make sure the two levee arcs go in the same direction. Sum the lengths of the node pair segments, reverse
        # the second arc, and compute the length of those segements. If the sum of lengths of the reversed arc's
        # node pair segments is less than the sum of the lengths with the original orientation, report an error.
        segment_lengths = 0.0
        reverse_segment_lengths = 0.0
        num_pairs = len(snap1['location'])
        for i in range(0, num_pairs):
            node1_loc = snap1['location'][i]
            node2_loc = snap2['location'][i]
            segment_lengths += math.sqrt((node2_loc[0] - node1_loc[0])**2 + (node2_loc[1] - node1_loc[1])**2)
            node2_loc = snap2['location'][num_pairs - i - 1]
            reverse_segment_lengths += math.sqrt((node2_loc[0] - node1_loc[0])**2 + (node2_loc[1] - node1_loc[1])**2)
        if reverse_segment_lengths < segment_lengths:
            msg = f'Arc with ID {arc_ids[0]} appears to have a different direction than arc with ID {arc_ids[1]}. ' \
                  f'Check arc orientation.'
            self._error_stack.append(('Error', msg))  # Don't log yet, this is a preprocess error
            self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_INVALID_DEFINITION)
            return None, None
        return snap1, snap2

    def _get_parametric_snap_lengths_from_elevs(self, levee_comp_id, do_arcs, snap1, snap2):
        """Get the feature arc ids and snap locations of a levee pair.

        Args:
            levee_comp_id (:obj:`int`): Comp id of the levee pair
            do_arcs (:obj:`dict`): Mapping of feature arc id to data_objects Arc

        Returns:
            levee_data (:obj:`xarray.Dataset`): The BC component data's levee Dataset.
        """
        # Find the arcs associated with this component id and snap them to the grid.
        arc_ids = self._bc_comp.get_xms_ids(TargetType.arc, levee_comp_id)
        pts_1 = [(pt[0], pt[1], pt[2]) for pt in list(do_arcs[arc_ids[0]].geometry.coords)]
        pts_2 = [(pt[0], pt[1], pt[2]) for pt in list(do_arcs[arc_ids[0]].geometry.coords)]

        idx_1 = index.Index()
        for i, loc in enumerate(pts_1):
            idx_1.insert(i, [loc[0], loc[1]])
        snap_locs_1 = snap1['location']
        for i in range(len(snap_locs_1)):
            # find the closest arc location and use the z
            nearest = list(idx_1.nearest([snap_locs_1[i][0], snap_locs_1[i][1]]))
            snap_locs_1[i][2] = pts_1[nearest[0]][2]

        idx_2 = index.Index()
        for i, loc in enumerate(pts_2):
            idx_2.insert(i, [loc[0], loc[1]])
        snap_locs_2 = snap2['location']
        for i in range(len(snap_locs_2)):
            # find the closest arc location and use the z
            nearest = list(idx_2.nearest([snap_locs_2[i][0], snap_locs_2[i][1]]))
            snap_locs_2[i][2] = pts_2[nearest[0]][2]

        para_1 = map_util.get_parametric_lengths(snap_locs_1)
        para_2 = map_util.get_parametric_lengths(snap_locs_2)
        zcrests = [max(snap_locs_1[i][2], snap_locs_2[i][2]) for i in range(len(snap_locs_1))]
        subcritical_coeff = float(self._bc_comp.data.arcs['subcritical_coeff'].loc[levee_comp_id].data)
        supercritical_coeff = float(self._bc_comp.data.arcs['supercritical_coeff'].loc[levee_comp_id].data)

        num_rows = len(para_1)
        data_dict = {
            'Parametric __new_line__ Length': para_1,
            'Parametric __new_line__ Length 2': para_2,
            'Zcrest (m)': zcrests,
            'Subcritical __new_line__ Flow Coef': [subcritical_coeff] * num_rows,
            'Supercritical __new_line__ Flow Coef': [supercritical_coeff] * num_rows,
            'comp_id': [levee_comp_id] * num_rows,
        }
        levee = pd.DataFrame.from_dict(data_dict).to_xarray()

        # Sort the levee by length along the line
        sorted_levee = levee.sortby('Parametric __new_line__ Length')
        total_length = sorted_levee['Parametric __new_line__ Length'][-1].data.item()
        # Normalize snapped nodestring lengths to between 0.0-1.0
        return para_1, para_2, sorted_levee, total_length

    def _map_all_levees_for_check(self, arguments):
        """Map all the levee pairs to the input mesh and map their attributes to the snapped mesh node locations.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`bool`): True if we successfully retrieved the input data, False if there was a fatal error and
            further checks should be skipped.
        """
        run_data = self._get_run_data(arguments)
        if not run_data:
            return False  # If there was a problem retrieving the input data, don't bother mapping the levees.
        (do_arcs, levee_comp_ids, levee_data, snapper) = run_data
        self.logger.info('Mapping levee curves to mesh nodes...')
        for levee_comp_id in levee_comp_ids:  # Loop through each levee pair.
            self._map_single_levee_for_check(levee_comp_id, levee_data, do_arcs, snapper)
            # If we encounter errors applying the levees to the mesh, store them up and report during the second pass.
            if self._error_stack:
                self._preprocess_errors[levee_comp_id] = self._error_stack
                self._error_stack = []
        self._check_for_duplicate_levee_nodes()
        return True

    def _map_single_levee_for_check(self, levee_comp_id, levee_data, do_arcs, snapper):
        """Snap a single levee pair to the input mesh and map its attributes to the snapped mesh node locations.

        Args:
            levee_comp_id (:obj:`int`): The levee pair's component id
            levee_data (:obj:`xarray.Dataset`): The BC component data's levee Dataset
            do_arcs (:obj:`dict`): The feature arcs of the input BC coverage, keyed by feature id
            snapper (:obj:`SnapExteriorArc`): The arc snapper
        """
        # Snap the levee feature arcs to the mesh
        arc_ids, snap1, snap2 = self._get_levee_arcs(levee_comp_id, do_arcs, snapper)
        t_lengths = []
        zcrests = []
        sub_coefs = []
        super_coefs = []
        if snap1 is not None:  # Map levee pair's attributes to mesh if snapping was successful.
            (arc1_id, arc2_id) = arc_ids
            node1_idx = snap1['id']
            node2_idx = snap2['id']
            # Get the parametric lengths along the levee of the node pairs and sort the levee curve by length.
            if levee_comp_id not in levee_data['comp_id']:
                p_len1, p_len2, sorted_levee, total_length = self._get_parametric_snap_lengths_from_elevs(
                    levee_comp_id, do_arcs, snap1, snap2)
            else:
                levee = levee_data.sel(dict(comp_id=levee_comp_id))
                p_len1, p_len2, sorted_levee, total_length = self._get_parametric_snap_lengths(levee, snap1, snap2)
            t_lengths = np.mean([p_len1, p_len2], axis=0).tolist()  # These are normalized 0.0-1.0
            # Use t-lengths of node pair locations to map the levee pair's attributes
            for pair_idx, t_length in enumerate(t_lengths):
                zcrest, _ = map_util.map_levee_atts(sorted_levee, t_length * total_length)
                zcrests.append(zcrest)
                sub_coefs.append(1.0)
                super_coefs.append(1.0)
                # Store this levee pair's info for both nodes in case the nodes appear in another levee.
                levee_def = levee_comp_id, pair_idx, arc1_id, arc2_id, zcrest
                self._levee_nodes.setdefault(node1_idx[pair_idx], []).append(levee_def)
                self._levee_nodes.setdefault(node2_idx[pair_idx], []).append(levee_def)
        # Store the levee's mapped data for the second pass
        self._snap_data[levee_comp_id] = {
            'arc_id': arc_ids,
            'snap1': snap1,
            'snap2': snap2,
            't_length': t_lengths,
            'zcrest': zcrests,
            'sub_coef': sub_coefs,
            'super_coef': super_coefs,
        }

    def _check_for_duplicate_levee_nodes(self):
        """If a node appears in multiple levee definitions, warn and adjust levee crest elevations to the minimum."""
        self.logger.info('Checking for nodes in multiple levee definitions...')
        adjust_crest = self._tool_type == const.TOOL_TYPE_GROUND_ELEVATION  # Only adjust crest elevation if ground tool
        for node_idx, levees in self._levee_nodes.items():
            if len(levees) > 1:
                report_table = []  # [str] - Tabular report for the log warning message
                update_levees = []  # [(levee_comp_idx, pair_idx)] - lookup for adjusting crest elevations to minimum
                min_zcrest = float('inf')  # Crest elevation to use at the node for all levees in which it appears
                for levee in levees:  # Think there can really only ever be two levees per node
                    (levee_comp_id, pair_idx, arc1_id, arc2_id, zcrest) = levee
                    min_zcrest = min(min_zcrest, zcrest)
                    update_levees.append((levee_comp_id, pair_idx))
                    report_table.append(f'{arc1_id:<11}{arc2_id:<11}{zcrest:<.6f}')
                    self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_NODE_IN_MULTIPLE_LEVEES)
                self._report_node_in_multiple_levees(node_idx, min_zcrest, report_table, adjust_crest)
                # Update the crest elevation of each levee the node appears in to be the minimum.
                if adjust_crest:
                    for levee_comp_id, pair_idx in update_levees:
                        self._snap_data[levee_comp_id]['zcrest'][pair_idx] = min_zcrest

    @staticmethod
    def _get_parametric_snap_lengths(levee_atts, snap1, snap2):
        """Normalize the snapped levee nodestrings to a length of 0.0-1.0 and sort the attributes by length.

        Args:
            levee_atts (:obj:`xarray.Dataset`): The unmapped attributes for the levee
            snap1 (:obj:`dict`): Snap result for the first arc in the levee pair
            snap2 (:obj:`dict`): Snap result for the second arc in the levee pair

        Returns:
            (:obj:`tuple(list, list, xr.Dataset, float`)): Parametric lengths for the first levee arc, parametric
            lengths for the second levee arc, the levee attributes sorted by length along the nodestring, the total
            length of the nodestring.
        """
        # Sort the levee by length along the line
        sorted_levee = levee_atts.sortby('Parametric __new_line__ Length')
        total_length = sorted_levee['Parametric __new_line__ Length'][-1].data.item()
        # Normalize snapped nodestring lengths to between 0.0-1.0
        para_len1 = map_util.get_parametric_lengths(snap1['location'])
        para_len2 = map_util.get_parametric_lengths(snap2['location'])
        return para_len1, para_len2, sorted_levee, total_length

    def _clear_global_error_stack(self):
        """Clear the global error stack and format for the results dialog.

        Returns:
            (:obj:`str`): Formatted global warning and error messages for displaying in the results dialog.
        """
        log_output = []
        self._global_error_stack.extend(self._get_error_msgs_from_dicts(self._global_error_dict,
                                                                        self._global_error_levels))
        for level, msg in self._global_error_stack:
            if level == 'Error':
                msg = f'{const.LOG_LEVEL_ERROR}{msg}'  # Display message in red text in results dialog
            elif level == 'Warning':
                msg = f'{const.LOG_LEVEL_WARNING}{msg}'  # Display message in orange text in results dialog
            log_output.append(msg)
        self._global_error_dict.clear()
        self._global_error_levels.clear()
        self._global_error_stack = []

        return f'\n{const.LOG_MESSAGE_DELIM}'.join(log_output)

    def _clear_error_stack(self, levee_comp_id):
        """Clear and report errors for a single levee pair.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee to report

        Returns:
            (:obj:`tuple(str,str)`): Levee check status, formatted log output messages for displaying in the
            results dialog
        """
        log_output = []
        status = const.CHECK_STATUS_UNADJUSTED
        # Report all warning/error messages from the first and second passes at the same time.
        stacked_messages = self._preprocess_errors.get(levee_comp_id, [])
        stacked_messages.extend(self._error_stack)
        stacked_messages.extend(self._get_error_msgs_from_dicts(self._error_dict, self._error_levels))
        for level, msg in stacked_messages:
            msg = f'{const.LOG_MESSAGE_BOLD}{msg}'
            if level == 'Error':
                status = const.CHECK_STATUS_UNADJUSTED_ISSUE
                log_output.append(f'{const.LOG_LEVEL_ERROR}{msg}')  # Display message in red text in results dialog
            else:
                status = const.CHECK_STATUS_UNADJUSTED_ISSUE
                log_output.append(f'{const.LOG_LEVEL_WARNING}{msg}')  # Display message in orange text in results dialog
        self._error_stack = []
        self._error_dict.clear()
        self._error_levels.clear()
        return status, f'\n{const.LOG_MESSAGE_DELIM}'.join(log_output)

    def _get_error_msgs_from_dicts(self, error_dict, error_levels):
        """Clear and report errors for a single levee pair.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee to report

        Returns:
            (:obj:`tuple(str,str)`): Levee check status, formatted log output messages for displaying in the
            results dialog
        """
        msg_stack = []
        for msg in error_dict.keys():
            ids = error_dict[msg]
            level = error_levels[msg]

            error_msg = f'{msg}\n'
            first = True
            for i, id_pair in enumerate(ids):
                if not first:
                    error_msg += ', '
                if i % 5 == 0:
                    error_msg += '\n'
                error_msg += f'{id_pair}'
                first = False

            msg_stack.append((level, error_msg))
        return msg_stack

    def _store_node_pair_row(self, *args):
        """Store the result plot variables for a levee node pair as its being processed in case we adjust and plot it.

        Args:
            *args: The result plot variables for the node pair
        """
        for arg, curve in zip(args, self._candidate_levee.values()):
            curve.append(arg)

    def _append_levee_to_results(self, arc1_id, arc2_id, status, log_output, status_string):
        """Add an adjusted levee's results curves to the plot DataFrame.

        Args:
            arc1_id (:obj:`int`): Feature arc id of the first arc in the levee pair
            arc2_id (:obj:`int`): Feature arc id of the second arc in the levee pair
            status (:obj:`str`): The check status of the levee. One of the CHECK_STATUS_* constants
            log_output (:obj:`str`): The logging output for this levee
            status_string (:obj:`str`): String summarizing the status of the levee
        """
        self._results['Arc 1 ID'].append(arc1_id)
        self._results['Arc 2 ID'].append(arc2_id)
        self._results['log_output'].append(log_output)
        self._results['status_string'].append(status_string)
        self._results['Status'].append(status)
        for key, curve in self._candidate_levee.items():
            self._results[key].append(curve)

    def _reset_candidate_levee(self):
        """Reset the temp storage for levee result plot curves."""
        # Skip the columns in the results DataFrame that are of a lower dimension than the curves. For each levee pair
        # row (arc1_id, arc2_id, adjusted, selected, curve1, ..., curveN), there are n-number of curves.
        self._candidate_levee = {
            key: []
            for idx, key in enumerate(self._results) if idx >= const.RESULTS_START_CURVE_COL_IDX
        }

    def _report_node_in_multiple_levees(self, node_idx, min_zcrest, report_table, adjust_crest):
        """Log a warning for a node that appears in multiple levees and store as a global warning for results dialog.

        Args:
            node_idx (:obj:`int`): 0-base index of the mesh node that appears in multiple levees
            min_zcrest (:obj:`float`): The minimum crest elevation assigned at the node across all levees
            report_table (:obj:`list[str]`): Formatted report table row for each of the levees the node appears in:
                ::
                    Arc 1 ID   Arc 2 ID   Crest Elevation  <-- Header
                    1          2          5.123456         <-- Levee row

            adjust_crest (:obj:`bool`): True if we are updating the crest elevation defined in the levees.
        """
        if adjust_crest:
            log_msg = f'{const.LOG_MESSAGE_BOLD}Mesh node with id {node_idx + 1} appears in multiple levee ' \
                      'definitions. The behavior of ADCIRC at this node is undefined. The minimum crest elevation ' \
                      'will be used at this node for all the levees.\n'
        else:
            log_msg = f'{const.LOG_MESSAGE_BOLD}Mesh node with id {node_idx + 1} appears in multiple levee ' \
                      'definitions. The behavior of ADCIRC at this node is undefined.\n'
        log_msg += 'Arc 1 ID   Arc 2 ID   Crest Elevation\n'
        log_msg += '\n'.join(report_table)
        log_msg += f'\nMinimum crest elevation = {min_zcrest:<.6f}'
        self._global_error_stack.append(('Warning', log_msg))

    def _get_status_string(self, levee_comp_id, adjusted):
        """Get a string summarizing the issues we encountered checking a levee.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee
            adjusted (:obj:`bool`): True if we adjusted elevation

        Returns:
            (:obj:`str`): See description
        """
        status = f'Adjusted: {const.LOG_MESSAGE_DELIM}' if adjusted else f'Unadjusted: {const.LOG_MESSAGE_DELIM}'
        issues = []
        if levee_comp_id in self._levee_issues:
            levee_issues = self._levee_issues[levee_comp_id]
            for levee_issue in sorted(levee_issues):
                issues.append(const.LEVEE_ISSUE_TEXT[levee_issue])
        issue_text = ', '.join(issues) if issues else 'No issues found'
        return status + issue_text

    def _report_levee_check(self, levee_comp_id, arc1_id, arc2_id, adjustments, total_pairs, min_check, max_check):
        """Print a table of adjustments for a levee if we adjusted its defined crest elevations.

        Also stores the result plot data for the levee if it was adjusted.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee to report
            arc1_id (:obj:`int`): Feature arc id of the first arc in the levee pair
            arc2_id (:obj:`int`): Feature arc id of the second arc in the levee pair
            adjustments (:obj:`list`): List of the adjustment strings. Fixed format based on tool-defined header to
                give the appearance of a table.
            total_pairs (:obj:`int`): The total number of node pairs on this levee
            min_check (:obj:`float`): Minimum elevation of source used for checks (check geometry or levee crest curve)
            max_check (:obj:`float`): Maximum elevation of source used for checks (check geometry or levee crest curve)
        """
        status, log_output = self._clear_error_stack(levee_comp_id)
        levee_issues = self._levee_issues.get(levee_comp_id, [])  # So we can check if levee was part of global issue
        if adjustments:  # At least one adjustment, print the report table
            adj_log = self._report_adjusted_levee(arc1_id, arc2_id, adjustments, total_pairs, min_check, max_check)
            log_output = f'{adj_log}\n{const.LOG_MESSAGE_DELIM}{log_output}' if log_output else adj_log

            if status == const.CHECK_STATUS_UNADJUSTED and not levee_issues:  # Levee had no specific or global issues
                self._num_adjusted += 1
                status = const.CHECK_STATUS_ADJUSTED  # No warnings/errors but was adjusted
            else:  # Levee had at least one specific or global warning/error
                self._num_adjusted_issue += 1
                status = const.CHECK_STATUS_ADJUSTED_ISSUE  # Had warnings/errors but we still made adjustments
        elif status == const.CHECK_STATUS_UNADJUSTED and not levee_issues:  # Didn't adjust, no specific or global issue
            log_output = self._report_passed_levee(arc1_id, arc2_id)  # All good, goes to green in report
        else:  # Didn't adjust anything but had at least one specific or global warning/error
            self._num_unadjusted_issue += 1
            # Ensure we report an issue status in case we had a global warning/error but no specific ones
            status = const.CHECK_STATUS_UNADJUSTED_ISSUE

        # Add plots for each levee if, even if we didn't adjust
        status_string = self._get_status_string(levee_comp_id, len(adjustments) > 0)
        self._append_levee_to_results(arc1_id, arc2_id, status, log_output, status_string)
        self._reset_candidate_levee()

    def _report_passed_levee(self, arc1_id, arc2_id):
        """Log a success message for a checked levee that passed and format message for the results dialog.

        Args:
            arc1_id (:obj:`int`): Feature arc id of the first arc in the levee pair
            arc2_id (:obj:`int`): Feature arc id of the second arc in the levee pair

        Returns:
            (:obj:`str`): The success report log message formatted for the results dialog
        """
        self._num_unadjusted += 1
        log_output = f'{const.LOG_MESSAGE_BOLD}No checks violated for levee with arcs {arc1_id} and {arc2_id}.'
        log_output = f'{const.LOG_LEVEL_SUCCESS}{log_output}'  # Display success message as bold green in dialog
        return log_output

    def _report_adjusted_levee(self, arc1_id, arc2_id, adjustments, total_pairs, min_check, max_check):
        """Logs the report for an adjusted levee and builds the formatted log output string for the results dialog.

        Args:
            arc1_id (:obj:`int`): Feature arc id of the first arc in the levee pair
            arc2_id (:obj:`int`): Feature arc id of the second arc in the levee pair
            adjustments (:obj:`list[str]`): The fixed format adjustment rows for adjusted node pairs. Format should
                match header column format defined in the tool's implementation.
            total_pairs (:obj:`int`): Total number of node pairs in the levee
            min_check (:obj:`float`): Minimum elevation of source used for checks (check geometry or levee crest curve)
            max_check (:obj:`float`): Maximum elevation of source used for checks (check geometry or levee crest curve)

        Returns:
            (:obj:`str`): The adjustment report log messages formatted for the results dialog
        """
        adjustments_table = '\n'.join(adjustments)
        elevation_src = const.TOOL_ELEVATION_SOURCE[self._tool_type]
        header = f'{const.LOG_MESSAGE_BOLD}Adjustments for levee with arcs {arc1_id} and {arc2_id}\n' \
                 f'Minimum {elevation_src} elevation = {min_check}\nMaximum {elevation_src} elevation = {max_check}'
        summary = f'{len(adjustments)} of {total_pairs} pairs adjusted :'
        report = f'{summary}\n{const.TOOL_REPORT_HEADERS[self._tool_type]}\n{adjustments_table}'
        log_output = f'{header}\n{report}'  # Display adjustment report as bold black in dialog
        return log_output

    def _report_summary(self):
        """Print the total number of checked, skipped, and adjusted levees after the tool finishes running."""
        msg = f'{const.LOG_MESSAGE_BOLD}Check summary\n' \
              f'Number of levees adjusted without issues = {self._num_adjusted}\n' \
              f'Number of levees adjusted with issues = {self._num_adjusted_issue}\n' \
              f'Number of levees unadjusted without issues = {self._num_unadjusted}\n' \
              f'Number of levees unadjusted with issues = {self._num_unadjusted_issue}\n'
        self.logger.info(msg)
        self._global_error_stack.append(('', msg))

    def _report_results(self):
        """Print the final summary report and write the results file for plotting, if applicable."""
        self._report_summary()
        df = pd.DataFrame(self._results)
        df.attrs['tool_type'] = self._tool_type
        # Store the projection info in the DataFrame, does not reliably get serialized to the H5 file.
        projection = self.query.display_projection
        df.attrs['wkt'] = projection.well_known_text
        df.attrs['coord_sys'] = projection.coordinate_system.upper()
        df.attrs['horiz_units'] = projection.horizontal_units.upper()
        df.attrs['vert_units'] = projection.vertical_units.upper()
        with open(map_util.check_levee_results_df_file(), 'wb') as f:
            pickle.dump({'df': df, 'global_log': self._clear_global_error_stack()}, f)

    def set_data_handler(self, data_handler):
        """Set up query attribute if we have a XMSDataHandler."""
        super().set_data_handler(data_handler)
        if hasattr(self._data_handler, "_query"):
            self.query = self._data_handler._query

    def initial_arguments(self):
        """Get initial arguments for tool - must override.

        Returns:
            (:obj:`list`): A list of the initial tool arguments.
        """
        pass

    def run(self, arguments):
        """Run the tool - must override.

        Args:
            arguments (:obj:`list`): The tool arguments.
        """
        pass
