"""Uses CoverageMapper to map data to AdH geometry."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.adh.components.bc_conceptual_component import BcConceptualComponent
from xms.adh.components.material_conceptual_component import MaterialConceptualComponent
from xms.adh.components.sediment_constituents_component import SedimentConstituentsComponent
from xms.adh.components.sediment_material_conceptual_component import SedimentMaterialConceptualComponent
from xms.adh.components.transport_constituents_component import TransportConstituentsComponent
from xms.adh.mapping.coverage_mapper import CoverageMapper


class CoverageMapperRunner(QThread):
    """Class for mapping coverages to a geometry for AdH."""
    processing_finished = Signal()

    def __init__(self, query):
        """Constructor.

        Args:
            query (Query): The object for getting information from XMS.
        """
        super().__init__()
        self._existing_mapped_component_uuids = []
        self._query = query
        self._logger = logging.getLogger('xms.adh')
        self._grid_file = ''
        self._sim_uuid = ''
        self._sim_item = None

    def run(self):
        """Creates the snap preview of coverages onto the mesh."""
        try:
            worker = CoverageMapper(generate_snap=True)
            self._get_domain(worker)
            self._get_bc_coverage(worker)
            self._get_material_coverage(worker)
            self._get_sediment_material_coverage(worker)
            self._get_uuids_of_existing_mapped_components()
            worker.do_map()

            # query_helper.mapped_comps = worker.mapped_comps
            self._add_mapped_components_to_xms(worker)
            if self._grid_file:
                os.remove(self._grid_file)
        except:  # noqa
            self._logger.exception('Error generating snap.')
        finally:
            self.processing_finished.emit()

    def _get_domain(self, worker):
        """Gets the mesh geometry of the domain.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        # get the mesh geometry.
        self._sim_uuid = self._query.parent_item_uuid()
        self._sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, self._sim_uuid)
        mesh_item = tree_util.descendants_of_type(
            self._sim_item, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, recurse=False, only_first=True
        )
        ugrid = self._query.item_with_uuid(mesh_item.uuid)
        self._grid_file = ugrid.cogrid_file
        co_grid = read_grid_from_file(self._grid_file)
        wkt = ugrid.projection.well_known_text
        worker.set_mesh(co_grid, wkt)

    def _get_bc_coverage(self, worker):
        """Gets the boundary condition coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        bc_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Boundary Conditions',
            model_name='AdH',
            recurse=False,
            only_first=True
        )
        if not bc_item:
            raise RuntimeError('Could not find Boundary Condition coverage.')
        main_file = self._query.item_with_uuid(bc_item.uuid, model_name='AdH', unique_name='BcConceptual').main_file
        bc_comp = BcConceptualComponent(main_file)
        self._query.load_component_ids(bc_comp, arcs=True, points=True)
        bc_cov = self._query.item_with_uuid(bc_item.uuid)
        worker.set_boundary_conditions(bc_cov, bc_comp)

        transport_uuid = bc_comp.data.info.attrs['transport_uuid']
        sediment_uuid = bc_comp.data.info.attrs['sediment_uuid']
        if transport_uuid:
            transport_comp = self._query.item_with_uuid(transport_uuid)
            if transport_comp:
                trans_comp = TransportConstituentsComponent(transport_comp.main_file)
                worker.set_transport_constituents(trans_comp)

        if sediment_uuid:
            sediment_comp = self._query.item_with_uuid(sediment_uuid)
            if sediment_comp:
                sed_comp = SedimentConstituentsComponent(sediment_comp.main_file)
                worker.set_sediment_constituents(sed_comp)

    def _get_material_coverage(self, worker):
        """Gets the materials coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        mat_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Materials',
            model_name='AdH',
            recurse=False,
            only_first=True
        )
        if not mat_item:
            return
        main_file = self._query.item_with_uuid(
            mat_item.uuid, model_name='AdH', unique_name='MaterialConceptual'
        ).main_file
        mat_comp = MaterialConceptualComponent(main_file)
        self._query.load_component_ids(mat_comp, polygons=True)
        mat_cov = self._query.item_with_uuid(mat_item.uuid)
        worker.set_materials(mat_cov, mat_comp)

    def _get_sediment_material_coverage(self, worker):
        """Gets the sediment materials coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        sed_item = tree_util.descendants_of_type(
            self._sim_item,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            coverage_type='Sediment Materials',
            model_name='AdH',
            recurse=False,
            only_first=True
        )
        if not sed_item:
            return
        main_file = self._query.item_with_uuid(
            sed_item.uuid, model_name='AdH', unique_name='SedimentMaterialConceptual'
        ).main_file
        mat_comp = SedimentMaterialConceptualComponent(main_file)
        self._query.load_component_ids(mat_comp, polygons=True)
        mat_cov = self._query.item_with_uuid(sed_item.uuid)
        worker.set_sediment_materials(mat_cov, mat_comp)

    def _add_mapped_components_to_xms(self, worker):
        """Add mapped components to the XMS project.

        Args:
            worker (CoverageMapper): A worker.
        """
        self._logger.info('Adding mapped display items.')
        # delete any existing mapped components
        for str_uuid in self._existing_mapped_component_uuids:
            self._query.delete_item(str_uuid)

        for mapped_comp in worker.mapped_comps:  # Will be None if we logged an error during the mapping operation.
            self._query.add_component(do_component=mapped_comp[0], actions=mapped_comp[1])
            self._query.link_item(taker_uuid=self._sim_uuid, taken_uuid=mapped_comp[0].uuid)

    def _get_uuids_of_existing_mapped_components(self):
        """Gets the uuids of any mapped components that are part of this simulation."""
        mapped_items = tree_util.descendants_of_type(
            self._sim_item, xms_types=['TI_COMPONENT'], model_name='AdH', recurse=False, unique_name='Mapped_Component'
        )
        for mapped_item in mapped_items:
            self._existing_mapped_component_uuids.append(mapped_item.uuid)
