"""VesselComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase as GmiCoverageComponent
from xms.gmi.data.generic_model import Section
from xms.gmi.gui.section_dialog import SectionDialog
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.adh.data.model import get_model
from xms.adh.data.vessel_data import VesselData


class VesselComponent(GmiCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the model's simulation."""
    def __init__(self, main_file):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.tree_commands.insert(0, ('Vessel Parameters...', self._open_vessel_parameters))
        self.default_tree_command = self._open_vessel_parameters

    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        return get_model().section_from_target_type(target)

    def get_arc_section(self):
        """Get the arc section with parameters."""
        return self._get_section(TargetType.arc)

    @cached_property
    def data(self) -> VesselData:
        """The component's data manager."""
        return VesselData(self.main_file)

    def _open_vessel_parameters(self, query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Run the vessel parameters dialog.

        Args:
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        section = self.data.vessel_data

        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=f'{self.module_name}.vessel',
            window_title='Vessel Parameters',
            **self._section_dialog_keyword_args,
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            self.data.vessel_values = values
            self.data.commit()
        return [], []
