"""StationData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import importlib.metadata
import os

# 2. Third party modules
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase

# 4. Local modules


class OutputData(XarrayBase):
    """Manages the data file for the output coverage hidden component."""
    def __init__(self, data_file: str):
        """Initializes the data class.

        Args:
            data_file: The netcdf file (including the path) associated with this instance data. Probably
                the owning component's main file.
        """
        # Initialize member variables before calling super so they are available for commit() call
        self._filename = data_file
        self._info = None
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets(str(data_file))
        super().__init__(data_file)

    @property
    def _xms_adh_version(self) -> str:
        """Get the current version of xmsadh.

        Returns:
            Current version of xmsadh.
        """
        version = importlib.metadata.version('xmsadh')
        return version

    def _get_default_datasets(self, data_file: str):
        """Create default datasets if needed.

        Args:
            data_file: Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'ADH_OUTPUT',
                'VERSION': self._xms_adh_version,
                'cov_uuid': '',
            }
            self._info = xr.Dataset(attrs=info)
            self.commit()

    @property
    def version(self):
        """Get the current file version.

        Returns:
            The current file version.
        """
        return self.info.attrs['VERSION']

    @property
    def cov_uuid(self):
        """Get the coverage UUID.

        Returns:
            The coverage UUID.
        """
        return self.info.attrs['cov_uuid']

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space."""
        if self._info is None:
            self._info = self.get_dataset('info', False)
        self.commit()  # Rewrite all datasets
