"""A dialog for assigning flux values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adh.gui.flux_dialog_ui import Ui_FluxDialog


class FluxDialog(XmsDlg):
    """A dialog for assigning flux output to arcs."""
    def __init__(self, win_cont):
        """Allows the user to edit arc friction.

        Args:
            win_cont (QWidget): Parent window
        """
        super().__init__(win_cont, 'xms.adh.gui.flux_dialog')
        self.ui = Ui_FluxDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        self._setup()

    def _setup(self):
        """Sets up any connections and last minute changes in the dialog."""
        self.ui.is_flux.stateChanged[int].connect(self._enable_flux)
        # QDialogButtonBox with Ok, Cancel, and Help buttons
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def _enable_flux(self, is_flux):
        """Called when the state of the flux output toggle changes. Enables option if checked."""
        self.ui.is_edgestring.setEnabled(is_flux)
        self.ui.is_midstring.setEnabled(is_flux)

    def set_flux(self, is_flux, is_edgestring, is_midstring):
        """Sets the flux output options.

        Args:
            is_flux (bool): True if this is a flux output.
            is_edgestring (bool): True if this flux output is going to an edgestring.
            is_midstring (bool): True if this flux output is going to a midstring.
        """
        self.ui.is_flux.setChecked(is_flux)
        self.ui.is_edgestring.setChecked(is_edgestring)
        self.ui.is_midstring.setChecked(is_midstring)
        self._enable_flux(is_flux)

    def get_flux(self):
        """Sets the flux output options.

        Returns:
            A tuple of three bools: True if this is a flux output, true if this flux output is going to an edgestring,
            true if this flux output is going to a midstring.
        """
        return self.ui.is_flux.isChecked(), self.ui.is_edgestring.isChecked(), self.ui.is_midstring.isChecked()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the user hits OK."""
        return super().accept()

    def reject(self):
        """Called when the user hits Cancel."""
        return super().reject()
