"""A dialog for the transport constituents assignment to strings."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem
from xms.api.tree import tree_util
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adh.gui.transport_constituent_assignment_dialog_ui import Ui_TransportConstituentAssignmentDialog
from xms.adh.gui.widgets.transport_constituent_assignment_widget import TransportConstituentAssignmentWidget
from xms.adh.gui.widgets.transport_constituent_bc_assignment_widget import TransportConstituentBCAssignmentWidget


class TransportConstituentAssignmentDialog(XmsDlg):
    """A dialog for assigning transport constituents for various string types."""
    def __init__(self, win_cont, title, pe_tree, use_transport=True, use_sediment=True, is_arc=None, time_series=None):
        """Allows the user to edit transport constituents.

        Args:
            win_cont (QWidget): Parent window
            title (str): Window title
            pe_tree (TreeNode): The project explorer tree.
            use_transport (bool): True if transport is being used.
            use_sediment (bool): True if sediment transport is being used.
            is_arc (bool): True if this should be used for arcs.
            time_series (dict): A dictionary of TimeSeries with an integer key.
        """
        super().__init__(win_cont, 'xms.adh.gui.transport_constituent_assignment_dialog')
        self.ui = Ui_TransportConstituentAssignmentDialog()
        self.ui.setupUi(self)
        # We need a copy of the project explorer tree since the TransportConstituentAssignmentWidget will modify it.
        sed_pe_tree = tree_util.trim_tree_to_items_of_type(pe_tree, [ComponentItem])  # This makes a copy.
        if is_arc is None:
            self.transport = TransportConstituentAssignmentWidget(self, pe_tree, use_transport, False)
            self.transport.hide_for_selection_only()
            self.sediment = TransportConstituentAssignmentWidget(self, sed_pe_tree, use_sediment, True)
            self.sediment.hide_for_selection_only()
        else:
            self.transport = TransportConstituentBCAssignmentWidget(
                self, pe_tree, is_arc, time_series, use_transport, False
            )
            self.sediment = TransportConstituentBCAssignmentWidget(
                self, sed_pe_tree, is_arc, time_series, use_sediment, True
            )
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Sediment_Transport_and_Bed_Layers'

        self.setWindowTitle(title)
        self._setup()

    def _setup(self):
        """Sets up the transport constituent assignment widget."""
        self.ui.top_layout.insertWidget(0, self.transport)
        self.ui.top_layout.insertWidget(1, self.sediment)
        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)
