"""This module defines a table widget that can be used for defining bed layers."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Signal

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules
from xms.adh.gui.bed_layer_model import BedLayerModel
from xms.adh.gui.grain_size_distribution_delegate import GrainSizeDistributionDelegate


class BedLayerTableWidget(BasicTableWidget):
    """Bed layer table widget class."""
    added = Signal(int)
    deleted = Signal(list)

    def __init__(self, parent, data, material_id, constituents, is_fixed=False):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data (SedimentMaterialsIO): The bed layer data.
            material_id (int): The material id to use. An id of 0 means that this is used for global settings.
            constituents (SedimentConstituentsIO): The sediment constituent data.
            is_fixed (bool): False if adding and deleting rows is allowed.
        """
        super().__init__(parent)
        self.data = data
        self.material_id = material_id
        data_frame = self.data.materials[self.material_id].bed_layers
        if len(data_frame.index) > 0 and data_frame.index[0] == 0:
            data_frame.index += 1  # Make vertical header 1-based
        self.model = QxPandasTableModel(data_frame, self)
        self.default_values = [0, 0.0, 0.0, 0.0, 0.0, 0.0]  # in source mode order
        self.model.set_default_values([self.default_values])
        self.select_col = 1
        self.allow_grain_size_reset = False

        self.dbl_validator = QxDoubleValidator()
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.grain_delegate = GrainSizeDistributionDelegate(self.data, self.material_id, self)
        self.grain_delegate.set_constituents(constituents)
        delegates = {
            0: self.edit_delegate,
            1: self.grain_delegate,
            2: self.edit_delegate,
            3: self.edit_delegate,
            4: self.edit_delegate,
            5: self.edit_delegate
        }

        super()._setup_ui(delegates, False, is_fixed, BedLayerModel(), material_id == 0)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.table_view.filter_model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
        new_index = self.model.index(row_idx, 0)
        self.model.setData(new_index, self.data.info.attrs['next_bed_layer_id'])
        self.data.info.attrs['next_bed_layer_id'] += 1

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )

    def _enable_delete(self, selected, deselected):
        """Enable/disable delete row button.

        Args:
            selected (QItemSelection): Selected indices
            deselected (QItemSelection): Deselected indices, unused

        """
        indices = selected.indexes()
        enable_delete = False
        if len(indices) > 0 and indices[0].isValid():
            enable_delete = True
        # disable the delete button if the unassigned/OFF material row (0) is selected
        for idx in indices:
            if idx.row() == 0:
                enable_delete = False
                break
        self.toolbar.widgetForAction(self.btn_actions[self.delete_icon]).setEnabled(enable_delete)

    def constituents_changed(self, constituents):
        """Called when the sediment constituents have changed.

        Args:
            constituents (SedimentConstituentsIO): The new sediment constituent data.
        """
        self.grain_delegate.set_constituents(constituents, self.allow_grain_size_reset)

    def bed_layer_type_changed(self, protocol):
        """Called when the user has changed the bed layer assignment protocol.

        Args:
            protocol (int): 0 if bed layer thickness assigned directly on SB cards, 1 if bed layer thickness assigned
            by strata elevation horizon on SB cards.
        """
        self.table_view.filter_model.set_bed_layer_type(protocol)
