"""This module defines a table widget that can be used for defining bed layers."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.widgets.adh_table_widget import AdhTableWidget


class ConsolidationModel(RenameModel):
    """A model to rename header titles, hide unwanted options."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        self._ID_SOURCE_COLUMN = 0
        super().__init__(column_names, parent)

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Filters out the 'ID' column.

        Args:
            source_column (int): The column index in the source model.
            source_parent (QModelIndex): The parent index of the view.

        Returns:
            True if the model should keep the column, false if the column should be hidden.
        """
        return source_column != self._ID_SOURCE_COLUMN


class ConsolidationTableWidget(AdhTableWidget):
    """Consolidation table widget class."""
    def __init__(self, parent, data, material_id, is_fixed=False):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data (SedimentMaterialsIO): The bed layer data.
            material_id (int): The material id to use. An id of 0 means that this is used for global settings.
            is_fixed (bool): False if adding and deleting rows is allowed.
        """
        self.data = data
        self.material_id = material_id

        self.dbl_validator = QxDoubleValidator()
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        delegates = {
            0: self.edit_delegate,
            1: self.edit_delegate,
            2: self.edit_delegate,
            3: self.edit_delegate,
            4: self.edit_delegate
        }

        self.rename_model = ConsolidationModel(
            ['Elapsed Time (sec)', 'Porosity', 'Critical Shear', 'Erosion Constant', 'Erosion Exponent']
        )

        super().__init__(
            parent,
            self.data.materials[self.material_id].consolidation,
            1,
            filter_model=self.rename_model,
            column_delegates=delegates,
            is_fixed=is_fixed
        )

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        row_idx = self.model.rowCount()
        new_index = self.model.index(row_idx - 1, 0)
        self.model.setData(new_index, self.data.info.attrs['next_time_id'])
        self.data.info.attrs['next_time_id'] += 1
