"""This module defines a table widget that can be used for defining output time ranges."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import typing

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, QModelIndex, Qt

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.widgets.adh_table_widget import AdhTableWidget


class OutputAutobuildModel(RenameModel):
    """A model to rename header titles, hide unwanted options."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        self._UNIT_COLUMN = 3
        self.unit_values = ['seconds', 'minutes', 'hours', 'days', 'weeks']
        super().__init__(column_names, parent)

    def data(self, index: QModelIndex, role=Qt.DisplayRole) -> typing.Any:
        """Gets the data for a table cell.

        Changes the unit column from an int to a string.

        Args:
            index (QModelIndex): The table cell index.
            role (int): The role of the data being asked for.

        Returns:
            The data that was asked for for the table cell.
        """
        source_index = self.sourceModel().index(index.row(), index.column())
        value = self.sourceModel().data(source_index, role)
        if index.column() == self._UNIT_COLUMN and role in [Qt.DisplayRole, Qt.EditRole]:
            if value not in self.unit_values:  # TODO: Text should NOT be here, but somehow it is. Fix it.
                value = self.unit_values[int(float(value))]
        return value

    def setData(self, index: QModelIndex, value: typing.Any, role=Qt.DisplayRole) -> bool:  # noqa: N802
        """Sets the data for a table cell.

        Changes the unit column from a string to an int for storage.

        Args:
            index (QModelIndex): The table cell index.
            value (typing.Any): The value to set for the table cell.
            role (int): The role of the data being set.

        Returns:
            True if the data was set.
        """
        if index.column() == self._UNIT_COLUMN and role in [Qt.DisplayRole, Qt.EditRole]:
            value = self.unit_values.index(value)
        return super(OutputAutobuildModel, self).setData(index, value, role)


class OutputAutobuildTableWidget(AdhTableWidget):
    """Output autobuild table widget class."""
    def __init__(self, parent, data):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data (TimeSeries): The time series of output frequencies.
        """
        self.data = data
        self.unit_col = 3

        self.rename_model = OutputAutobuildModel(['Start Time', 'End Time', 'Interval', 'Unit'])

        self.dbl_validator = QxDoubleValidator()
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.unit_delegate = QxCbxDelegate()
        self.unit_delegate.set_strings(self.rename_model.unit_values)
        delegates = {0: self.edit_delegate, 1: self.edit_delegate, 2: self.edit_delegate, 3: self.unit_delegate}

        super().__init__(
            parent,
            self.data.time_series,
            0,
            filter_model=self.rename_model,
            column_delegates=delegates,
            is_fixed=False
        )
        self.default_values = [0.0, 0.0, 0.0, 0]

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        self.model.setData(self.model.index(row_idx, self.unit_col), 0)
        new_index = self.model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
