"""This module defines a table widget that can be used for defining output times."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.widgets.adh_table_widget import AdhTableWidget


class OutputFrequencyModel(RenameModel):
    """A model to rename header titles, hide unwanted options."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        self._TIME_COLUMN = 0
        super().__init__(column_names, parent)

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Filters out all columns except the 'Output Time' column.

        Args:
            source_column (int): The column index in the source model.
            source_parent (QModelIndex): The parent index of the view.

        Returns:
            True if the model should keep the column, false if the column should be hidden.
        """
        return source_column == self._TIME_COLUMN


class OutputFrequencyTableWidget(AdhTableWidget):
    """Output frequency table widget class."""
    def __init__(self, parent, data):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data (TimeSeries): The time series of output frequencies.
        """
        self.data = data

        self.dbl_validator = QxDoubleValidator()
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        delegates = {0: self.edit_delegate}

        self.rename_model = OutputFrequencyModel(['Output Time'])

        super().__init__(
            parent,
            self.data.time_series,
            0,
            filter_model=self.rename_model,
            column_delegates=delegates,
            is_fixed=False
        )
