"""Class to snap a node string arc."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid as CoGrid
from xms.data_objects.parameters import Arc
from xms.snap.snap_exterior_arc import SnapExteriorArc
from xms.snap.snap_interior_arc import SnapInteriorArc

# 4. Local modules


class SnapNodeString(object):
    """Class to snap a node string arc."""
    def __init__(self):
        """Initialize an instance."""
        self._snap_exterior = SnapExteriorArc()
        self._snap_interior = SnapInteriorArc()

    def set_grid(self, grid: CoGrid, target_cells: bool = False):
        """
        Set the grid for snapping.

        Args:
            grid: The grid.
            target_cells: Whether to snap to grid cells or points.
        """
        self._snap_exterior.set_grid(grid, target_cells=target_cells)
        self._snap_interior.set_grid(grid, target_cells=target_cells)

    def get_snapped_points(self, locations: Arc | list[tuple[float, float, float]]):
        """
        Get the snapped points for an arc to the grid.

        Args:
            locations: Locations for which snapped points are to be retrieved.

        Returns:
            dict: A dictionary containing the snapped points for the given locations. The dictionary has two keys with
            parallel lists:
                - 'location' (list): A list of snapped points for the provided locations.
                - 'id' (object): A list of grid cell or point IDs.

        Note:
            The 'location' key in the returned dictionary will be empty if no snapped points are found.
        """
        result = self._snap_interior.get_snapped_points(locations)
        if len(result['location']) == 0:
            result = self._snap_exterior.get_snapped_points(locations)
        return result
