"""Python wrapping for xms.api._xmsapi.dmi.ActionRequest."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api._xmsapi.dmi import ActionRequest as CActionRequest, DialogModality

DialogModalityStrings = {
    'NO_DIALOG': DialogModality.NO_DIALOG,
    'MODAL': DialogModality.MODAL,
    'MODELESS': DialogModality.MODELESS,
}


class ActionRequest:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.ActionRequest objects."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            kwargs:
                main_file (str): Full path to the component's main file

                modality (str): See dialog_modality.setter for more details

                class_name (str): Component's Python class name

                module_name (str): Component's full module import name

                method_name (str): Name of the method to call

                comp_uuid (str): UUID of the component

                sim_uuid (str): UUID of the simulation for solution load

                parameters (dict): Dict of arguments to pass to the ActionRequest method. Keys need to be of type str
                or int, and all keys must be of the same type (i.e. all str keys or all int keys).
                instance (CActionRequest): The C++ ActionRequest object to wrap
        """
        self._instance = kwargs.get('instance')
        if not self._instance:
            self._instance = CActionRequest()

        if 'main_file' in kwargs:
            self._instance.SetMainFile(kwargs['main_file'])
        if 'modality' in kwargs:
            # Use property setter to validate
            self.dialog_modality = kwargs['modality']
        if 'class_name' in kwargs:
            self._instance.SetClass(kwargs['class_name'])
        if 'module_name' in kwargs:
            self._instance.SetModule(kwargs['module_name'])
        if 'method_name' in kwargs:
            self._instance.SetMethodAction(kwargs['method_name'])
        if 'comp_uuid' in kwargs:
            self._instance.SetComponentUuid(kwargs['comp_uuid'])
        if 'sim_uuid' in kwargs:
            self._instance.SetSimulationUuidForSolution(kwargs['sim_uuid'])
        if 'parameters' in kwargs:
            self._instance.SetActionParameterItems(kwargs['parameters'])

    @property
    def dialog_modality(self):
        """Returns the ActionRequest's dialog modality type as a str."""
        # Convert the C++ enum to it's associated Python string constant.
        c_enum = self._instance.GetDialogModality()
        return next(key for key, value in DialogModalityStrings.items() if value == c_enum)

    @dialog_modality.setter
    def dialog_modality(self, modality):
        """Sets the ActionRequest's dialog modality type.

        Args:
            modality (str): Modality type of the ActionRequest. One of the DialogModalityStrings keys.
        """
        # Convert the Python string constant to the associated C++ enum DialogModalityStrings keys.
        modality_enum = DialogModalityStrings.get(modality.upper())
        if modality_enum is None:
            raise ValueError(f'modality must be one of the following options: {DialogModalityStrings.keys()}')
        self._instance.SetDialogModality(modality_enum)

    @property
    def module_name(self):
        """Returns the name of the class that will be called with the ActionRequest."""
        return self._instance.GetModule()

    @module_name.setter
    def module_name(self, name):
        """Sets the name of the module that will be called with the ActionRequest.

        Args:
            name (str): Name of the ActionRequest module
        """
        self._instance.SetModule(name)

    @property
    def class_name(self):
        """Returns the name of the class that will be called with the ActionRequest."""
        return self._instance.GetClass()

    @class_name.setter
    def class_name(self, name):
        """Sets the name of the class that will be called with the ActionRequest.

        Args:
            name (str): Name of the ActionRequest class
        """
        self._instance.SetClass(name)

    @property
    def method_name(self):
        """Returns the name of the method that will be called with the ActionRequest."""
        return self._instance.GetMethodAction()

    @method_name.setter
    def method_name(self, name):
        """Sets the name of the method that will be called with the ActionRequest.

        Args:
            name (str): Name of the method to call
        """
        self._instance.SetMethodAction(name)

    @property
    def action_parameters(self):
        """Returns the argument dict that will be passed to the ActionRequest method."""
        return self._instance.GetActionParameterItems()

    @action_parameters.setter
    def action_parameters(self, parameters):
        """Sets the argument dict that will be passed to the ActionRequest method.

        Args:
            parameters (dict): Dict of arguments to pass to the ActionRequest method. Keys need to be of type str
            or int, and all keys must be of the same type (i.e. all str keys or all int keys).
        """
        self._instance.SetActionParameterItems(parameters)

    @property
    def main_file(self):
        """Returns the ActionRequest component's main file."""
        return self._instance.GetMainFile()

    @main_file.setter
    def main_file(self, comp_main_file):
        """Sets the ActionRequest component's main file.

        Args:
            comp_main_file (str): main file of the component
        """
        self._instance.SetMainFile(comp_main_file)

    @property
    def component_uuid(self):
        """Returns the ActionRequest component's main file."""
        return self._instance.GetComponentUuid()

    @component_uuid.setter
    def component_uuid(self, comp_uuid):
        """Sets the ActionRequest component's UUID.

        Args:
            comp_uuid (str): UUID of the component
        """
        self._instance.SetComponentUuid(comp_uuid)

    @property
    def simulation_uuid_for_solution(self):
        """Returns the UUID of the simulation the ActionRequest loads solutions for."""
        return self._instance.GetSimulationUuidForSolution()

    @simulation_uuid_for_solution.setter
    def simulation_uuid_for_solution(self, sim_uuid):
        """Sets the UUID of the simulation the ActionRequest loads solutions for.

        Note that this is only applicable if this is a solution load ActionRequest. Allows XMS to organize solution
        datasets into simulation folders under geometries.

        Args:
            sim_uuid (str): UUID of the simulation
        """
        self._instance.SetSimulationUuidForSolution(sim_uuid)
