"""Dialog for defining a 3D Bridge."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
from PySide2.QtCore import QSize
from PySide2.QtWidgets import (
    QCheckBox, QDialog, QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QMessageBox, QPushButton,
    QSizePolicy, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.constraint.ugrid_builder import UGridBuilder
from xms.data_objects.parameters import UGrid as DoUGrid
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.bridge.grid_builder import GridBuilder


class BridgeDialog(XmsDlg):
    """Dialog to display and edit an XY Series.

    From https://doc-snapshots.qt.io/qtforpython/tutorials/datavisualize/plot_datapoints.html

    """
    def __init__(self, input, parent=None):
        """Initializes the class.

        Args:
            input (:obj:`dict`): inputs to the dialog
            parent (:obj:`Qt Object`): parent window for the dialog
        """
        super().__init__(parent, 'xmsbridge.bridge_dialog')

        self.bridge = input['bridge']
        self.arc_up = input['arc_up']
        self.arc_down = input['arc_down']
        self.wkt = input['wkt']
        self.main_file = input['main_file']
        self.cover = input['cover']
        self.output_ugrid = None

        self.txt_uparc_id = None
        self.txt_downarc_id = None
        self.error_widget = None
        self.edt_manning = None
        self.tog_add_ugrid = None
        self.grp_down = None
        self.tog_downstream = None
        self.main_layout = None

        self.models = {}
        self.mp_figure = {}
        self.mp_ax = {}
        self.mp_canvas = {}

        self.btn_box = None
        self.export_filename = ''

        self.setup_ui()
        self.load()

    def setup_ui(self):
        """Sets up the UI."""
        self.setWindowTitle('Bridge')

        self.main_layout = QVBoxLayout()
        main_layout = self.main_layout

        # Error
        self.error_widget = QWidget()
        hlayout_error = QHBoxLayout()
        self.error_widget.setLayout(hlayout_error)
        txt_error = QLabel('Bridge requires a coverage with two arcs.')
        hlayout_error.addWidget(txt_error)
        main_layout.addWidget(self.error_widget)

        hlayout_swap = QHBoxLayout()
        # Swap button
        btn_swap = QPushButton('Swap Arcs')
        size_policy = QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        btn_swap.setSizePolicy(size_policy)
        hlayout_swap.addWidget(btn_swap)
        # upstream arc id
        txt_uparc = QLabel('Upstream arc id:')
        hlayout_swap.addWidget(txt_uparc)
        self.txt_uparc_id = QLabel('0')
        hlayout_swap.addWidget(self.txt_uparc_id)
        # downstream arc id
        txt_downarc = QLabel('Downstream arc id:')
        hlayout_swap.addWidget(txt_downarc)
        self.txt_downarc_id = QLabel('0')
        hlayout_swap.addWidget(self.txt_downarc_id)
        hlayout_swap.addStretch()
        main_layout.addLayout(hlayout_swap)

        # manning's n
        txt_manning = QLabel("Bridge ceiling manning roughness coefficient:")
        main_layout.addWidget(txt_manning)
        self.edt_manning = QLineEdit(f'{self.bridge.manning_n}')
        pos_validator = QxDoubleValidator(parent=self)
        pos_validator.setBottom(0.0)
        self.edt_manning.setValidator(pos_validator)
        main_layout.addWidget(self.edt_manning)

        # add UGrid
        self.tog_add_ugrid = QCheckBox('Add 3D Bridge UGrid to SMS on OK')
        main_layout.addWidget(self.tog_add_ugrid)

        # Top chart
        grp_top = QGroupBox('Top profile')
        hlayout_top = QHBoxLayout()
        self.add_mp_figure('top', hlayout_top)

        btn_edit_top = QPushButton('Edit...')
        hlayout_top.addWidget(btn_edit_top)
        grp_top.setLayout(hlayout_top)
        main_layout.addWidget(grp_top)

        # Upstream chart
        grp_up = QGroupBox('Upstream profile')
        hlayout_up = QHBoxLayout()
        self.add_mp_figure('up', hlayout_up)

        btn_edit_up = QPushButton('Edit...')
        hlayout_up.addWidget(btn_edit_up)
        grp_up.setLayout(hlayout_up)
        main_layout.addWidget(grp_up)

        # toggle for Downstream chart
        self.tog_downstream = QCheckBox('Specify downstream profile')
        self.tog_downstream.setChecked(self.bridge.specify_downstream_profile)
        main_layout.addWidget(self.tog_downstream)
        self.tog_downstream.stateChanged.connect(self._on_specify_downstream)

        # Downstream chart
        self.grp_down = QGroupBox('Downstream profile')
        hlayout_down = QHBoxLayout()
        self.add_mp_figure('down', hlayout_down)

        btn_edit_down = QPushButton('Edit...')
        hlayout_down.addWidget(btn_edit_down)
        self.grp_down.setLayout(hlayout_down)
        main_layout.addWidget(self.grp_down)

        # Button box
        self.btn_box = QDialogButtonBox()
        btns = QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        self.btn_box.setStandardButtons(btns)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        main_layout.addWidget(self.btn_box)

        self.setLayout(main_layout)

        # Signals
        btn_swap.clicked.connect(self.on_btn_swap)
        btn_edit_top.clicked.connect(self.on_btn_edit_top)
        btn_edit_up.clicked.connect(self.on_btn_edit_up)
        btn_edit_down.clicked.connect(self.on_btn_edit_down)

    def add_mp_figure(self, name, layout):
        """Adds a matplotlib figure to the layout.

        Args:
             name (:obj:`str`): name of the figure
             layout (:obj:`QLayout`): qt widget
        """
        self.mp_figure[name] = Figure()
        # self.mp_figure[name].set_tight_layout(True)
        self.mp_figure[name].set_layout_engine(layout='tight')
        self.mp_canvas[name] = FigureCanvas(self.mp_figure[name])
        self.mp_canvas[name].setMinimumWidth(100)
        self.mp_canvas[name].setMinimumHeight(100)
        self.mp_canvas[name].setSizePolicy(QSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.MinimumExpanding))
        self.mp_ax[name] = self.mp_figure[name].add_subplot(111)
        self.mp_ax[name].grid(True)
        layout.addWidget(self.mp_canvas[name])

    def add_line_series(self, name):
        """Add line plot.

        Args:
            name (:obj:`str`): name of the series added
        """
        self.mp_ax[name].clear()
        self.mp_ax[name].grid(True)
        x_column = []
        y_column = []
        if self.models[name].rowCount() > 0:
            # Add data to plot
            x_column = self.models[name].data_frame.iloc[:, 0]
            y_column = self.models[name].data_frame.iloc[:, 1]

        self.mp_ax[name].plot(x_column, y_column)
        self.mp_canvas[name].draw()

    def load(self):
        """Loads the bridge data into the dialog widgets."""
        if self.bridge.arc_id_upstream > 0:
            self.txt_uparc_id.setText(str(self.bridge.arc_id_upstream))
        else:
            self.txt_uparc_id.setText('none')

        if self.bridge.arc_id_downstream > 0:
            self.txt_downarc_id.setText(str(self.bridge.arc_id_downstream))
        else:
            self.txt_downarc_id.setText('none')

        if not self.arc_up or not self.arc_down:
            self.error_widget.setVisible(True)
        else:
            self.error_widget.setVisible(False)

        self.models['top'] = QxPandasTableModel(self.bridge.df_top)
        self.models['up'] = QxPandasTableModel(self.bridge.df_upstream)
        self.models['down'] = QxPandasTableModel(self.bridge.df_downstream)
        self.add_line_series('top')
        self.add_line_series('up')
        self.add_line_series('down')

        self.grp_down.setVisible(self.tog_downstream.isChecked())

    def sizeHint(self):  # noqa: N802
        """Overridden method to help the dialog have a good minimum size.

        Returns:
            (:obj:`QSize`): Size to use for the initial dialog size.
        """
        return QSize(500, 700)

    def on_btn_edit(self, name, index, data_frame):
        """Opens the XySeriesEditor.

        Args:
            name (:obj:`str`): Name of the xy series.
            index (:obj:`str`): The index into the dictionaries.
            data_frame (:obj:`pandas.DataFrame`): The DataFrame.

        Returns:
            (:obj:`tuple`): tuple containing:

                rv (:obj:`bool`): True on OK, else False.

                modified (:obj:`pandas.DataFrame`): The modified dataframe on OK, else None.
        """
        xy_series_editor = XySeriesEditor(data_frame=data_frame, series_name=name, parent=self)
        if xy_series_editor.exec() == QDialog.Accepted:
            df = xy_series_editor.model.data_frame
            df = df.sort_values(df.columns[0])
            self.models[index] = QxPandasTableModel(df)
            self.add_line_series(index)
            return True, df
        return False, None

    def on_btn_swap(self):
        """Swaps the upstream and downstream arc IDs."""
        if not self.arc_up or not self.arc_down:
            QMessageBox.warning(
                self, '3D Bridge', 'Can’t swap. Bridge requires a coverage with two arcs.', QMessageBox.Ok
            )
            return

        self.bridge.arc_id_upstream = int(self.txt_downarc_id.text())
        self.bridge.arc_id_downstream = int(self.txt_uparc_id.text())
        self.txt_uparc_id.setText(str(self.bridge.arc_id_upstream))
        self.txt_downarc_id.setText(str(self.bridge.arc_id_downstream))

    def check_data_for_export(self, show_message=True):
        """Checks if data is valid for exporting an xmugrid file.

        Args:
            show_message (:obj:`bool`): If True, display a message box warning user if data is invalid for export

        Returns:
            rv (:obj:`bool`): True if data is OK for export, else False.
        """
        # Warn and abort if missing arcs
        if not self.arc_up or not self.arc_down:
            if show_message:
                msg = 'Unable to generate UGrid. Bridge requires a coverage with two arcs.'
                QMessageBox.warning(self, '3D Bridge', msg, QMessageBox.Ok)
            return False

        # Warn and abort if missing profile lines
        if self.bridge.df_top.shape[0] < 2 or self.bridge.df_upstream.shape[0] < 2 or \
                self.bridge.df_downstream.shape[0] < 2:
            if show_message:
                msg = 'Unable to generate UGrid. Profile lines are missing.'
                QMessageBox.warning(self, '3D Bridge', msg, QMessageBox.Ok)
            return False
        return True

    def on_btn_edit_top(self):
        """Opens the XY Series Editor to display the bridge top profile."""
        rv, df = self.on_btn_edit('Top', 'top', self.bridge.df_top)
        if rv:
            self.bridge.df_top = df

    def on_btn_edit_up(self):
        """Opens the XY Series Editor to display the bridge upstream profile."""
        rv, df = self.on_btn_edit('Upstream', 'up', self.bridge.df_upstream)
        if rv:
            self.bridge.df_upstream = df

    def on_btn_edit_down(self):
        """Opens the XY Series Editor to display the bridge downstream profile."""
        rv, df = self.on_btn_edit('Downstream', 'down', self.bridge.df_downstream)
        if rv:
            self.bridge.df_downstream = df

    def _on_specify_downstream(self):
        """Called when the specify downstream profile check box is clicked."""
        self.grp_down.setVisible(self.tog_downstream.isChecked())

    def accept(self):
        """Build the UGrid representation of the 3D Bridge on dialog accepeted."""
        if not self.tog_downstream.isChecked():
            self.bridge.df_downstream = self.bridge.df_upstream.copy()
        if self.check_data_for_export(False):
            grid_builder = GridBuilder()
            grid_builder.build_top_and_bottom(self.bridge, self.arc_up, self.arc_down, self.wkt, self.main_file)
            if self.tog_add_ugrid.isChecked():
                ug_file = os.path.join(os.path.dirname(self.main_file), 'ugrid.xmc')
                ug = grid_builder.build(self.bridge, self.arc_up, self.arc_down, self.wkt, ug_file)
                if ug is not None:
                    co_builder = UGridBuilder()
                    co_builder.set_ugrid(ug)
                    co_grid = co_builder.build_grid()
                    co_grid.write_to_file(ug_file)
                    self.output_ugrid = DoUGrid(
                        cogrid_file=ug_file, name=self.cover.name, projection=self.cover.projection
                    )

        self.bridge.manning_n = float(self.edt_manning.text())
        self.bridge.specify_downstream_profile = self.tog_downstream.isChecked()
        super().accept()
