"""Uses CoverageMapper to map data to CMS-Flow quadtree."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.cmsflow.components.bc_component import BCComponent
from xms.cmsflow.components.rm_structures_component import RMStructuresComponent
from xms.cmsflow.components.save_points_component import SavePointsComponent
from xms.cmsflow.mapping.coverage_mapper import CoverageMapper


class CoverageMapperRunner(QThread):
    """Class for mapping coverages to a quadtree for CMS-Flow."""
    processing_finished = Signal()

    def __init__(self, query):
        """Constructor.

        Args:
            query (Query): The object for getting information from XMS.
        """
        super().__init__()
        self._existing_mapped_component_uuids = []
        self._query = query
        self._logger = logging.getLogger('xms.cmsflow')
        self._sim_node = None
        self.mapper = None

    @property
    def sim_node(self):
        """Make sure we have the simulation tree node."""
        if not self._sim_node:
            sim_uuid = self._query.parent_item_uuid()
            self._sim_node = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
        return self._sim_node

    def run(self):
        """Creates the snap preview of coverages onto the quadtree."""
        try:
            self.mapper = CoverageMapper(generate_snap=True)
            self._get_domain(self.mapper)
            self._get_activity(self.mapper)
            self._get_bc_coverage(self.mapper)
            self._get_save_points_coverage(self.mapper)
            self._get_rm_coverage(self.mapper)
            if not self.mapper.check_mappable_items():
                self._logger.error(
                    'No items to generate snap preview from.  Please link an Activity Classification, '
                    'Boundary Conditions, or Rubble Mound Jetties coverage to the simulation.'
                )
                raise RuntimeError()
            self._get_uuids_of_existing_mapped_components()
            self.mapper.do_map()
        except:  # noqa
            self._logger.exception('Error generating snap.')
        finally:
            self.processing_finished.emit()

    def _get_domain(self, worker):
        """Gets the quadtree geometry of the domain.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        # get the Quadtree geometry.
        quad_item = tree_util.descendants_of_type(
            self.sim_node,
            xms_types=['TI_UGRID_PTR', 'TI_CGRID2D_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True
        )
        if not quad_item:
            raise RuntimeError(
                'Unable to find CMS-Flow domain grid. Link a quadtree to the simulation to generate a snap preview.'
            )
        quad_geom = self._query.item_with_uuid(quad_item.uuid)
        if quad_geom:
            co_grid = read_grid_from_file(quad_geom.cogrid_file)
            wkt = quad_geom.projection.well_known_text
        else:
            raise RuntimeError(
                'Unable to find CMS-Flow domain grid. Link a quadtree to the simulation to generate a snap preview.'
            )
        worker.set_quadtree(co_grid, wkt)

    def _get_activity(self, worker):
        """Gets the activity coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        # get the activity coverage.
        activity_item = tree_util.descendants_of_type(
            self.sim_node,
            xms_types=['TI_COVER_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True,
            coverage_type='ACTIVITY_CLASSIFICATION'
        )
        activity_cov = None
        if activity_item:
            activity_cov = self._query.item_with_uuid(activity_item.uuid, generic_coverage=True)
        worker.set_activity(activity_cov)
        return True

    def _get_bc_coverage(self, worker):
        """Gets the boundary condition coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        bc_item = tree_util.descendants_of_type(
            self.sim_node,
            xms_types=['TI_COVER_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True,
            coverage_type='Boundary Conditions'
        )
        if bc_item:
            bc_cov = self._query.item_with_uuid(bc_item.uuid)
            if bc_cov:
                bc_do_comp = self._query.item_with_uuid(bc_item.uuid, model_name='CMS-Flow', unique_name='BC_Component')
                bc_py_comp = BCComponent(bc_do_comp.main_file)
                if bc_py_comp.cov_uuid == '':
                    bc_py_comp.cov_uuid = bc_cov.uuid
                self._query.load_component_ids(bc_py_comp, arcs=True)
                worker.set_boundary_conditions(bc_cov, bc_py_comp)

    def _get_save_points_coverage(self, worker):
        """Gets the Save Points coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        save_pts_item = tree_util.descendants_of_type(
            self.sim_node,
            xms_types=['TI_COVER_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True,
            coverage_type='Save Points'
        )
        if save_pts_item:
            save_pts_cov = self._query.item_with_uuid(save_pts_item.uuid)
            if save_pts_cov:
                do_comp = self._query.item_with_uuid(
                    save_pts_item.uuid, model_name='CMS-Flow', unique_name='Save_Points_Component'
                )
                py_comp = SavePointsComponent(do_comp.main_file)
                if py_comp.cov_uuid == '':
                    py_comp.cov_uuid = save_pts_cov.uuid
                self._query.load_component_ids(py_comp, points=True)
                worker.set_save_points(save_pts_cov, py_comp)

    def _get_rm_coverage(self, worker):
        """Gets the rubble mound jetties coverage.

        Args:
            worker (CoverageMapper): The worker to do the snapping of coverages to the domain.
        """
        rm_item = tree_util.descendants_of_type(
            self.sim_node,
            xms_types=['TI_COVER_PTR'],
            recurse=False,
            allow_pointers=True,
            only_first=True,
            coverage_type='Rubble Mound Jetties'
        )
        if rm_item:
            self._rm_cov = self._query.item_with_uuid(rm_item.uuid)
            if self._rm_cov:
                rm_do_comp = self._query.item_with_uuid(
                    rm_item.uuid, model_name='CMS-Flow', unique_name='RM_Structures_Component'
                )
                rm_py_comp = RMStructuresComponent(rm_do_comp.main_file)
                if rm_py_comp.cov_uuid == '' and self._rm_cov:
                    rm_py_comp.cov_uuid = self._rm_cov.uuid
                self._query.load_component_ids(rm_py_comp, polygons=True)
                worker.set_rubble_mound(self._rm_cov, rm_py_comp)

    def send(self):
        """Add mapped components to the XMS project."""
        self._logger.info('Adding mapped display items.')
        # delete any existing mapped components
        for str_uuid in self._existing_mapped_component_uuids:
            self._query.delete_item(str_uuid)

        for mapped_comp in self.mapper.mapped_comps:  # Will be None if we logged an error during the mapping operation.
            self._query.add_component(mapped_comp[0], actions=mapped_comp[1])
            self._query.link_item(self.sim_node.uuid, mapped_comp[0].uuid)

    def _get_uuids_of_existing_mapped_components(self):
        """Gets the uuids of any existing mapped components."""
        comp_items = tree_util.descendants_of_type(self.sim_node, xms_types=['TI_COMPONENT'], recurse=False)
        self._existing_mapped_component_uuids = [comp_child.uuid for comp_child in comp_items]
