"""Module for the get_model function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['get_model', 'parameter_to_file', 'parameter_from_file']

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel, Parameter
from xms.guipy.widgets.table_with_tool_bar import IntColumnType, TableDefinition

# 4. Local modules


def get_model() -> GenericModel:
    """Create the structures data definition."""
    gm = GenericModel(exclusive_point_conditions=True, exclusive_arc_conditions=True, exclusive_polygon_conditions=True)

    pp = gm.polygon_parameters
    # Rubble mound properties for polygons
    gp = pp.add_group('rubble_mound', 'Rubble Mound', is_active=False)
    gp.add_text(name='name', label='Name')

    opts = ['Constant', 'Dataset']
    rock_diameter_type = gp.add_option(
        name='rock_diameter_type', label='Rock diameter type', default=opts[0], options=opts
    )
    p = gp.add_float(name='rock_diameter', label='Rock diameter', default=0.0)
    flags = {opts[0]: True, opts[1]: False}
    p.add_dependency(parent=rock_diameter_type, flags=flags)
    p = gp.add_dataset('rock_diameter_dataset', label='Rock diameter dataset')
    flags = {opts[0]: False, opts[1]: True}
    p.add_dependency(parent=rock_diameter_type, flags=flags)

    porosity_type = gp.add_option(name='porosity_type', label='Porosity type', default=opts[0], options=opts)
    p = gp.add_float(name='porosity', label='Porosity', default=0.0)
    flags = {opts[0]: True, opts[1]: False}
    p.add_dependency(parent=porosity_type, flags=flags)
    p = gp.add_dataset('porosity_dataset', label='Porosity dataset')
    flags = {opts[0]: False, opts[1]: True}
    p.add_dependency(parent=porosity_type, flags=flags)

    base_depth_type = gp.add_option(name='base_depth_type', label='Base depth type', default=opts[0], options=opts)
    p = gp.add_float(name='base_depth', label='Base depth', default=0.0)
    flags = {opts[0]: True, opts[1]: False}
    p.add_dependency(parent=base_depth_type, flags=flags)
    p = gp.add_dataset('base_depth_dataset', label='Base depth dataset')
    flags = {opts[0]: False, opts[1]: True}
    p.add_dependency(parent=base_depth_type, flags=flags)

    calc_opts = ['Sidiropoulou et al. (2007)', 'Kadlec and Knight (1996)', 'Ward (1964)']
    gp.add_option(name='calc_method', label='Calculation method', default=calc_opts[0], options=calc_opts)

    ap = gm.arc_parameters
    # Culvert properties for arcs
    gp = ap.add_group(group_name='culvert', label='Culvert')
    opts = ['Circular', 'Box']
    culvert_type = gp.add_option(name='culvert_type', label='Type of culvert', default=opts[0], options=opts)
    gp.add_boolean(name='flap_gate', label='With flap gate', default=False)
    p = gp.add_float(name='radius', label='Radius', default=1.0)
    flags = {opts[0]: True, opts[1]: False}
    p.add_dependency(culvert_type, flags)
    p = gp.add_float(name='width', label='Width', default=1.0)
    flags = {opts[0]: False, opts[1]: True}
    p.add_dependency(culvert_type, flags)
    p = gp.add_float(name='height', label='Height', default=1.0)
    p.add_dependency(culvert_type, flags)
    gp.add_float(name='length', label='Length', default=1.0)
    label = 'Darcy-Weisbach friction coefficient for the culvert fully occupied by flow'
    gp.add_float(name='darcy_friction', label=label, default=0.025)
    label = 'Manning friction coefficient for the culvert partially occupied by flow'
    gp.add_float(name='manning_n', label=label, default=0.025)
    gp.add_float(name='entry_head_loss_bay', label='Bay side entrance head loss', default=0.0)
    gp.add_float(name='exit_head_loss_bay', label='Bay side exit head loss', default=0.0)
    gp.add_float(name='entry_head_loss_sea', label='Sea side entrance head loss', default=0.0)
    gp.add_float(name='exit_head_loss_sea', label='Sea side exit head loss', default=0.0)

    # Tide gate properties for arcs
    gp = ap.add_group(group_name='tide_gate', label='Tide Gate')
    gp.add_float(name='distribution_coefficient', label='Lateral distribution coefficient', default=0.0)
    opts = ['North', 'East', 'South', 'West']
    gp.add_option(
        name='orientation_sea', label='Orientation of gate (direction of sea side)', default=opts[0], options=opts
    )
    gp.add_float(name='flow_coeff_from_bay', label='Flow coefficient - bayside to seaside', default=0.0)
    gp.add_float(name='flow_coeff_from_sea', label='Flow coefficient - seaside to bayside', default=0.0)
    gp.add_float(name='opening_height', label='Opening height of tide gate', default=0.0)
    gp.add_float(name='bottom_elevation', label='Bottom elevation of tide gate (mean water level)', default=0.0)
    opts = ['Approach 1', 'Approach 2']
    gp.add_option(name='method', label='Method to calculate flux through tide gate', default=opts[0], options=opts)

    options = [
        'Uncontrolled',
        'Regular time interval',
        'Designated times and durations',
        'Open for ebb tide, close for flood',
    ]
    schedule_type = gp.add_option('schedule_type', 'Schedule operation type', options[1], options)

    scheduling_table_columns = [
        IntColumnType(header='Start time (hour)', default=0),
        IntColumnType(header='Open duration (hour)', default=0)
    ]
    scheduling_table_definition = TableDefinition(scheduling_table_columns)
    default = [[0, 0]]
    designated_times_table = gp.add_table('designated_times', 'Schedule', default, scheduling_table_definition)
    flags = {option: False for option in options}
    flags['Designated times and durations'] = True
    designated_times_table.add_dependency(schedule_type, flags)

    flags = {option: False for option in options}
    flags['Regular time interval'] = True
    start_time = gp.add_integer('start_time', 'Start time (hour)', default=0, low=0)
    start_time.add_dependency(schedule_type, flags)
    open_frequency = gp.add_integer('open_frequency', 'Opening frequency (hour)', default=1, low=0)
    open_frequency.add_dependency(schedule_type, flags)
    open_duration = gp.add_integer('open_duration', 'Open duration(hour)', default=1, low=0)
    open_duration.add_dependency(schedule_type, flags)

    # Weir properties for arcs
    gp = ap.add_group(group_name='weir', label='Weir')
    gp.add_float(name='distribution_coeff', label='Lateral distribution coefficient', default=0.0)
    opts = ['North', 'East', 'South', 'West']
    gp.add_option(
        name='orientation', label='Orientation of weir (direction of sea side)', default=opts[0], options=opts
    )
    opts = ['Sharp-crested', 'Broad-crested']
    gp.add_option(name='weir_type', label='Type of weir', default=opts[0], options=opts)
    gp.add_float(name='flow_coeff_bay_to_sea', label='Flow coefficient - bayside to seaside', default=0.0)
    gp.add_float(name='flow_coeff_sea_to_bay', label='Flow coefficient - seaside to bayside', default=0.0)
    gp.add_float(name='crest_elevation', label='Crest elevation (mean water level)', default=0.0)
    opts = ['Approach 1', 'Approach 2']
    gp.add_option(name='calc_method', label='Method to calculate flux over the weir', default=opts[0], options=opts)

    return gm


def parameter_to_file(parameter: Parameter) -> str:
    """
    Convert a value from its model representation to its file representation.

    E.g. the file might use three-letter codes in all caps, while the parameter displays long-form human-friendly text.

    Args:
         parameter: The parameter to convert.

    Returns:
        The parameter's value, as it should be represented in the file.
    """
    mapping = {}

    if parameter.group_name == 'tide_gate':
        if parameter.parameter_name == 'orientation_sea':
            mapping = {'North': 'NORTH', 'East': 'EAST', 'South': 'SOUTH', 'West': 'WEST'}
        elif parameter.parameter_name == 'schedule_type':
            mapping = {
                'Uncontrolled': 'UNCONTROLLED',
                'Regular time interval': 'REGULAR',
                'Designated times and durations': 'DESIGNATED',
                'Open for ebb tide, close for flood': 'EBB_TIDE',
            }
        elif parameter.parameter_name == 'method':
            mapping = {'Approach 1': '1', 'Approach 2': '2'}

    if parameter.value not in mapping:
        raise AssertionError(f'Unable to format {parameter.group_name}/{parameter.parameter_name} for file.')

    return mapping[parameter.value]


def parameter_from_file(line_number: int, value: str, parameter: Parameter):
    """
    Convert a value from its file representation to its model representation.

    E.g. the file might use three-letter codes in all caps, while the parameter displays long-form human-friendly text.

    Args:
         line_number: Line in the file being read. Used for error reporting.
         value: The parameter's value, as formatted in the file.
         parameter: Where to store the result.
    """
    mapping = {}
    if parameter.group_name == 'culvert':
        if parameter.parameter_name == 'culvert_type':
            mapping = {'CIRCLE': 'Circular', 'BOX': 'Box'}
        elif parameter.parameter_name == 'flap_gate':
            mapping = {'ON': True, 'OFF': False}
        elif parameter.parameter_name == 'crest_type':
            mapping = {'SHARP': 'Sharp-crested', 'BROAD': 'Broad-crested'}
        elif parameter.parameter_name == 'calc_method':
            mapping = {'1': 'Approach 1', '2': 'Approach 2'}
    elif parameter.group_name == 'weir':
        if parameter.parameter_name == 'orientation':
            mapping = {'NORTH': 'North', 'SOUTH': 'South', 'EAST': 'East', 'WEST': 'West'}
        elif parameter.parameter_name == 'weir_type':
            mapping = {'SHARP': 'Sharp-crested', 'BROAD': 'Broad-crested'}
        elif parameter.parameter_name == 'calc_method':
            mapping = {'1': 'Approach 1', '2': 'Approach 2'}

    if value not in mapping:
        raise RuntimeError(f'Line {line_number}: Invalid value for card')

    parameter.value = mapping[value]
