"""This module creates a thread for work to be done on and a progress dialog."""

# 1. Standard Python modules
import os
import shlex
import sys

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.data_objects.parameters import Dataset
from xms.datasets.dataset_io import update_geom_uuid
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.cmsflow.feedback.xmlog import XmLog


class ReadSolutionWorkerThread(ProcessFeedbackThread):
    """Worker thread for reading CMS-Flow solution."""
    def __init__(self, query, file_location, proj_name, parent, grid_uuid):
        """Construct the worker.

        Args:
            query (:obj:`data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (str): The directory of the solution to load.
            proj_name (str): The project name.
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
            grid_uuid (Optional[str]): UUID of the linked geometry if being called from the sim runner (Query.send()
                will not be called).
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.query = query
        self.file_location = file_location
        self.proj_name = proj_name
        self.geom_uuid = grid_uuid

    def _do_work(self):
        """Thread runner that reads CMS-Flow solution."""
        try:
            self.read_solution_file()
        except Exception:
            XmLog().instance.exception('Error! Could not read CMS-Flow solution.')

    def send(self):
        """Send imported datasets to SMS."""
        if self.query and not self.geom_uuid:  # Only send if user manually opened a solution file.
            self.query.send()

    def read_solution_file(self):
        """Reads the CMS-Flow Solution.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        sol_file_name = os.path.join(self.file_location, f'{self.proj_name}_sol.h5')
        wse_file_name = os.path.join(self.file_location, f'{self.proj_name}_wse.h5')
        vel_file_name = os.path.join(self.file_location, f'{self.proj_name}_vel.h5')
        visc_file_name = os.path.join(self.file_location, f'{self.proj_name}_visc.h5')
        wave_file_name = os.path.join(self.file_location, f'{self.proj_name}_wave.h5')
        morph_file_name = os.path.join(self.file_location, f'{self.proj_name}_morph.h5')
        trans_file_name = os.path.join(self.file_location, f'{self.proj_name}_trans.h5')
        wind_file_name = os.path.join(self.file_location, f'{self.proj_name}_met.h5')

        card_file_name = os.path.join(self.file_location, f'{self.proj_name}.cmcards')
        dataset_files = [wse_file_name, vel_file_name]
        label = ''
        is_single_file = False

        if not os.path.isfile(card_file_name):
            XmLog().instance.error('Could not find *.cmcards file.')
            return [], []
        with open(card_file_name, 'r') as file:
            card_lines = file.readlines()
            for card_line in card_lines:
                card_line = card_line.strip()
                if not card_line or card_line.startswith('!') or card_line.startswith('*'):
                    continue  # Skip blank lines, comment lines, and old SMS cards
                line_data = card_line.split()
                if line_data:
                    line_data[0].upper()  # only want the card name uppercase, nothing else!
                    if line_data[0] == 'SIMULATION_LABEL':
                        shlex_split = shlex.split(card_line, posix="win" not in sys.platform)
                        label = shlex_split[1].strip('"')
                    elif line_data[0] == 'OUTPUT_FILE_TYPE':
                        if line_data[1] == 'ASCII':
                            XmLog().instance.warning('Output is in ASCII format, and will not be read.')
                            return [], []
                    elif line_data[0] == 'USE_COMMON_SOLUTION_FILE':
                        if line_data[1] == 'ON':
                            is_single_file = True
                    elif len(line_data) == 2:
                        if line_data[0].endswith('_OUT_TIMES_LIST'):
                            try:
                                int(line_data[1])
                                if line_data[0] == 'MORPH_OUT_TIMES_LIST':
                                    dataset_files.append(morph_file_name)
                                elif line_data[0] == 'TRANS_OUT_TIMES_LIST':
                                    dataset_files.append(trans_file_name)
                                elif line_data[0] == 'WAVE_OUT_TIMES_LIST':
                                    dataset_files.append(wave_file_name)
                                elif line_data[0] == 'WIND_OUT_TIMES_LIST':
                                    dataset_files.append(wind_file_name)
                                elif line_data[0] == 'VISC_OUT_TIMES_LIST':
                                    dataset_files.append(visc_file_name)
                            except ValueError:
                                pass

        if is_single_file:
            dataset_files = [sol_file_name]
        for ds_file_name in dataset_files:
            try:
                if self.geom_uuid:
                    update_geom_uuid(ds_file_name, self.geom_uuid, label)
                with h5py.File(ds_file_name, 'r') as file:
                    for ds in file[f'{label}'].keys():
                        dset_group = f'{label}/{ds}'
                        if isinstance(file[dset_group], h5py.Dataset):
                            continue  # Only add child groups as datasets, not the geom UUID H5 dataset.
                        self.query.add_dataset(Dataset(ds_file_name, dset_group, 'CELL', 'NULL'))
            except Exception:
                XmLog().instance.error(f'Could not read solution from {ds_file_name} file.')
        return [], []


def load_solution_file(query, win_cont, file_location, proj_name, grid_uuid):
    """Reads the CMS-Flow Solution.

    Args:
        query (:obj:`data_objects.parameters.Query`): a Query object to communicate with SMS.
        win_cont (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        file_location (str): The directory of the solution to load.
        proj_name (str): The project name.
        grid_uuid (Optional[str]): UUID of the linked geometry if being called from the sim runner (Query.send() will
            not be called).

    Returns:
        (:obj:`tuple`): tuple containing:
            - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
              tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
              text.
            - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
    """
    # Don't want debug info about Python invocation in feedback dialog
    XmLog().instance

    worker = ReadSolutionWorkerThread(query, file_location, proj_name, win_cont, grid_uuid)
    display_text = {
        'title': 'Reading CMS-Flow Solution',
        'working_prompt': 'Reading CMS-Flow solution. Please wait...',
        'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
        'success_prompt': 'Successfully read CMS-Flow solution.',
        'note': '',
        'auto_load': 'Auto load solution into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.cmsflow', worker, win_cont)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.send()  # Send data back to XMS
    if win_cont is not None:
        win_gui.raise_main_xms_window(win_cont)  # Bring top-level Python window to foreground
    return [], []
