"""This is a class for logging messages."""

# 1. Standard Python modules
import logging
import os
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class XmLog:
    """Class for logging messages in XMS Python scripts."""
    top_module = 'xms.cmsflow'

    class __XmLog:
        MAX_LOG_SIZE = 1e+6  # In bytes (1 MB), log files larger than this will be wiped.

        def __init__(self):
            """Construct the singleton."""
            self._logger = logging.getLogger(XmLog.top_module)
            self._logger.setLevel(logging.DEBUG)
            log_file = self._setup_log_file()
            if log_file:
                fh = logging.FileHandler(log_file)
                fh.setLevel(logging.DEBUG)
                formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
                fh.setFormatter(formatter)
                self._logger.addHandler(fh)

        def _setup_log_file(self):
            """Make sure the log file directory exists and any existing log file is reasonable in size."""
            try:
                log_file = self._get_log_filename()
                if os.path.isfile(log_file) and os.path.getsize(log_file) > self.MAX_LOG_SIZE:
                    os.remove(log_file)
            except Exception:
                return None
            return log_file

        def _get_log_filename(self):
            """Get the log filename."""
            appdata = os.path.join(os.environ.get('APPDATA', ''), 'xms_python_logs')
            os.makedirs(appdata, exist_ok=True)
            return os.path.join(appdata, f'{XmLog.top_module}.log')

        def debug(self, msg, logger_name=None):
            """Log a debug level message."""
            logger = self._get_logger(logger_name)
            logger.debug(msg)

        def info(self, msg, logger_name=None):
            """Log an info level message."""
            logger = self._get_logger(logger_name)
            logger.info(msg)

        def warning(self, msg, logger_name=None):
            """Log a warning level message."""
            logger = self._get_logger(logger_name)
            logger.warning(msg)

        def error(self, msg, logger_name=None):
            """Log an error level message."""
            logger = self._get_logger(logger_name)
            logger.error(msg)

        def critical(self, msg, logger_name=None):
            """Log a critical level message."""
            logger = self._get_logger(logger_name)
            logger.error(msg)

        def exception(self, msg, logger_name=None):
            """Log a critical level message."""
            logger = self._get_logger(logger_name)
            logger.exception(msg)

        def _get_logger(self, logger_name):
            """Get the appropriate logger to use with a log message.

            Args:
                logger_name (str): Name of the logger to use (using logging standard library conventions). If None,
                    returns the top-level module logger.

            Returns:
                See description

            """
            if logger_name is not None:
                return logging.getLogger(logger_name)
            else:
                return self._logger

    instance: Optional[__XmLog] = None

    def __init__(self):
        """Construct the logger."""
        if not XmLog.instance:
            XmLog.instance = XmLog.__XmLog()

    def __getattr__(self, name):
        """Call through to singleton instance for attributes."""
        return getattr(self.instance, name)
