"""Module for the WeirStructureWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['write_weir_structures']

# 1. Standard Python modules
import logging
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UGrid2d
from xms.data_objects.parameters import Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData

# 4. Local modules
from xms.cmsflow.file_io.row_or_column_structure_writer import RowOrColumnStructureWriter


def write_weir_structures(
    coverage: Coverage, data: CoverageBaseData, ugrid: UGrid2d, logger: logging.Logger, wrote_header: bool,
    cards: TextIO
) -> bool:
    """
    Write all the data needed by weirs in a structure coverage.

    Args:
        coverage: Coverage containing geometry to write.
        data: Data manager for the coverage. Should have its component_id_map initialized.
        ugrid: The QuadTree to snap the coverage to.
        logger: Where to log any warnings or errors.
        wrote_header: Whether the structures header has already been written.
        cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.

    Returns:
        Whether the `!Structures` header was written (either before or by calling this function).
    """
    writer = WeirStructureWriter(coverage, data, ugrid, logger, wrote_header, cards)
    return writer.write()


class WeirStructureWriter(RowOrColumnStructureWriter):
    """Class for writing all the data needed by weirs in a structure coverage."""
    def __init__(
        self, coverage: Coverage, data: CoverageBaseData, ugrid: UGrid2d, logger: logging.Logger, wrote_header: bool,
        cards: TextIO
    ):
        """
        Initialize the writer.

        Args:
            coverage: Coverage containing geometry to write.
            data: Data manager for the coverage. Should have its component_id_map initialized.
            ugrid: The QuadTree to snap the coverage to.
            logger: Where to log any warnings or errors.
            cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.
            wrote_header: Whether the structures header has already been written.
        """
        super().__init__(coverage, data, ugrid, logger, wrote_header, cards)
        self._structure_display_name: str = 'weir'
        self._structure_group_name: str = 'weir'

    def _write_specific_structure(self, cells: list[int]):
        """Write the information for a weir."""
        self._cards.write('WEIR_STRUCT_BEGIN', indent=0)

        line_cells = ' '.join(str(cell + 1) for cell in cells)
        value = f'{len(cells)}   {line_cells}'
        self._cards.write('CELL_IDS', value)

        group = self._section.group('weir')

        distribution = group.parameter('distribution_coeff').value
        self._cards.write('DISTRIBUTION_COEFFICIENT', distribution)

        orientation = group.parameter('orientation').value.upper()
        self._cards.write('ORIENTATION_SEA', orientation)

        crest = group.parameter('weir_type').value
        crest = 'SHARP' if crest == 'Sharp-crested' else 'BROAD'
        self._cards.write('CREST_TYPE', crest)

        bay = group.parameter('flow_coeff_bay_to_sea').value
        self._cards.write('FLOW_COEFF_FROM_BAY', bay)

        sea = group.parameter('flow_coeff_sea_to_bay').value
        self._cards.write('FLOW_COEFF_FROM_SEA', sea)

        elevation = group.parameter('crest_elevation').value
        self._cards.write('CREST_ELEVATION', elevation)

        method = group.parameter('calc_method').value
        index = group.parameter('calc_method').options.index(method)
        self._cards.write('METHOD', index + 1)

        self._cards.write('WEIR_STRUCT_END', indent=0)
