"""Determines how to run an CMS-Wave simulation and read the solution."""

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand
from xms.components.bases.run_base import RunBase

# 4. Local modules
from xms.cmswave.feedback.solution_import_runner import import_solution_with_feedback


class SimulationRun(RunBase):
    """A class that tells SMS how to run CMS-Wave and load its solution."""
    def __init__(self, dummy_mainfile=''):
        """Constructor."""
        super().__init__()

    def get_executables(self, sim, query, filelocation):
        """
        Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.


        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (:obj:`str`): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        # Get the simulation hidden component
        sim_comp = query.item_with_uuid(sim.uuid, model_name='CMS-Wave', unique_name='Sim_Component')

        # Setup the solution load ActionRequest
        load_sol = self._create_solution_load_action(filelocation, sim_comp.main_file)

        cmd = ExecutableCommand(
            executable='CMS-Wave',
            model='CMS-Wave',
            executable_order=0,
            display_name='CMS-Wave',
            run_weight=100,
            progress_script='xml_entry_points/simulation_progress.py'
        )
        cmd.add_commandline_arg(f'{sim.name}.sim')
        cmd.add_solution_file(load_sol)
        commands = [cmd]

        return commands

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ActionRequest]`): The solution load ActionRequests for the simulation
        """
        sim_item = query.item_with_uuid(sim.uuid, model_name='CMS-Wave', unique_name='Sim_Component')
        return [self._create_solution_load_action(file_location=filelocation, sim_comp_mainfile=sim_item.main_file)]

    def _create_solution_load_action(self, file_location, sim_comp_mainfile):
        """Create an ActionRequest to read a solution.

        Args:
            file_location (:obj:`str`): Path to the project export folder
            sim_comp_mainfile (:obj:`str`): Path to the simulation hidden component's main file

        Returns:
            (:obj:`ActionRequest`): The solution load ActionRequest
        """
        return ActionRequest(
            main_file=os.path.join(file_location, 'fort.15'),
            modality='MODAL',
            class_name='SimulationRun',
            module_name='xms.cmswave.simulation_runner.simulation_run',
            method_name='read_solution',
            parameters={'sim_mainfile': sim_comp_mainfile}
        )

    def read_solution(self, query, params, win_cont):
        """Reads the CMS-Wave Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
                are required for adding vertices to the Query Context with Add().
            win_cont (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                    if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        sim_name = query.parent_item().name
        filenames = [f'{sim_name}.obs', f'{sim_name}.nst', f'{sim_name}.wav', f'{sim_name}.brk', f'{sim_name}.rad']
        import_solution_with_feedback(filenames, win_cont, query)
        return [], []
