"""Runs XMS DMI component-based migration for XMS project I/O events."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import argparse
import os
import sqlite3

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv

# 4. Local modules


def project_io_main(db_path, save_type):
    """Callable entry point for the component project IO script.

    Args:
        db_path (str): Path to the project merge DMI database file
        save_type (str): The save type event
    """
    os.environ['XMS_OPENING_PROJECT'] = 'TRUE'
    query = Query(timeout=300000)

    conn = sqlite3.connect('file:' + db_path.replace('"', ''), uri=True)
    conn.execute('BEGIN TRANSACTION;')
    select = f'SELECT Class, Module, MainFile, NewLocation, rowid FROM ProjectIO WHERE SaveType = "{save_type}";'
    cursor = conn.execute(select)
    action_requests = []
    messages = []
    disp_opts = []

    for component in cursor:
        try:
            class_name = component[0]
            module_name = component[1]
            old_main_file = component[2]
            new_path = component[3]
            row_id = component[4]

            mod = __import__(module_name, fromlist=[class_name])
            klass = getattr(mod, class_name)
            class_instance = klass(old_main_file)

            new_main_file, comp_messages, comp_actions = class_instance.save_to_location_base(new_path, save_type)
            action_requests.extend(comp_actions)
            messages.extend(comp_messages)
            if hasattr(class_instance, 'get_display_options'):
                get_display_option_method = class_instance.get_display_options
                disp_opts.extend(get_display_option_method())

            update = 'UPDATE ProjectIO SET MainFile = "' + new_main_file + '" WHERE rowid = ' + str(row_id) + ';'
            conn.execute(update)
        except Exception as err:  # Log the exception for debuggin purposes. XMS will not load this component
            XmEnv.report_error(err)
    conn.execute('COMMIT TRANSACTION;')

    arg_list = []
    if action_requests:
        # Unwrap the pure Python ActionRequests.
        arg_list.append({"actions": [action._instance for action in action_requests]})
    if messages:
        arg_list.append({"messages": messages})
    if disp_opts:
        arg_list.append({"display_options": disp_opts})
    if arg_list:
        query._impl._instance.Set(arg_list)
        query.send(True)


if __name__ == "__main__":
    arguments = argparse.ArgumentParser(description="Component method runner.")
    arguments.add_argument(dest='script', type=str, help='script to run')
    arguments.add_argument(dest='db', type=str, help='database of project components')
    arguments.add_argument(dest='save_type', type=str, help='one of: DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK')
    parsed_args = arguments.parse_args()
    project_io_main(parsed_args.db, parsed_args.save_type)
