from .. import _xmsconstraint


class RefineLevel(object):
    """ Refinement level used by quad grid cells."""

    def __init__(self, level=0, instance=None):
        """Initialize with given level of refinement.
        Args:
            level (int): The refinement level.
            instance: Used to set the wrapped C++ instance.
        """
        if instance is not None:
            self._instance = instance
            return

        if not level >= 0:
            raise ValueError('Level must be greater than or equal to 0.')

        self._instance = _xmsconstraint.quadtree.RefineLevel(level)

    def __eq__(self, other):
        other_instance = getattr(other, '_instance', None)
        if not other_instance or \
                not isinstance(other_instance,
                               _xmsconstraint.quadtree.RefineLevel):
            return False
        return other_instance.ToInteger() == self._instance.ToInteger()

    def __ne__(self, other):
        result = self.__eq__(other)
        return not result

    def __str__(self):
        return '<RefineLevel: level={}>'.format(self.to_integer())

    def __repr__(self):
        return "RefineLevel(level={})".format(self.to_integer())

    def to_integer(self):
        """The refinement level as an integer."""
        return self._instance.ToInteger()

    def next_less_refined(self):
        """The next less refined level of refinement."""
        if self._instance.ToInteger() == 0:
            raise ValueError('Cannot be less refined than zero refinement.')
        return RefineLevel(instance=self._instance.NextLessRefined())

    def next_more_refined(self):
        """The next more refined level of refinement."""
        return RefineLevel(instance=self._instance.NextMoreRefined())

    def more_refined_than(self, level):
        """Check if level is more refined than another.
        Args:
            level: The level to compare to.
        Returns:
            True if self is more refined than level, false otherwise.
        """
        if not isinstance(level, RefineLevel):
            raise ValueError("level must be of type RefineLevel")
        return self._instance.MoreRefinedThan(level._instance)

    def less_refined_than(self, level):
        """Check if level is less refined than another.
        Args:
            level: The level to compare to.
        Returns:
            True if self is less refined than level, false otherwise.
        """
        if not isinstance(level, RefineLevel):
            raise ValueError("level must be of type RefineLevel")
        return self._instance.LessRefinedThan(level._instance)

    def as_refined_as(self, level):
        """Check if level is as refined as another.
        Args:
            level: level to compare to.
        Returns:
            True if self is as refined as level, false otherwise.
        """
        if not isinstance(level, RefineLevel):
            raise ValueError("level must be of type RefineLevel")
        return self._instance.AsRefinedAs(level._instance)

    def same_refinement(self, level):
        """Check if level is the same refinement level as another.
        Args:
            level: The level to compare to.
        Returns:
            True if self is more the same refinement level, false otherwise.
        """
        if not isinstance(level, RefineLevel):
            raise ValueError("level must be of type RefineLevel")
        return self._instance.SameRefinement(level._instance)

    @property
    def unset(self):
        """
        Is the level unset?
        """
        return self._instance.IsUnset()
