"""Class to manage display for ObsTargetComponent."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import csv
import os
import shutil

# 2. Third party modules
import numpy as np
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.guipy import time_format as tf
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.coverage.components.generic_coverage_component import FEATURE_TYPE_TEXT
from xms.coverage.data.obs_target_data import ObsTargetData
from xms.coverage.gui.obs_target_dialog import ObsTargetDialog

OBS_INITIAL_ID_FILES = {
    ObsTargetData.OBS_CATEGORY_POINT: 'initial_point.ids',
    ObsTargetData.OBS_CATEGORY_ARC: 'initial_arc.ids',
    ObsTargetData.OBS_CATEGORY_ARC_GROUP: 'initial_arc_group.ids',
    ObsTargetData.OBS_CATEGORY_POLY: 'initial_poly.ids',
}
OBS_TARGET_DISPLAY_FILES = {
    ObsTargetData.OBS_CATEGORY_POINT: 'obs_point_disp.json',
    ObsTargetData.OBS_CATEGORY_ARC: 'obs_arc_disp.json',
    ObsTargetData.OBS_CATEGORY_ARC_GROUP: 'obs_arc_group_disp.json',
    ObsTargetData.OBS_CATEGORY_POLY: 'obs_poly_disp.json',
}
OBS_CATEGORY_NAMES = {
    ObsTargetData.OBS_CATEGORY_POINT: 'point',
    ObsTargetData.OBS_CATEGORY_ARC: 'arc',
    ObsTargetData.OBS_CATEGORY_ARC_GROUP: 'arc_group',
    ObsTargetData.OBS_CATEGORY_POLY: 'poly'
}
REG_KEY_PREFIX = 'observation_target_display_options'


class ObsTargetComponentDisplay:
    """Class to manage display for ObsTargetComponent."""
    def __init__(self, comp):
        """Initializes the base component class.

        Args:
            comp (ObsTargetComponent): The component to manage
        """
        self._comp = comp
        self.selected_ids = {}

    def _unpack_xms_data(self, params):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            params (dict): The ActionRequest parameter map
        """
        # Get the XMS attribute ids of the selected features.
        selection = params.get('selection', {})

        # Get the component id map of the selected features.
        have_files = True if params.get('id_files') else False
        if have_files:
            target_types = [TargetType.point, TargetType.arc, TargetType.arc_group, TargetType.polygon]
            self._comp.load_coverage_component_id_map(params['id_files'])
            # Find the component ids of all the selected features. These coverages are meant to be read only, so we
            # just use the same id for the feature and the component ids when creating one. A user could do something
            # dumb though, like deleting a feature and renumbering.
            for idx, feature_type in enumerate(FEATURE_TYPE_TEXT):
                feature_ids = selection.get(feature_type, [])
                comp_ids = []
                for feature_id in feature_ids:
                    comp_id = self._comp.get_comp_id(target_types[idx], feature_id)
                    # User could also be dumb and add a new feature. Nothing currently blocks them, but they can't add
                    # any observations to it. Don't put it in the dialog.
                    if comp_id > -1:
                        comp_ids.append(comp_id)
                self.selected_ids[feature_type] = comp_ids

        # Delete the id files dumped by xms.
        shutil.rmtree(os.path.join(os.path.dirname(self._comp.main_file), 'temp'), ignore_errors=True)

    def _get_selection(self, target_type):
        """Get a list of the currently selected features of a given type.

        Assumes component ids for the currently selected features have already been filled in the component id maps.

        Args:
            target_type (TargetType): The feature type's TargetType enum value from xmsguipy

        Returns:
            list[int]: The selected feature ids of the specified type, or empty list if there are none
        """
        # Get the component id map for the coverage
        cov_dict = self._comp.comp_to_xms.get(self._comp.cov_uuid, {})
        # Get the component id map for the requested feature type
        entity_dict = cov_dict.get(target_type, {})
        # Get all the feature ids, which are the values of the dict and are lists. Return a flattened version.
        feature_ids = np.array(list(entity_dict.values()))
        return feature_ids.flatten().tolist()

    def ensure_display_options_exist(self, custom_display=None):
        """Ensure the component display options json file exists.

        Args:
            custom_display (Optional[list[str]]): List of custom display options JSON files to override defaults with.
                Should be size and order of feature types in ObsTargetData.OBS_CATEGORY_* enum. Elements should be
                None for feature types that you don't want to override. Elements should be the absolute path to the
                display options JSON file for the feature type with custom display.
        """
        comp_dir = os.path.dirname(self._comp.main_file)
        commit = False
        data_folder = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data')
        for idx, json_file in OBS_TARGET_DISPLAY_FILES.items():
            self._comp.disp_opts_files[idx] = os.path.join(comp_dir, json_file)
            if not os.path.exists(self._comp.disp_opts_files[idx]):
                commit = True
                # Read the default display options, and save ourselves a copy with a randomized UUID.
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                if custom_display and custom_display[idx]:
                    default_file = custom_display[idx]
                else:
                    default_file = os.path.join(data_folder, json_file)
                json_dict = read_display_options_from_json(default_file)
                json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self._comp.main_file))
                categories.from_dict(json_dict)
                write_display_options_to_json(self._comp.disp_opts_files[idx], categories)
                # Save our display list UUID to the main file
                self._comp.data.info.attrs[f'display_uuid_{idx}'] = categories.uuid
        if commit:
            self._comp.data.commit()

    def initialize_display(self, query):
        """Initialize the XMS component display list.

        Args:
            query (Query): The XMS interprocess communication object.

        Returns:
            tuple([], []): Empty message and ActionRequest lists
        """
        # Get the UUID of our owning map module coverage.
        self._comp.cov_uuid = query.parent_item_uuid()
        if not self._comp.cov_uuid:
            return [('ERROR', 'Could not get observation target coverage UUID.')], []
        self._comp.data.info.attrs['cov_uuid'] = self._comp.cov_uuid

        self.initialize_component_ids()

        # Save data and request a display refresh in XMS.
        self._comp.data.commit()
        return [], []

    def initialize_component_ids(self):
        """Initializes the component ids."""
        # If we came from a model native read, initialize the component ids.
        target_types = [TargetType.point, TargetType.arc, TargetType.arc_group, TargetType.polygon]
        for idx, basename in OBS_INITIAL_ID_FILES.items():
            filename = os.path.join(os.path.dirname(self._comp.main_file), basename)
            if os.path.isfile(filename):
                target_type = target_types[idx]
                ids = read_display_option_ids(filename)
                filesystem.removefile(filename)
                for att_id in ids:
                    self._comp.update_component_id(target_type, att_id, att_id)

        json_files = self._comp.disp_opts_files
        cov_uuid = self._comp.cov_uuid
        coverage_type = self._comp.data.info.attrs['feature_type']
        if coverage_type == ObsTargetData.UNINITIALIZED_COMP_ID:  # Coverage has all feature types
            self._comp.display_option_list = [
                XmsDisplayMessage(file=json_files[ObsTargetData.OBS_CATEGORY_POINT], edit_uuid=cov_uuid),
                XmsDisplayMessage(file=json_files[ObsTargetData.OBS_CATEGORY_ARC], edit_uuid=cov_uuid),
                XmsDisplayMessage(file=json_files[ObsTargetData.OBS_CATEGORY_ARC_GROUP], edit_uuid=cov_uuid),
                XmsDisplayMessage(file=json_files[ObsTargetData.OBS_CATEGORY_POLY], edit_uuid=cov_uuid),
            ]
        else:  # Coverage has been restricted to a certain type
            self._comp.display_option_list = [
                XmsDisplayMessage(file=json_files[coverage_type], edit_uuid=cov_uuid),
            ]

    def open_display_options(self, query, parent):
        """Opens the Display Options dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            parent (QWidget): The Qt parent window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Load the existing display options from the JSON file and show the display options dialog.
        all_categories = []
        coverage_type = self._comp.data.info.attrs['feature_type']
        disp_file_idxs = []
        registry_keys = []
        show_all = coverage_type == ObsTargetData.UNINITIALIZED_COMP_ID
        for idx, feature_type in enumerate(OBS_TARGET_DISPLAY_FILES):
            if show_all or feature_type == coverage_type:
                categories = CategoryDisplayOptionList()
                json_dict = read_display_options_from_json(self._comp.disp_opts_files[feature_type])
                categories.from_dict(json_dict)
                all_categories.append(categories)
                disp_file_idxs.append(idx)
                registry_keys.append(reg_key_from_obs_category(feature_type))

        dlg = CategoryDisplayOptionsDialog(
            category_lists=all_categories,
            parent=parent,
            package_name='xmscoverage',
            registry_keys=registry_keys,
            show_targets=True
        )
        dlg.setWindowIcon(QIcon(get_xms_icon()))
        if dlg.exec():
            # Update the display options JSON file and request a display refresh if the user accepts the dialog.
            category_lists = dlg.get_category_lists()
            for idx, category_list in enumerate(category_lists):
                disp_file_idx = disp_file_idxs[idx]
                write_display_options_to_json(self._comp.disp_opts_files[disp_file_idx], category_list)
                if self._comp.cov_uuid:
                    self._comp.display_option_list.append(
                        XmsDisplayMessage(
                            file=self._comp.disp_opts_files[disp_file_idx], edit_uuid=self._comp.cov_uuid
                        )
                    )
        return [], []

    def open_observations(self, params, parent, query=None):
        """Opens the Observations dialog for a point.

        Args:
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            parent (QWidget): The QT parent window container
            query (Query): XMS interprocess communication object

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                  level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        self._unpack_xms_data(params[0])

        # Get the time format from the XMS preferences
        time_format = '%x %X'  # regional format by default
        format_json = query.global_time_settings if query else ''
        if format_json:
            time_formatter = tf.XmsTimeFormatter(format_json)
            time_format = time_formatter.abs_specifier

        dlg = ObsTargetDialog(
            parent=parent, target_data=self._comp.data.targets, selected_ids=self.selected_ids, time_format=time_format
        )
        dlg.exec()
        return [], []

    def update_id_files(self, initial):
        """Writes the display id files.

        Args:
            initial (bool): True if we should write the initial component att/comp id files (solution import). Note it
                is assumed that the feature ids assigned to the features match the component ids.
        """
        comp_path = os.path.dirname(self._comp.main_file)
        target_types = [TargetType.point, TargetType.arc, TargetType.arc_group, TargetType.polygon]
        for idx, (feature_type, initial_file) in enumerate(OBS_INITIAL_ID_FILES.items()):
            # Get all the observation target rows in the table of this feature type
            vals = self._comp.data.targets.where(self._comp.data.targets.feature_type == feature_type, drop=True)
            if vals.comp_id.size == 0:
                continue

            # Write an ID file for each display category.
            feature_ids = []
            target_type = target_types[idx]
            for cat_id, dset in vals.groupby('category'):
                id_file = os.path.join(comp_path, f'obs_{feature_type}_{int(cat_id)}.obsid')
                csv_file = f'{id_file}_targets'
                comp_ids = dset.comp_id.data.tolist()
                label_texts = self._comp.get_label_texts(target_type, comp_ids)
                write_display_option_ids(id_file, comp_ids, label_texts)
                feature_ids.extend(comp_ids)
                # Write the observation target data to a CSV.
                df = dset.drop_vars(['feature_type', 'category']).to_dataframe()
                df.to_csv(csv_file, quoting=csv.QUOTE_NONNUMERIC, date_format=ISO_DATETIME_FORMAT, header=False)

            # Copy to initial id file, if importing
            if initial and feature_ids:
                write_display_option_ids(os.path.join(comp_path, initial_file), feature_ids)

    def update_reftime(self):
        """Update the observation reference datetime in the display options JSON files."""
        comp_path = os.path.dirname(self._comp.main_file)
        for json_filename in OBS_TARGET_DISPLAY_FILES.values():
            json_filepath = os.path.join(comp_path, json_filename)
            if os.path.isfile(json_filepath):
                json_dict = read_display_options_from_json(json_filepath)
                categories = CategoryDisplayOptionList()
                categories.from_dict(json_dict)
                categories.obs_reftime = self._comp.data.reftime
                write_display_options_to_json(json_filepath, categories)

    def update_dset_uuid(self):
        """Update the observation dset uuid in the display options JSON files."""
        comp_path = os.path.dirname(self._comp.main_file)
        for json_filename in OBS_TARGET_DISPLAY_FILES.values():
            json_filepath = os.path.join(comp_path, json_filename)
            if os.path.isfile(json_filepath):
                json_dict = read_display_options_from_json(json_filepath)
                categories = CategoryDisplayOptionList()
                categories.from_dict(json_dict)
                categories.obs_dset_uuid = self._comp.data.dset_uuid
                write_display_options_to_json(json_filepath, categories)

    def get_feature_selection_params(self, query):
        """Get the feature selection parameters passed to the feature selection dialog methods.

        This is used to get feature selection info for coverage tree item actions.

        Args:
            query (Query): The XMS interprocess communicator

        Returns:
            list[dict]: The parameters to pass to the feature selection dialog method. See `params` arg of the
            open_observations method.
        """
        id_files = query.load_component_ids(
            self._comp, points=True, arcs=True, arc_groups=True, polygons=True, only_selected=True, delete_files=False
        )
        target_types = [TargetType.point, TargetType.arc, TargetType.arc_group, TargetType.polygon]
        selections = {FEATURE_TYPE_TEXT[i]: self._get_selection(target_types[i]) for i in range(len(target_types))}
        return [{'selection': selections, 'id_files': id_files}]


def reg_key_from_obs_category(obs_category: int) -> str:
    """Return the registry key string given the feature type.

    Args:
        obs_category: ObsTargetData.OBS_CATEGORY_POINT etc. (0=point, 1=arc, 2=arc_group, 3=poly).

    Returns:
        See description.
    """
    return f'{REG_KEY_PREFIX}-{OBS_CATEGORY_NAMES[obs_category]}'
