"""Creates an observation targets coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.data_objects.parameters import Component
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.coverage.components.obs_target_component import ObsTargetComponent
from xms.coverage.components.obs_target_component_display import ObsTargetComponentDisplay
from xms.coverage.data.obs_target_data import ObsTargetData


def create_component_folder(comp_uuid):
    """Create a folder for a component in the XMS temp components folder.

    Args:
        comp_uuid (str): UUID of the new component

    Returns:
        str: Path to the new component folder
    """
    temp_comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(temp_comp_dir)
    return temp_comp_dir


class ObsTargetComponentBuilder:
    """Builds hidden component for a generic observation targets coverage."""
    FEATURE_TYPES = ['Point', 'Arc', 'Arc Group', 'Polygon']  # For building default names
    ENTITY_LOOKUP = {  # Mapping from ObsTargetData.OBS_CATEGORY_* enum values to xmsguipy TargetType enum values
        ObsTargetData.OBS_CATEGORY_POINT: TargetType.point,
        ObsTargetData.OBS_CATEGORY_ARC: TargetType.arc,
        ObsTargetData.OBS_CATEGORY_ARC_GROUP: TargetType.arc_group,
        ObsTargetData.OBS_CATEGORY_POLY: TargetType.polygon,
    }

    def __init__(self, coverage_feature_type=ObsTargetData.UNINITIALIZED_COMP_ID):
        """Construct the builder.

        Args:
            coverage_feature_type (Optional[int]): The coverage level feature object type restriction. If anything other
                than the default, right-click menus and display options will be hidden for the other feature types.
                If specified should be one of the ObsTargetData.OBS_CATEGORY_* enum values.
        """
        self._comp_uuid = str(uuid.uuid4())
        self._comp_dir = create_component_folder(self._comp_uuid)

        # Set these to a display options JSON file that will be used for the target categories for the feature type.
        # Files should be In ObsTargetData.OBS_CATEGORY_* index order. If you want to override the default categories
        # for a feature type, your id file names in the JSON file should have the naming convention of:
        # 'obs_<feature type>_<category id>.obsid' Where <feature type> is the ObsTargetData.OBS_CATEGORY_* enum of the
        # feature type and <category id> is the CategoryDisplayOption.id of the id file's category in the JSON file,
        # e.g. 'obs_0_0.obsid', 'obs_0_1.obsid'
        self.custom_display_options = [None, None, None, None]

        # This will be used by XMS when rendering observation whisker plots to find the closest observations to the
        # current solution dataset timestep if the dataset does not have a reference datetime defined. Only applicable
        # if the observations are transient.
        self.reftime = None  # Should be a datetime.datetime object

        # Uuid of the dataset associated with the observations. Targets should only be drawn if it is the active dset.
        self.dset_uuid = ''

        # This the coverage level feature type restriction. If not default, right-click menus and display options for
        # feature objects other than the type specified will be hidden.
        self.coverage_feature_type = coverage_feature_type  # All feature types allowed by default

        # These will be used when adding rows individually and the optional kwarg to add_observation() is not provided.
        self.current_feature_type = ObsTargetData.OBS_CATEGORY_POINT
        self.current_disp_category = 0
        self.current_feature_name = None

        # These are the arrays used to build the observation targets Dataset, need to be parallel
        self.feature_type = []  # Feature attribute type
        self.category = []  # CategoryDisplayOption.id in JSON file for the row's display list category
        self.name = []
        self.time = []  # np.datetime64 timestamps
        self.interval = []
        self.observed = []
        self.computed = []
        self.comp_ids = []  # Coordinates of the Dataset, should be same as the id assigned in the coverage geometry
        self.label_texts = {}  # Key=TargetType, value=dict{comp/feature id: display label text}

    def _build_obs_target_data(self):
        """Create the observation targets coverage's hidden component and data.

        Returns:
            xarray.Dataset: The observation targets dataset
        """
        targets = {
            'feature_type': ('comp_id', np.array(self.feature_type, dtype=np.int32)),
            'category': ('comp_id', np.array(self.category, dtype=np.int32)),
            'name': ('comp_id', np.array(self.name, dtype=object)),
            'time': ('comp_id', np.array(self.time, dtype=np.datetime64)),
            'interval': ('comp_id', np.array(self.interval, dtype=np.float64)),
            'observed': ('comp_id', np.array(self.observed, dtype=np.float64)),
            'computed': ('comp_id', np.array(self.computed, dtype=np.float64)),
        }
        coords = {'comp_id': self.comp_ids}
        return xr.Dataset(data_vars=targets, coords=coords)

    def _build_python_component(self, main_file, cov_uuid) -> ObsTargetComponent:
        """Build the Python component and set the data we have stored up.

        Args:
            main_file (str): Path to the component's main file, will be created here
            cov_uuid (str): UUID of the component's owning coverage

        Returns:
            ObsTargetComponent: The new Python component
        """
        py_comp = ObsTargetComponent(main_file, custom_display=self.custom_display_options)
        py_comp.cov_uuid = cov_uuid
        py_comp.data.info.attrs['cov_uuid'] = cov_uuid
        py_comp.data.info.attrs['feature_type'] = self.coverage_feature_type
        if self.reftime is not None:
            py_comp.data.reftime = self.reftime  # Use property setter to convert to string
        py_comp.data.dset_uuid = self.dset_uuid
        py_comp.data.targets = self._build_obs_target_data()
        py_comp.data.commit()
        return py_comp

    def _set_display(self, py_comp, importing_component, cov_uuid):
        """Write/update display option and id files and make sure the display gets intitialized in XMS.

        Args:
            py_comp (ObsTargetComponent): The built Python component object
            importing_component (bool): True if importing from a pooled component script.
            cov_uuid (str): UUID of the component's owning coverage
        """
        if self.label_texts:  # Set display label texts for individual features if specified
            py_comp.label_texts = {cov_uuid: self.label_texts}
        display_helper = ObsTargetComponentDisplay(py_comp)
        display_helper.update_id_files(True)
        if self.reftime is not None:
            display_helper.update_reftime()
        if self.dset_uuid:
            display_helper.update_dset_uuid()
        # Initialize the display in XMS
        if importing_component:
            # We only need to write the initial id files if we are building for a model native import. If building for
            # a component ActionRequest script, XMS can handle initializing the display with the data we give it here.
            display_helper.initialize_component_ids()

    def add_observation(
        self,
        feature_id,
        time,
        interval,
        observed,
        computed,
        feature_name=None,
        feature_type=None,
        category=None,
        label_text=None
    ):
        """Add a single observation row to the target observation data.

        Args:
            feature_id (int): Feature attribute ID assigned to the feature object in the geometry, will be used for the
                component ID.
            time (Union[datetime.datetime, np.datetime64]): The timestamp of the observation record
            interval (float): The observation record interval
            observed (float): The observation record observed value
            computed (float): The observation record computed value
            feature_name (Optional[str]): Name of the feature record, usually same for all observations of a given
                feature object. If not provided, current class variable value will be used.
            feature_type (Optional[int]): The feature object type, one of the ObsTargetData.OBS_CATEGORY_* enum values.
                If not provided, current class variable value will be used.
            category (Optional[int]): The display option category ID of the observation. Should match the enum of the
            label_text (Optional[str]): Display label text for the feature
        """
        # Use class member defaults for unspecified kwargs
        feature_type = self.current_feature_type if feature_type is None else feature_type
        feature_name = self.current_feature_name if not feature_name else feature_name
        if not feature_name:  # Generate a default name
            feature_name = f'{self.FEATURE_TYPES[feature_type]} #{feature_id}'
        category = self.current_disp_category if category is None else category

        feature_type = int(feature_type)
        self.feature_type.append(feature_type)  # Feature attribute type
        self.category.append(int(category))  # CategoryDisplayOption.id in JSON file for the row's display list category
        self.name.append(feature_name)
        if isinstance(time, datetime.datetime):  # Allow datetimes to be passed in
            # time = np.datetime64(time)
            time = np.datetime64(time, 'ns')  # Use nanosecond precision or Xarray logs a deprecation warning
        self.time.append(time)  # np.datetime64 timestamps
        self.interval.append(interval)
        self.observed.append(observed)
        self.computed.append(computed)
        comp_id = int(feature_id)
        self.comp_ids.append(comp_id)
        if label_text is not None:
            entity_type = self.ENTITY_LOOKUP[feature_type]
            self.label_texts.setdefault(entity_type, {})[comp_id] = label_text  # Store display label text if specified

    def build_obs_target_component(self, cov_uuid, importing_component):
        """Create an observation targets coverage's hidden component and data_object.

        Args:
            cov_uuid (str): UUID of the observation targets coverage to build component for
            importing_component (bool): True if importing from a pooled component script.

        Returns:
            tuple(Component, list): data_object for the new ObsTargetComponent to send back to SMS; component keyword
            kwargs for the call to add_coverage
        """
        # Create a new UUID and folder for the component data
        main_file = os.path.join(self._comp_dir, 'obs_comp.nc')
        # Create the data_object Component to send back to SMS
        do_comp = Component(
            comp_uuid=self._comp_uuid,
            main_file=main_file,
            model_name='Generic Coverages',
            unique_name='ObsTargetComponent'
        )
        # Create the Python component and write files
        py_comp = self._build_python_component(main_file, cov_uuid)
        self._set_display(py_comp, importing_component, cov_uuid)
        comp_data = [
            {
                'component_coverage_ids': [py_comp.uuid, py_comp.update_ids],
                'display_options': py_comp.get_display_options()
            }
        ]
        return do_comp, comp_data
