"""Creates a coverage with polygons."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon

# 4. Local modules


class PolygonCoverageBuilder:
    """Given points and lists of lines defining polygons (perhaps with holes), creates a coverage with the polygons.

    - output is a data_objects.parameters.Coverage
    - build_coverage: Builds the data_objects Coverage.

    Glossary:
        I've tried to be consistent with the following terms to help avoid
        confusion:

        poly = A poly is a list of point indexes with the last point
        equal to the first point. Poly points can be in any order, clockwise
        or counterclockwise: [9, 3, 1, 6, 9]

        polygon = A polygon is a list of poly, outer first, then inner holes
        if any: [[3, 6, 8, 2, 3], [6, 7, 8, 6]]

        multipoly = list of polygon, each one associated with the same dataset value.
    """
    def __init__(self, point_locations, projection, coverage_name, logger=None):
        """Initializes the class.

        Args:
            point_locations: List of xyz points which are 3-tuples, e.g. [(1.0, 3.0, 0.0), (2.0, 1.0, 1.0)]
            projection: The map projection.
            coverage_name (str): Name to be given to the new coverage.
            logger (Optional[Logger]): The logger to use. If not provided will be the xms.coverage logger.
        """
        self._projection = projection
        self._coverage_name = coverage_name
        self._logger = logger if logger is not None else logging.getLogger('xms.coverage')

        self._pt_locs = point_locations  # xyz locations of points

        # Stuff for building a coverage
        self.dataset_polygon_ids = {}  # Dict of dataset value -> list of Polygon IDs
        self._next_pt_id = 1
        self._next_arc_id = 1
        self._next_poly_id = 1
        self._pt_hash = {}  # Dict of point idx -> Point
        self._arc_hash = {}  # Dict of (node id1, node id2) -> Arc

    def _get_or_hash_point(self, point_idx):
        """Get an existing data_objects point using a int hash (creates the point if it doesn't exist).

        Args:
            point_idx (int): Grid point index.

        Returns:
            data_objects.parameters.Point: The existing or newly created point associated with the passed in location
        """
        if point_idx not in self._pt_hash:
            do_point = Point(self._pt_locs[point_idx][0], self._pt_locs[point_idx][1], self._pt_locs[point_idx][2])
            do_point.id = self._next_pt_id
            self._next_pt_id += 1
            self._pt_hash[point_idx] = do_point
            return do_point
        else:
            return self._pt_hash[point_idx]

    def _get_or_hash_arc(self, start_node, end_node):
        """Get an existing data_objects arc using end node ids (creates the arc if it doesn't exist).

        Order of the end nodes does not matter. Arcs are hashed by the ordered pair of their end node ids.

        Args:
            start_node (data_objects.parameters.Point): The arc's start node
            end_node (data_objects.parameters.Point): The arc's end node

        Returns:
            data_objects.parameters.Arc: The existing or newly created arc associated with the passed in end node ids

        """
        node1_id = start_node.id
        node2_id = end_node.id
        if node1_id < node2_id:
            arc_hash = hash((node1_id, node2_id))
        else:
            arc_hash = hash((node2_id, node1_id))
        if arc_hash not in self._arc_hash:
            do_arc = Arc()
            do_arc.start_node = start_node
            do_arc.end_node = end_node
            do_arc.id = self._next_arc_id
            self._next_arc_id += 1
            self._arc_hash[arc_hash] = do_arc
            return do_arc
        else:
            return self._arc_hash[arc_hash]

    def _build_polygon_arcs(self, poly, arcs):
        """Construct a polygon arc definition given its points.

        Args:
            poly (iterable): List of polygon point definitions, in order
            arcs (list): Out variable to return the arcs that define the polygon

        """
        # Get the arcs for the outer polygon
        # x, y = poly.exterior.coords.xy
        # z = [coord[-1] for coord in poly.exterior.coords[:-1]]
        if len(poly) > 3:
            pfirst = self._get_or_hash_point(poly[0])
            p1 = None
            p0 = pfirst
            for i in range(1, len(poly) - 1):
                p1 = self._get_or_hash_point(poly[i])
                arc = self._get_or_hash_arc(p0, p1)
                p0 = p1
                arcs.append(arc)

            arc = self._get_or_hash_arc(p1, pfirst)
            arcs.append(arc)

    def build_coverage(self, multipolys):
        """Creates a data_objects Coverage with the Polygons and Arcs.

        Args:
            multipolys: Dict of dataset value -> multipoly. A multipoly
             is a list of polygon and a polygon is a list of poly (outer, inners),
             and a poly is a list of ugrid point indexes with the last point
             equal to the first point.

        Returns:
            data_objects Coverage.

        """
        self._logger.info('Creating coverage polygons...')
        cov_polys = []
        outer_arcs_set = set()  # All outer arcs
        inner_arc_set = set()  # All inner arcs
        for dataset_value, multi_poly in multipolys.items():
            for polygons in multi_poly:

                # outer polygon is first
                outer_arcs = []
                self._build_polygon_arcs(polygons[0], outer_arcs)
                outer_arcs_set.update(outer_arcs)

                # inner polygons follow
                inner_polys = []
                for poly in polygons[1:]:
                    inner_arcs = []
                    self._build_polygon_arcs(poly, inner_arcs)
                    inner_polys.append(inner_arcs)
                    inner_arc_set.update(inner_arcs)

                # Keep track of which polygons go with which dataset value
                if dataset_value not in self.dataset_polygon_ids:
                    self.dataset_polygon_ids[dataset_value] = []
                self.dataset_polygon_ids[dataset_value].append(self._next_poly_id)

                # Create a data_objects Polygon
                cov_poly = Polygon()
                cov_poly.id = self._next_poly_id
                self._next_poly_id += 1
                cov_poly.set_arcs(outer_arcs)
                if inner_polys:
                    cov_poly.set_interior_arcs(inner_polys)
                cov_polys.append(cov_poly)

        # Create the coverage
        self._logger.info('Writing coverage to disk...')
        coverage = Coverage()
        coverage.polygons = cov_polys
        leftover_inner_arcs = inner_arc_set - outer_arcs_set  # inner arcs that aren't outer arcs
        coverage.arcs = list(leftover_inner_arcs)  # This will append to the coverage's arc list
        coverage.name = self._coverage_name
        if self._projection:
            coverage.projection = self._projection
        coverage.uuid = str(uuid.uuid4())
        coverage.complete()
        return coverage
