"""A hidden Dynamic Model Interface (DMI) component for the EWN Features coverage."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from packaging import version

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.xms_display_message import XmsDisplayMessage

# 4. Local modules
from xms.ewn.components import ewn_component_display as ecd
from xms.ewn.components import ewn_component_manager as ecm
from xms.ewn.data.ewn_cov_data import EwnCovData
from xms.ewn.tools.runners.tool_runner import ToolRunner


class EwnCoverageComponent(CoverageComponentBase):
    """XMS DMI component class implementation for the EWN Features coverage."""
    def __init__(self, main_file):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        comp_dir = os.path.dirname(self.main_file)
        self.class_name = 'EwnCoverageComponent'
        self.module_name = 'xms.ewn.components.ewn_coverage_component'
        self.uuid = os.path.basename(comp_dir)
        self.data = EwnCovData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('EWN Classification Properties... ', 'open_ewn_properties'),
        ]
        self.tool_commands = [
            ('Insert EWN Features...', 'insert_ewn_features'),
            ('Generate Roughness Data Set...', 'generate_roughness_dataset'),
            ('Insert ADCIRC Levees...', 'insert_levees'),
            ('Update ADH Friction...', 'update_adh_friction'),
            ('Refine Quadtree...', 'refine_quadtree'),
        ]
        self.polygon_commands = [('Assign EWN Polygon Features...', 'open_assign_polygon_features')]
        self.arc_commands = [('Assign EWN Arc Features...', 'open_assign_arc_features')]
        self.disp_opts_file = os.path.join(comp_dir, ecd.EWN_DISPLAY_OPTIONS_JSON)
        helper = ecd.EwnCoverageDisplay(self)
        helper.ensure_display_options_exist()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.

            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        manager = ecm.EwnComponentManager(self)
        return manager.handle_save(new_path, save_type)

    def project_open_event(self, new_path):
        """Called when XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            new_path (:obj:`str`): Path to the new save location.
        """
        # If we are not a legacy file from 13.1, we do not need to initialize our display because SMS already did.
        # Will be 0.0 if no currently saved project.
        proj_version = version.parse(XmEnv.xms_environ_project_version())
        if proj_version < version.parse('1.0') or proj_version >= version.parse('13.2'):
            return
        new_disp_opts = os.path.join(new_path, os.path.basename(self.disp_opts_file))
        self.display_option_list.append(XmsDisplayMessage(file=new_disp_opts, edit_uuid=self.cov_uuid))

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list[str]`): A list of the main files of the selected components of this type.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component
        manager = ecm.EwnComponentManager(self)
        return manager.get_tree_menus()

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (:obj:`dict`): A dictionary with the key being a string of the feature entity type
                (POINT, ARC, POLYGON). The value of the dictionary is a list of IntegerLiteral ids of the selected
                feature objects.
            lock_state (:obj:`bool`): True if the the component is locked for editing. Do not change the files if
                locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        manager = ecm.EwnComponentManager(self)
        return manager.get_poly_menus(selection, id_files)

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (:obj:`dict`): A dictionary with the key being a string of the feature entity type
                (POINT, ARC, POLYGON). The value of the dictionary is a list of IntegerLiteral ids of the selected
                feature objects.
            lock_state (:obj:`bool`): True if the the component is locked for editing. Do not change the files
                if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        manager = ecm.EwnComponentManager(self)
        poly_menus = manager.get_poly_menus(selection, id_files)
        if len(poly_menus) > 1:  # Only one action allowed on double-click, first item is a spacer.
            return [], [poly_menus[1].action_request]
        return [], []

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        helper = ecd.EwnCoverageDisplay(self)
        return [], [helper.get_initialize_display_action()]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        helper = ecd.EwnCoverageDisplay(self)
        helper.initialize_display(query)
        return [], []

    def open_ewn_properties(self, query, params, win_cont):
        """Opens the EWN coverage attribute and display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        helper = ecd.EwnCoverageDisplay(self)
        helper.do_ewn_feature_properties(win_cont)
        return [], []

    def insert_ewn_features(self, query, params, win_cont):
        """Called on the 'Insert EWN Features...' right-click command.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the tuple being the
                message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        runner = ToolRunner(self, query, win_cont)
        return runner.run_tool(ToolRunner.INSERT_FEATURES_TOOL), []

    def insert_levees(self, query, params, win_cont):
        """Called on the 'Insert Levee Features...' right-click command.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        runner = ToolRunner(self, query, win_cont)
        return runner.run_tool(ToolRunner.INSERT_LEVEE_TOOL), []

    def generate_roughness_dataset(self, query, params, win_cont):
        """Called on the 'Generate Roughness Data Set...' right-click command.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        runner = ToolRunner(self, query, win_cont)
        return runner.run_tool(ToolRunner.GENERATE_ROUGHNESS_TOOL), []

    def open_assign_polygon_features(self, query, params, win_cont):
        """Opens the Assign EWN Features dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        poly_ids = []
        if 'selection' in params[0]:
            poly_ids = params[0]['selection']
        num_polys = len(poly_ids)
        if num_polys == 0:
            return [('INFO', 'No polygons selected. Select one or more polygons to assign EWN features.')], []

        helper = ecd.EwnCoverageDisplay(self)
        helper.assign_polygon_features(query, win_cont, poly_ids)
        return [], []

    def open_assign_arc_features(self, query, params, win_cont):
        """Opens the Assign EWN Features dialog for arcs and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.
        """
        arc_ids = []
        if 'selection' in params[0]:
            arc_ids = params[0]['selection']
        num_arcs = len(arc_ids)
        if num_arcs == 0:
            return [('INFO', 'No arcs selected. Select one or more arcs to assign EWN features.')], []

        helper = ecd.EwnCoverageDisplay(self)
        helper.assign_arc_features(query, win_cont, arc_ids)
        return [], []

    def update_adh_friction(self, query, params, win_cont):
        """Called on the 'Update ADH Friction...' right-click command.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        runner = ToolRunner(self, query, win_cont)
        return runner.run_tool(ToolRunner.UPDATE_ADH_FRICTION_TOOL), []

    def refine_quadtree(self, query, params, win_cont):
        """Called on the 'Refine QuadTree...' right-click command.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        runner = ToolRunner(self, query, win_cont)
        return runner.run_tool(ToolRunner.REFINE_QUADTREE), []
