"""The Assign EWN Feature polygon attribute dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
from enum import IntEnum
import math
import uuid
import webbrowser

# 2. Third party modules
import folium
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QCheckBox, QColorDialog, QComboBox, QDialogButtonBox, QDoubleSpinBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit,
    QMessageBox, QPushButton, QStyle, QTableWidget, QTableWidgetItem, QVBoxLayout
)
from shapely.geometry import Polygon

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import GridType
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon as Poly
from xms.gdal.utilities import gdal_utils
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts
from xms.ewn.gui.assign_ewn_polygon_dialog import flip_long_lat
from xms.ewn.gui.mesh_viewer_dialog import MeshViewerDialog
from xms.ewn.gui.widgets.preview_widget import add_preview_widget
from xms.ewn.tools.ewn_process import EwnProcess, meters_to_decimal_degrees
from xms.ewn.tools.runners import runner_util
from xms.ewn.tools.runners.ewn_preview_runner import preview_ewn_insert_feedback, preview_get_geometry_feedback


@dataclass
class DisplayItem:
    """DisplayItem data class."""
    name: str
    display_name: str
    color: str
    priority: str
    weight: float
    visible: bool = False
    enabled: bool = True


class Columns(IntEnum):
    """Column enum for display options."""
    PRIORITY = 0
    CHECK = 1
    NAME = 2
    COLOR = 3
    WEIGHT = 4


class AssignEwnArcDialog(XmsDlg):
    """A dialog for assigning materials to polygons."""
    def __init__(self, parent, dlg_data):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            dlg_data (:obj:`dict`): data for dialog
        """
        super().__init__(parent, 'xms.ewn.gui.assign_arc_feature_dialog')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:SMS'

        self._data = dlg_data['arc_data']
        self._vert_units = dlg_data['vertical_units']
        self._dlg_data = dlg_data
        self._units_str = '(m)' if self._vert_units == 'METERS' else '(ft)'
        self._url_file = dlg_data['map_html']
        self._mesh_viewer = None
        self._arc_pts = None
        self._box_min = None
        self._box_max = None
        self._transform = None
        self._pe_tree = self._dlg_data['query'].project_tree
        self._clipped_ugrid = None
        self._insert_ugrid = None
        self._insert_transition_polygon = None
        self._buffered_arc = None
        self._subset = None
        self._geom_uuid = None
        self._ugrids = {}
        self._subsets = {}
        self._bounding_box_factor = 1.0
        self._initial_transition_factor = 0.5
        self._initial_transition_distance = 0.5
        self._plot_transition_pts = None

        self._dbl_validator = None
        self._pos_validator = None
        self._greater_than_zero_validator = None
        self._change_limit_validator = None
        self._positive_validator = None

        self._is_geographic = self._dlg_data['projection'].coordinate_system == 'GEOGRAPHIC'
        self._is_local = self._dlg_data['projection'].coordinate_system == 'None'
        self._display_map = True

        self.is_cartesian = False
        self.is_quadtree = False

        self.new_ugrid = None
        self.transition_poly_cov = None

        self._order_and_color = [
            DisplayItem(
                name='output', color='grey', priority=0, display_name='Output Mesh/Grid', weight=1, visible=True
            ),
            DisplayItem(
                name='input', color='orange', priority=3, display_name='Input Mesh/Grid', weight=0.5, visible=True
            ),
            DisplayItem(
                name='dune_polygon',
                color='gray',
                priority=4,
                display_name='Calculated Dune Shape',
                weight=2,
                visible=True,
                enabled=True
            ),
            DisplayItem(
                name='dune_arc',
                color='#000000',
                priority=5,
                display_name='Dune Arc',
                weight=2,
                visible=True,
                enabled=True
            ),
            DisplayItem(
                name='transition',
                color='red',
                priority=1,
                display_name='Transition Area',
                weight=2,
                visible=True,
                enabled=True
            )
        ]

        if not self._is_geographic and not self._is_local:
            # Set up a transformation from input non-geographic to WGS 84
            src_wkt = self._dlg_data['projection'].well_known_text
            _, trg_wkt = gdal_utils.wkt_from_epsg(4326)
            self._transform = gdal_utils.get_coordinate_transformation(
                src_wkt, trg_wkt, set_traditional_axis_mapping=True
            )

        self._widgets = {}
        self._setup_ui()

    def _setup_ui(self):
        """Setup the ui."""
        self.setWindowTitle('Assign Arc Feature')
        self._add_main_layout()
        self._add_props_widgets_and_labels()

        # Add the preview info
        add_preview_widget(self._widgets)
        self._widgets['txt_lock_dataset'].hide()
        self._widgets['btn_select_lock_dataset'].hide()
        self._setup_display_options_table()

        self._add_button_box()

        self._setup_validators()
        self._setup_connections()

        self._setup_arc_to_polygon()

        self._draw_preview_display()

    def _add_main_layout(self):
        """Add the main layout."""
        self._widgets['dlg_v_layout'] = QVBoxLayout()

        # Group box around everything with vertical layout.
        self._widgets['grp_main'] = QGroupBox('')
        self._widgets['dlg_v_layout'].addWidget(self._widgets['grp_main'])
        self._widgets['dlg_v_layout'] = QVBoxLayout()
        self._widgets['grp_main'].setLayout(self._widgets['dlg_v_layout'])

        # Vertical layout for properties and tabs.
        self._widgets['props_h_layout'] = QHBoxLayout()
        self._widgets['dlg_v_layout'].addLayout(self._widgets['props_h_layout'])

        self._widgets['props_lables_v_layout'] = QVBoxLayout()
        self._widgets['props_h_layout'].addLayout(self._widgets['props_lables_v_layout'])
        self._widgets['props_widgets_v_layout'] = QVBoxLayout()
        self._widgets['props_h_layout'].addLayout(self._widgets['props_widgets_v_layout'])

        # Preview Layout
        self._widgets['plot_v_layout'] = QVBoxLayout()
        self._widgets['dlg_v_layout'].addLayout(self._widgets['plot_v_layout'])

        # Add Stretch
        self._widgets['dlg_v_layout'].addStretch()

        # Set Dialog Layout
        self.setLayout(self._widgets['dlg_v_layout'])

    def _add_props_widgets_and_labels(self):
        """Add labels and widgets for the arc data widgets."""
        # Name
        self._widgets['lbl_prop_name'] = QLabel('Name:')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_name'])
        self._widgets['props_edt_name'] = QLineEdit()
        self._widgets['props_edt_name'].setText(str(self._data['arc_name'].item()))
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['props_edt_name'])

        # Insert Feature
        self._widgets['lbl_prop_checkbox_insert'] = QLabel('')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_checkbox_insert'])
        self._widgets['tog_insert'] = QCheckBox('Insert feature')
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['tog_insert'])
        if int(self._data['insert_feature'].item()) == 1:
            self._widgets['tog_insert'].setChecked(True)

        # Crest Elevation
        self._widgets['lbl_prop_crest_elevation'] = QLabel(f'Crest elevation {self._units_str}:')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_crest_elevation'])
        self._widgets['edt_crest_elevation'] = QLineEdit()
        self._widgets['edt_crest_elevation'].setText(str(self._data['elevation'].item()))
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['edt_crest_elevation'])

        # Use Slope
        self._widgets['lbl_checkbox_empty_slope'] = QLabel('')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_checkbox_empty_slope'])
        self._widgets['tog_slope'] = QCheckBox('Use slope')
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['tog_slope'])
        if 'use_slope' in self._data and int(self._data['use_slope'].item()) == 1:
            self._widgets['tog_slope'].setChecked(True)

        # Side Slope
        side_slope = self._data['side_slope'].item() if 'side_slope' in self._data else 1.0
        self._widgets['lbl_side_slope'] = QLabel(f'Slope (Rise/Run {self._units_str}):')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_side_slope'])
        self._widgets['edt_side_slope'] = QLineEdit()
        self._widgets['edt_side_slope'].setText(str(side_slope))
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['edt_side_slope'])
        if not self._widgets['tog_slope'].isChecked():
            self._widgets['lbl_side_slope'].hide()
            self._widgets['edt_side_slope'].hide()

        # Max Slope Distance
        max_slope_distance = self._data['max_slope_distance'].item() \
            if 'max_slope_distance' in self._data else 5.0
        self._widgets['lbl_prop_max_slope_distance'] = QLabel(f'Max slope distance {self._units_str}:')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_max_slope_distance'])
        self._widgets['edt_max_slope_distance'] = QLineEdit()
        self._widgets['edt_max_slope_distance'].setText(str(max_slope_distance))
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['edt_max_slope_distance'])
        if not self._widgets['tog_slope'].isChecked():
            self._widgets['lbl_prop_max_slope_distance'].hide()
            self._widgets['edt_max_slope_distance'].hide()

        # Top Width
        self._widgets['lbl_prop_top_width'] = QLabel(f'Top width {self._units_str}:')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_top_width'])
        self._widgets['edt_top_width'] = QLineEdit()
        self._widgets['edt_top_width'].setText(str(self._data['top_width'].item()))
        self._widgets['props_widgets_v_layout'].addWidget(self._widgets['edt_top_width'])

        # Transition Distance
        self._widgets['lbl_prop_transition_distance'] = QLabel('Maximum transition distance:')
        self._widgets['props_lables_v_layout'].addWidget(self._widgets['lbl_prop_transition_distance'])
        self._widgets['horiz_layout_transition'] = QHBoxLayout()
        self._widgets['props_widgets_v_layout'].addLayout(self._widgets['horiz_layout_transition'])
        self._widgets['cbx_transition_method'] = QComboBox()
        for idx, text in consts.EWN_TRANSITION_METHOD.items():
            if text == 'Specified distance':
                text = f'{text} {self._units_str}'
            self._widgets['cbx_transition_method'].addItem(text, idx)
        self._widgets['cbx_transition_method'].setCurrentIndex(
            self._data.transition_method.item() if 'transition_method' in
            self._data else consts.TRANSITION_METHOD_FACTOR
        )
        self._widgets['horiz_layout_transition'].addWidget(self._widgets['cbx_transition_method'])
        self._widgets['edt_transition'] = QLineEdit()
        self._widgets['edt_transition'].setText(
            str(self._data.transition_distance.item() if 'transition_distance' in self._data else 0.5)
        )
        self._widgets['horiz_layout_transition'].addWidget(self._widgets['edt_transition'])

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['dlg_v_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def _setup_validators(self):
        """Setup the validators for the widgets."""
        self._dbl_validator = QxDoubleValidator(parent=self)
        self._greater_than_zero_validator = QxDoubleValidator(parent=self)
        self._pos_validator = QxDoubleValidator(parent=self)
        self._greater_than_zero_validator.setBottom(0.1)
        self._change_limit_validator = QxDoubleValidator(parent=self)
        self._positive_validator = QxDoubleValidator(parent=self)
        self._positive_validator.setBottom(0.0)
        self._widgets['edt_crest_elevation'].setValidator(self._dbl_validator)
        self._widgets['edt_side_slope'].setValidator(self._greater_than_zero_validator)
        self._widgets['edt_max_slope_distance'].setValidator(self._greater_than_zero_validator)
        self._widgets['edt_transition'].setValidator(self._pos_validator)
        self._widgets['edt_top_width'].setValidator(self._positive_validator)
        self._widgets['edt_change_limit'].setValidator(self._change_limit_validator)

    def _setup_connections(self):
        """Setup the connections between widgets."""
        self._widgets['btn_select_geom'].clicked.connect(self._on_select_geom)
        self._widgets['btn_insert_feature'].clicked.connect(self._on_generate_preview)
        self._widgets['edt_top_width'].editingFinished.connect(self._on_edit_finished_top_width)
        self._widgets['tog_slope'].stateChanged.connect(self._on_slope_toggle)
        self._widgets['cbx_transition_method'].currentIndexChanged.connect(self._on_transition_method_change)
        self._widgets['edt_transition'].editingFinished.connect(self._on_edit_finished_transition)

    def _setup_arc_to_polygon(self):
        self._buffered_arc = Polygon(
            runner_util.get_polygon_from_arc(
                self._dlg_data['arc'], float(self._widgets['edt_top_width'].text()), self._dlg_data['cov_geom'],
                float(self._widgets['edt_crest_elevation'].text())
            )
        )

        if len(self._buffered_arc.exterior.coords) == 0:
            msg = 'The arc with the specified top width is to concave to create a polygon.'
            msg += ' Please set a different top width.'
            msg_box = QMessageBox(QMessageBox.Warning, 'SMS', msg, QMessageBox.Ok, self)
            msg_box.exec()
            self._buffered_arc = None

    def _background_check_change(self):
        self._display_map = self._widgets['background_map_check'].checkState() == Qt.Checked
        self._draw_preview_display()

    def _setup_display_options_table(self):
        # Display Options
        self._widgets['grp_display'] = QGroupBox('Display Options')
        self._widgets['v_layout_display_options'] = QVBoxLayout()
        self._widgets['h_layout_display_options'] = QHBoxLayout()
        self._widgets['grp_display'].setLayout(self._widgets['v_layout_display_options'])

        # Background Map Checkbox
        self._widgets['background_map_check'] = QCheckBox('Background Map')
        self._widgets['background_map_check'].setCheckState(Qt.Checked if self._display_map else Qt.Unchecked)
        if self._is_local:
            self._widgets['background_map_check'].setCheckState(Qt.Unchecked)
            self._widgets['background_map_check'].setEnabled(False)
        self._widgets['v_layout_display_options'].addWidget(self._widgets['background_map_check'])
        self._widgets['background_map_check'].stateChanged.connect(self._background_check_change)

        # Create the Table
        self._widgets['display_options_table'] = QTableWidget()
        self._widgets['display_options_table'].setMinimumHeight(250)
        self._widgets['display_options_table'].setColumnCount(5)
        self._widgets['display_options_table'].setRowCount(len(self._order_and_color))

        # Set the Headers
        self._widgets['display_options_table'].setHorizontalHeaderLabels(["Priority", "", "Name", "Color", "Weight"])

        for row, display_option in enumerate(self._order_and_color):
            self._add_display_option_to_display_option_table(display_option, row)
        self._widgets['display_options_table'].setColumnHidden(0, True)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._widgets['display_options_table'].resizeColumnsToContents()

        # Add to Dialog
        self._widgets['h_layout_display_options'].addWidget(self._widgets['display_options_table'])
        self._widgets['v_layout_display_options'].addLayout(self._widgets['h_layout_display_options'])

        # Priority Buttons
        self._widgets['h_layout_display_options_arrows'] = QHBoxLayout()
        self._widgets['display_up_arrow'] = QPushButton()
        pixmapi_up = QStyle.SP_ArrowUp
        self._widgets['display_up_arrow'].setIcon(self.style().standardIcon(pixmapi_up))
        self._widgets['h_layout_display_options_arrows'].addWidget(self._widgets['display_up_arrow'])
        self._widgets['display_down_arrow'] = QPushButton()
        pixmapi_down = QStyle.SP_ArrowDown
        self._widgets['display_down_arrow'].setIcon(self.style().standardIcon(pixmapi_down))
        self._widgets['h_layout_display_options_arrows'].addWidget(self._widgets['display_down_arrow'])
        self._widgets['h_layout_display_options_arrows'].addStretch()
        self._widgets['v_layout_display_options'].addLayout(self._widgets['h_layout_display_options_arrows'])
        self._widgets['plot_v_layout'].addWidget(self._widgets['grp_display'])

    def _add_display_option_to_display_option_table(self, display_option, row):
        # Priority
        self._widgets['display_options_table'].setItem(
            row, Columns.PRIORITY, QTableWidgetItem(str(display_option.priority))
        )
        # Check
        check_item_check = QCheckBox()
        # check_item.setFlags(Qt.ItemIsUserCheckable | Qt.ItemIsEnabled)
        check_item_check.setCheckState(Qt.Checked if display_option.visible else Qt.Unchecked)
        check_item_check.clicked.connect(self._display_visible_change)
        self._widgets['display_options_table'].setCellWidget(row, Columns.CHECK, check_item_check)
        self._widgets['display_options_table'].cellWidget(row, Columns.CHECK).stateChanged.connect(
            self._display_options_changed
        )
        # Name
        name_item = QTableWidgetItem(str(display_option.display_name))
        self._widgets['display_options_table'].setItem(row, Columns.NAME, name_item)
        # Color
        color_button = QPushButton()
        color_button.setStyleSheet(f'background-color: {display_option.color}')
        color_button.clicked.connect(self._display_color_change)
        self._widgets['display_options_table'].setCellWidget(row, Columns.COLOR, color_button)
        self._widgets['display_options_table'].setItem(row, Columns.COLOR, QTableWidgetItem(str(display_option.color)))
        # Weight
        weight_spinner = QDoubleSpinBox()
        weight_spinner.setMaximum(5.0)
        weight_spinner.setMinimum(0.1)
        weight_spinner.setSingleStep(0.1)
        weight_spinner.setValue(display_option.weight)
        weight_spinner.valueChanged.connect(self._display_weight_changed)
        self._widgets['display_options_table'].setCellWidget(row, Columns.WEIGHT, weight_spinner)
        self._widgets['display_options_table'].setItem(
            row, Columns.WEIGHT, QTableWidgetItem(str(display_option.weight))
        )

    def _display_visible_change(self):
        current_row = self._widgets['display_options_table'].currentRow()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        checkbox = self._widgets['display_options_table'].cellWidget(current_row, Columns.CHECK)
        display_option.visible = checkbox.checkState() == Qt.Checked
        self._display_options_changed()

    def _display_weight_changed(self):
        current_row = self._widgets['display_options_table'].currentRow()
        spinner = self._widgets['display_options_table'].cellWidget(current_row, Columns.WEIGHT)
        new_weight = spinner.value()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        display_option.weight = float(new_weight)
        self._widgets['display_options_table'].item(current_row, Columns.WEIGHT).setText(str(new_weight))
        self._display_options_changed()

    def _display_color_change(self):
        color = QColorDialog.getColor()
        current_row = self._widgets['display_options_table'].currentRow()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        display_option.color = color.name()
        button = self._widgets['display_options_table'].cellWidget(current_row, Columns.COLOR)
        button.setStyleSheet(f'background-color: {display_option.color}')
        self._widgets['display_options_table'].item(current_row, Columns.COLOR).setText(display_option.color)
        self._display_options_changed()

    def _display_options_changed(self):
        self._draw_preview_display()

    def _raise_display_priority(self):
        current_row = self._widgets['display_options_table'].currentRow()
        if current_row <= 0:
            return
        new_row = current_row - 1
        current_priority = self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).text()
        new_priority = self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).text()
        current_item = self._order_and_color[len(self._order_and_color) - 1 - current_row]
        other_item = self._order_and_color[len(self._order_and_color) - 1 - new_row]
        current_item.priority = int(new_priority)
        other_item.priority = int(current_priority)
        self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).setText(new_priority)
        self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).setText(current_priority)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._display_options_changed()

    def _lower_display_priority(self):
        current_row = self._widgets['display_options_table'].currentRow()
        if current_row == -1 or current_row == len(self._order_and_color) - 1:
            return
        new_row = current_row + 1
        current_priority = self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).text()
        new_priority = self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).text()
        current_item = self._order_and_color[len(self._order_and_color) - 1 - current_row]
        other_item = self._order_and_color[len(self._order_and_color) - 1 - new_row]
        current_item.priority = int(new_priority)
        other_item.priority = int(current_priority)
        self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).setText(new_priority)
        self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).setText(current_priority)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._display_options_changed()

    def _on_slope_toggle(self):
        if self._widgets['tog_slope'].isChecked():
            self._widgets['lbl_side_slope'].show()
            self._widgets['edt_side_slope'].show()
            self._widgets['lbl_prop_max_slope_distance'].show()
            self._widgets['edt_max_slope_distance'].show()
        else:
            self._widgets['lbl_side_slope'].hide()
            self._widgets['edt_side_slope'].hide()
            self._widgets['lbl_prop_max_slope_distance'].hide()
            self._widgets['edt_max_slope_distance'].hide()

    def _on_edit_finished_top_width(self):
        """Called when the user is done editing the top width."""
        self.setCursor(Qt.WaitCursor)
        self._setup_arc_to_polygon()
        self._draw_preview_display()
        self.unsetCursor()

    def _on_transition_method_change(self, index):
        """Called when the user changes the transition method."""
        self.setCursor(Qt.WaitCursor)
        if self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_POLYGON:
            if 'transition_poly_pts' not in self._dlg_data:
                self._widgets['cbx_transition_method'].setCurrentIndex(0)
                txt = consts.EWN_TRANSITION_METHOD[consts.TRANSITION_METHOD_POLYGON]
                msg = f'Unable to change "Maximum transition distance" option to {txt!r}.\n' \
                      f'No transition polygon exists that contains the current polygon.'
                msg_box = QMessageBox(QMessageBox.Warning, 'SMS', msg, QMessageBox.Ok, self)
                msg_box.exec()
        if self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_FACTOR:
            self._widgets['edt_transition'].setText(str(self._initial_transition_factor))
        elif self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_DISTANCE:
            self._widgets['edt_transition'].setText(str(self._initial_transition_distance))
        self._on_edit_finished_transition()
        self.unsetCursor()

    def _on_edit_finished_transition(self):
        """Called when the user is done editing the transition distance."""
        transition = self._widgets['cbx_transition_method'].currentText()
        self._widgets['edt_transition'].setEnabled(True)
        if transition == 'Transition polygon':
            self._widgets['edt_transition'].setEnabled(False)
        self._get_polys_for_plot()
        self._draw_preview_display()

    def _get_polys_for_plot(self):
        """Sets member variables for polygons to draw in the plot."""
        if 'arc_pts' not in self._dlg_data:
            return
        poly_pts = self._dlg_data['arc_pts']
        self._plot_transition_pts = []
        method = self._widgets['cbx_transition_method'].currentIndex()
        if method == consts.TRANSITION_METHOD_POLYGON and 'transition_poly_pts' in self._dlg_data:
            trans_poly = self._dlg_data['transition_poly_pts']
            xcoord, ycoord, _ = zip(*trans_poly)
            self._plot_transition_pts = [(pt[0], pt[1]) for pt in trans_poly]
        else:
            xcoord, ycoord, _ = zip(*poly_pts)
        self._box_min = [min(xcoord), min(ycoord)]
        self._box_max = [max(xcoord), max(ycoord)]

        dist = 0.0
        if method == consts.TRANSITION_METHOD_FACTOR:
            dist = float(self._widgets['edt_transition'].text())
            length = math.sqrt((self._box_max[0] - self._box_min[0])**2 + (self._box_max[1] - self._box_min[1])**2)
            dist = dist * length
        elif method == consts.TRANSITION_METHOD_DISTANCE:
            dist = float(self._widgets['edt_transition'].text())
            if self._is_geographic:
                _, _, center_y = self._length_and_center_from_box()
                dist = meters_to_decimal_degrees(dist, center_y)
        self._box_min = [self._box_min[0] - dist, self._box_min[1] - dist]
        self._box_max = [self._box_max[0] + dist, self._box_max[1] + dist]

        if not self._plot_transition_pts:
            self._plot_transition_pts = [
                (self._box_min[0], self._box_min[1]), (self._box_max[0], self._box_min[1]),
                (self._box_max[0], self._box_max[1]), (self._box_min[0], self._box_max[1]),
                (self._box_min[0], self._box_min[1])
            ]
        # must be in longitude, latitude
        self._clipped_ugrid = None

    def _on_select_geom(self):
        """Select a ugrid or mesh."""
        problem = self._check_project_explorer()
        if problem:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS',
                'No mesh or ugrid defined. Preview is only supported for meshes and ugrids.', QMessageBox.Ok, self
            )
            msg_box.exec()
            return
        selector_dlg = TreeItemSelectorDlg(
            title='Select Source Geometry',
            target_type='',
            pe_tree=self._pe_tree,
            parent=self,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS']
        )

        dlg_ran = selector_dlg.exec()
        if not dlg_ran:
            return

        source_geom_uuid = selector_dlg.get_selected_item_uuid()
        if not source_geom_uuid:  # No geometry selected
            self._reset_selected_geom()
            return

        self._geom_uuid = source_geom_uuid
        tree_path = tree_util.build_tree_path(self._pe_tree, source_geom_uuid)

        self._widgets['txt_geom'].setText(tree_path)
        self.setCursor(Qt.WaitCursor)

        if source_geom_uuid not in self._ugrids:
            self._dlg_data['geom_uuid'] = self._geom_uuid
            preview_get_geometry_feedback(self)
        self._dlg_data['ugrid'] = self._ugrids[source_geom_uuid]
        self.is_cartesian = self.cogrid.grid_type in [GridType.rectilinear_2d, GridType.rectilinear_3d]
        self.is_quadtree = self.cogrid.grid_type in [GridType.quadtree_2d, GridType.quadtree_3d]

        if self.is_cartesian or self.is_quadtree:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Preview only supports meshes and non quadtree ugrids.', QMessageBox.Ok,
                self
            )
            msg_box.exec()
            self._reset_selected_geom()
            self.unsetCursor()
            return

        self._subset = self._subsets[source_geom_uuid]
        self._clipped_ugrid = None
        if 'ugrid' in self._dlg_data:
            if self._dlg_data['ugrid'].dimension_counts[3] > 0:
                self._reset_selected_geom()
                msg_box = QMessageBox(
                    QMessageBox.Warning, 'SMS', 'Selected Mesh/UGrid is 3D. Only 2D is supported.', QMessageBox.Ok, self
                )
                msg_box.exec()
                self.unsetCursor()
                return
            self._draw_preview_display()

    def _reset_selected_geom(self):
        self._widgets['txt_geom'].setText('(none selected)')
        self._dlg_data['ugrid'] = None
        self._geom_uuid = None
        self._dlg_data['geom_uuid'] = None
        self._clipped_ugrid = None
        self._subset = None
        self._widgets['txt_lock_dataset'].setEnabled(False)
        self._widgets['btn_select_lock_dataset'].setEnabled(False)

    def _draw_preview_display(self):
        """Draw the preview display."""
        # Get arc points
        self._get_polys_for_plot()
        if self._arc_pts is None:
            self._get_arcs_for_plot()

        buffered_arc_pts = None
        if self._buffered_arc is not None:
            buffered_arc_pts = self._transform_points(list(self._buffered_arc.exterior.coords))

        # Get clipped ugrid
        if 'ugrid' in self._dlg_data and self._dlg_data['ugrid'] is not None and self._clipped_ugrid is None:
            box = [(self._box_min[0], self._box_min[1]), (self._box_max[0], self._box_max[1])]
            self._clipped_ugrid = self._subset.subset_from_box(box)
            self._clipped_ugrid = self._subset.subset_from_box(box)

        # Create the map
        folium_map = folium.Map(tiles='')
        if not self._is_local and self._display_map:
            url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}'
            tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI Street Map', attr='ESRI')
            tl.add_to(folium_map)
            url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
            tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI World Imagery', attr='ESRI')
            tl.add_to(folium_map)
            folium.LayerControl().add_to(folium_map)

        arc_pts = self._transform_points(self._arc_pts)
        self._order_and_color.sort(key=lambda x: x.priority)

        for display_item in self._order_and_color:
            if display_item.name == 'dune_arc' and display_item.visible:
                # Draw the arc
                arc_layer = folium.vector_layers.PolyLine(
                    locations=flip_long_lat(arc_pts), weight=5, color='black', tooltip='arc'
                )
                arc_layer.add_to(folium_map)

            if display_item.name == 'dune_polygon' and (
                buffered_arc_pts is not None or float(self._widgets['edt_top_width'].text()) == 0.0
            ):
                # Draw the buffered arc
                if float(self._widgets['edt_top_width'].text()) == 0.0:
                    buffer_poly = folium.vector_layers.PolyLine(locations=flip_long_lat(arc_pts), weight=3, color='red')
                else:
                    buffer_poly = folium.vector_layers.PolyLine(
                        locations=flip_long_lat(buffered_arc_pts), weight=3, color='red'
                    )
                if display_item.visible:
                    buffer_poly.add_to(folium_map)

            # Draw transition polygon
            if self._insert_transition_polygon is not None:
                transition_poly_pts = [(pt[0], pt[1]) for pt in self._insert_transition_polygon]
                tip = 'Mesh transition polygon'
                transition_poly_plot_pts = self._transform_points(transition_poly_pts)
                folium.vector_layers.PolyLine(
                    locations=flip_long_lat(transition_poly_plot_pts), weight=6, color='purple', tooltip=tip
                ).add_to(folium_map)

            if display_item.name == 'input' and display_item.visible:
                # Input Mesh
                self._add_ugrid_edges_to_plot(
                    folium_map,
                    display_item.display_name,
                    self._clipped_ugrid,
                    display_item.color,
                    weight=display_item.weight
                )

            if display_item.name == 'output' and display_item.visible:
                # Draw new mesh
                self._add_ugrid_edges_to_plot(
                    folium_map,
                    display_item.display_name,
                    self._insert_ugrid,
                    display_item.color,
                    weight=display_item.weight
                )

            if display_item.name == 'transition' and display_item.visible:
                # Draw transition polygon
                transition_plot_pts = self._transform_points(self._plot_transition_pts)
                trans_poly = folium.vector_layers.PolyLine(
                    locations=flip_long_lat(transition_plot_pts + [transition_plot_pts[0]]),
                    weight=display_item.weight,
                    color=display_item.color,
                    tooltip='Maximum transition polygon'
                )
                if display_item.visible:  # Do this here so we can get the bounds.
                    trans_poly.add_to(folium_map)
                bounds = trans_poly.get_bounds()
                folium_map.fit_bounds(bounds)

        # Use io, rather than a file on disk, to store the map as html in memory
        folium_map.save(self._url_file)

        if self._mesh_viewer is None:
            self._mesh_viewer = MeshViewerDialog(parent=self, map_url=self._url_file, folium_map=folium_map)
            self._mesh_viewer.show()
        self._mesh_viewer.update_display()
        self.unsetCursor()

    def _add_ugrid_edges_to_plot(self, folium_map, name, ugrid, color, weight=1):
        """Draw the ugrid on the map.

        Args:
            folium_map (:obj:`folium.Map`): the map control
            name (:obj:`str`): name of this layer
            ugrid (:obj:`xms.grid.UGrid`): ugrid
            color (:obj:`str`): color to draw the grid
            weight (:obj:`int`): weight of line being drawn
        """
        if ugrid is None:
            return
        # Get the bounds of the grid points passed in
        grid_points = ugrid.locations

        # Hash the edges
        cell_count = ugrid.cell_count
        edge_set = set()  # Use this set to remove duplicate edges
        for cell_idx in range(cell_count):
            cell_edges = ugrid.get_cell_edges(cell_idx)
            for edge in cell_edges:
                if edge[0] > edge[1]:
                    edge_set.add((edge[1], edge[0]))
                else:
                    edge_set.add((edge[0], edge[1]))

        # fg = folium.map.FeatureGroup(name).add_to(folium_map)
        ml_list = []
        for edge in edge_set:
            pts = [
                (grid_points[edge[0]][0], grid_points[edge[0]][1]), (grid_points[edge[1]][0], grid_points[edge[1]][1])
            ]
            pts = self._transform_points(pts)
            pts = flip_long_lat(pts)
            ml_list.append([pts[0], pts[1]])
        if len(ml_list) > 0:
            folium.vector_layers.PolyLine(locations=ml_list, weight=weight, color=f'{color}').add_to(folium_map)

    def _on_generate_preview(self):
        """Generate preview of new mesh."""
        self._save_data()
        if self._clipped_ugrid is None:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Must select a mesh/ugrid to insert a feature.', QMessageBox.Ok, self
            )
            msg_box.exec()
            return

        self._insert_ugrid = None
        arc_data = self._get_arc_data()
        arc_data['top_width'] = float(self._widgets['edt_top_width'].text())
        arc_data['transition_distance'] = float(self._widgets['edt_transition'].text())

        bias = 1.0 - float(self._widgets['edt_change_limit'].text())
        ewn = EwnProcess(
            self._clipped_ugrid,
            self._is_geographic,
            False,  # is_levee
            bias,  # bias
            lock_dataset=None,
            is_cartesian=self.is_cartesian
        )
        ewn.show_transition_warning = False
        ewn.set_polygon_data(arc_data)
        # ewn.set_polygon_data(poly_data)
        preview_ewn_insert_feedback(self, ewn)
        if ewn.stitched_ugrid is not None:
            processed_grid = ewn.stitched_ugrid
            if arc_data['arc_pts'] is not None:
                processed_grid = runner_util.post_process_ugrid_for_line(arc_data, processed_grid)
            self._insert_ugrid = processed_grid
            self._insert_transition_polygon = ewn.transition_polygon
            self._draw_preview_display()

    def _transform_points(self, points):
        """Transforms the points from their current CRS (coordinate reference system) to the new one.

        See https://gis.stackexchange.com/questions/226200/how-to-call-gdaltransform-with-its-input

        Args:
            points (:obj:`list`): List of points.

        Returns:
            new_points (:obj:`list`): The transformed points.
        """
        if not self._transform and not self._is_local:
            return points

        # Transform each point
        new_points = []
        for point in points:
            if self._transform:
                # new_points = gdal_utils.transform_points([(p[0], p[1]) for p in points], self._transform)
                coords = gdal_utils.transform_point(point[0], point[1], self._transform)
                new_points.append([coords[0], coords[1]])
            else:
                x = meters_to_decimal_degrees(point[0], 0.0)
                y = meters_to_decimal_degrees(point[1], 0.0)
                new_points.append([x, y])

        return new_points

    def _get_arcs_for_plot(self):
        """Get the arcs for plotting."""
        if 'arc_pts' not in self._dlg_data:
            return
        arc_pts = self._dlg_data['arc_pts']

        xcoords, ycoords, _ = zip(*arc_pts)
        self._box_min = [min(xcoords), min(ycoords)]
        self._box_max = [max(xcoords), max(ycoords)]

        self._arc_pts = [(pt[0], pt[1]) for pt in arc_pts]

    def _check_project_explorer(self):
        """Check if the project explorer is available."""
        if self._pe_tree is None:
            return 'Unable to retrieve the SMS project explorer tree.'

        for root_child in self._pe_tree.children:  # Loop through children of the project tree root
            if root_child.children and root_child.name in ['Mesh Data', 'UGrid Data']:
                # Found either the 2D Mesh root or the UGrid root and it has children. Assume there is at least
                # one valid geometry.
                return ''
        return 'A 2D Mesh or UGrid is required to run this command.'

    def _get_arc_data(self):
        arc_inputs = runner_util.get_ewn_arc_input(self._dlg_data['cov_geom'], self._dlg_data['ewn_comp'])
        for arc in arc_inputs:
            if arc['polygon_id'] == self._dlg_data['arc_id']:
                return arc
        return None

    def _create_transition_coverage(self):
        new_transition_cov = Coverage(name='EWN Transition Polygon', uuid=str(uuid.uuid4()))
        pts = []
        cur_pt_id = 1
        for pt in self._insert_transition_polygon[:-1]:
            d_pt = Point(pt[0], pt[1], 0.0)
            d_pt.id = cur_pt_id
            cur_pt_id += 1
            pts.append(d_pt)
        new_transition_arc = Arc(start_node=pts[0], end_node=pts[0], vertices=pts[1:], feature_id=1)
        new_transition_polygon = Poly()
        new_transition_polygon.id = 1
        new_transition_polygon.set_arcs([new_transition_arc])
        new_transition_cov.arcs = [new_transition_arc]
        new_transition_cov.polygons = [new_transition_polygon]
        new_transition_cov.complete()
        self.transition_poly_cov = new_transition_cov

    def _save_data(self):
        """Save data to persistent storage."""
        self._data['arc_name'] = self._widgets['props_edt_name'].text()
        self._data['insert_feature'] = 1 if self._widgets['tog_insert'].isChecked() else 0
        self._data['elevation'] = float(self._widgets['edt_crest_elevation'].text())
        self._data['use_slope'] = 1 if self._widgets['tog_slope'].isChecked() else 0
        self._data['side_slope'] = float(self._widgets['edt_side_slope'].text())
        self._data['max_slope_distance'] = float(self._widgets['edt_max_slope_distance'].text())
        self._data['slope'] = float(self._widgets['edt_side_slope'].text())
        self._data['max_distance'] = float(self._widgets['edt_max_slope_distance'].text())
        self._data['top_width'] = float(self._widgets['edt_top_width'].text())
        self._data['transition_method'] = self._widgets['cbx_transition_method'].currentIndex()
        self._data['transition_distance'] = float(self._widgets['edt_transition'].text())
        if self._widgets['tog_add_to_xms'].isChecked() and self._insert_ugrid is not None:
            self.new_ugrid = self._subset.stitch_grid(self._insert_ugrid)
        if self._widgets['tog_add_transition_cov_to_xms'].isChecked() and self._insert_ugrid is not None:
            self._create_transition_coverage()

    def accept(self):
        """Save data to persistent storage on OK."""
        self._save_data()
        if self._mesh_viewer is not None:
            self._mesh_viewer.close()
        super().accept()

    def reject(self):
        """Cancel command."""
        if self._mesh_viewer is not None:
            self._mesh_viewer.close()
        super().reject()

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)
