"""The Assign EWN Feature polygon attribute dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
from enum import IntEnum
import math
import uuid
import webbrowser

# 2. Third party modules
import folium
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QCheckBox, QColorDialog, QComboBox, QDialogButtonBox, QDoubleSpinBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit,
    QMessageBox, QPushButton, QStyle, QTableWidget, QTableWidgetItem, QTabWidget, QTextEdit, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import GridType
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon as Poly
from xms.gdal.utilities import gdal_utils
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts
from xms.ewn.gui.mesh_viewer_dialog import MeshViewerDialog
from xms.ewn.gui.tool_input_dialog import is_scalar_nodes_if_dset
from xms.ewn.gui.widgets.preview_widget import add_preview_widget
from xms.ewn.tools.ewn_process import decimal_degrees_to_meters, EwnProcess, meters_to_decimal_degrees
from xms.ewn.tools.runners.ewn_preview_runner import preview_ewn_insert_feedback, preview_get_geometry_feedback


@dataclass
class DisplayItem:
    """DisplayItem data class."""
    name: str
    display_name: str
    color: str
    priority: str
    weight: float
    visible: bool = False
    enabled: bool = True


class Columns(IntEnum):
    """Column enum for display options."""
    PRIORITY = 0
    CHECK = 1
    NAME = 2
    COLOR = 3
    WEIGHT = 4


def is_applicable_geom(item):
    """A method to determine if the item is a mesh or ugrid."""
    if item.item_typename in ['TI_MESH2D', 'TI_UGRID_SMS']:
        return True
    return False


class AssignEwnPolygonDialog(XmsDlg):
    """A dialog for assigning materials to polygons."""
    def __init__(self, parent, dlg_data):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            dlg_data (:obj:`dict`): data for dialog
        """
        super().__init__(parent, 'xms.ewn.gui.assign_polygon_feature_dialog')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:EWN'
        self._multi_select_warning = dlg_data['atts_mismatch']
        self._dbl_validator = None
        self._pos_validator = None
        self._greater_than_zero_validator = None
        self._change_limit_validator = None
        self._data = dlg_data['poly_data']
        self._vert_units = dlg_data['vertical_units']
        self._meta_data = dlg_data['df']
        self._dlg_data = dlg_data
        self._classifications = None
        self._current_classification = 'Unassigned'
        self._units_str = '(m)' if self._vert_units == 'METERS' else '(ft)'
        self._first_time_reload = True
        self._clipped_ugrid = None
        self._insert_ugrid = None
        self._insert_transition_polygon = None
        self._plot_transition_pts = None
        self._box_min = None
        self._box_max = None
        self._plot_poly_pts = None
        self._url_file = dlg_data['map_html']
        self._is_geographic = self._dlg_data['projection'].coordinate_system == 'GEOGRAPHIC'
        self._is_local = self._dlg_data['projection'].coordinate_system == 'None'
        self._subset = None
        self._pe_tree = self._dlg_data['query'].project_tree
        self._geom_uuid = None
        self._lock_dataset_uuid = None
        self._clipped_lock_dataset = None
        self._locked_node_locations = None
        self.cogrid = None
        self._ugrids = {}
        self._subsets = {}
        self._transform = None
        self._mesh_viewer = None
        self._display_map = True
        self.is_cartesian = False
        self.is_quadtree = False

        self.new_ugrid = None
        self.transition_poly_cov = None

        self._order_and_color = [
            DisplayItem(
                name='output', color='grey', priority=0, display_name='Output Mesh/Grid', weight=1, visible=True
            ),
            DisplayItem(name='lock_nodes', color='red', priority=1, display_name='Lock Nodes', weight=5, visible=True),
            DisplayItem(
                name='transition_polygon',
                color='magenta',
                priority=2,
                display_name='Transition Polygon',
                weight=4,
                visible=True
            ),
            DisplayItem(
                name='input', color='orange', priority=3, display_name='Input Mesh/Grid', weight=0.5, visible=True
            ),
            DisplayItem(
                name='transition_boundary',
                color='red',
                priority=4,
                display_name='Transition Boundary',
                weight=2,
                visible=True,
                enabled=True
            ),
            DisplayItem(
                name='ewn_polygon',
                color='#000000',
                priority=5,
                display_name='EWN Polygon',
                weight=2,
                visible=True,
                enabled=True
            ),
        ]

        if not self._is_geographic and not self._is_local:
            # Set up a transformation from input non-geographic to WGS 84
            src_wkt = self._dlg_data['projection'].well_known_text
            _, trg_wkt = gdal_utils.wkt_from_epsg(4326)
            self._transform = gdal_utils.get_coordinate_transformation(
                src_wkt, trg_wkt, set_traditional_axis_mapping=True
            )

        self._widgets = {}
        self._setup_ui()
        # initial factor and distance for transition method
        self._initial_transition_factor = 0.5
        self._initial_transition_distance = 0.5
        self._calc_init_transition_distance()
        if self._data.transition_method.item() == consts.TRANSITION_METHOD_FACTOR:
            self._initial_transition_factor = self._data.transition_distance.item()
            self._calc_init_transition_distance()
        elif self._data.transition_method.item() == consts.TRANSITION_METHOD_DISTANCE:
            self._initial_transition_distance = self._data.transition_distance.item()
            self._calc_init_transition_factor()

    def _calc_init_transition_distance(self):
        """Calculates the initial transition distance based on the initial transition factor."""
        length, _, center_y = self._length_and_center_from_box()
        if self._is_geographic:
            length = decimal_degrees_to_meters(length, center_y)
        self._initial_transition_distance = length * self._initial_transition_factor
        pass

    def _calc_init_transition_factor(self):
        """Calculates the initial transition factor based on the initial transition distance."""
        length, _, center_y = self._length_and_center_from_box()
        dd = self._initial_transition_distance
        if self._is_geographic:
            dd = meters_to_decimal_degrees(dd, center_y)
        self._initial_transition_factor = dd / length

    def _length_and_center_from_box(self):
        """Gets the length and the latitude from the calculated box.

        Returns:
            (:obj:`tuple(float, float, float)`): length of diagonal of box, x, y center of box
        """
        if self._box_min is None or self._box_max is None:
            return 0.0, 0.0, 0.0
        length = math.sqrt((self._box_max[0] - self._box_min[0])**2 + (self._box_max[1] - self._box_min[1])**2)
        center_x = (self._box_max[0] + self._box_min[0]) / 2
        center_y = (self._box_max[1] + self._box_min[1]) / 2
        return length, center_x, center_y

    def _setup_ui(self):
        """Sets up the dialog widgets."""
        self.setWindowTitle('EWN Polygon Properties')
        self._add_main_layout()
        self._add_polygon_data_labels()
        self._add_polygon_data_widgets()
        self._add_button_box()
        self._setup_validators()
        self._setup_connections()
        self._set_meta_data_text()
        self._dim_stuff()
        self._draw_polygon_plot()

    def _add_main_layout(self):
        """Add main layout to dialog."""
        self._widgets['grp_main'] = QGroupBox('')
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['main_vert_layout'].addWidget(self._widgets['grp_main'])
        self._widgets['grp_main_vert_layout'] = QVBoxLayout()
        self._widgets['grp_main'].setLayout(self._widgets['grp_main_vert_layout'])
        self._widgets['horiz_layout'] = QHBoxLayout()
        self._widgets['grp_main_vert_layout'].addLayout(self._widgets['horiz_layout'])
        self._widgets['left_label_layout'] = QVBoxLayout()
        self._widgets['horiz_layout'].addLayout(self._widgets['left_label_layout'])
        self._widgets['right_widget_layout'] = QVBoxLayout()
        self._widgets['horiz_layout'].addLayout(self._widgets['right_widget_layout'])
        self._widgets['horiz_layout'].addStretch()
        self.setLayout(self._widgets['main_vert_layout'])

    def _add_polygon_data_labels(self):
        """Add labels for the polygon data widgets."""
        self._widgets['lbl_name'] = QLabel('Name:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_name'])
        self._widgets['lbl_classification'] = QLabel('Classification:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_classification'])
        self._widgets['lbl_checkbox_empty'] = QLabel('')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_checkbox_empty'])
        self._widgets['lbl_manning_n'] = QLabel("Manning's N:")
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_manning_n'])
        self._widgets['lbl_elevation_method'] = QLabel(f'Elevation {self._units_str}:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_elevation_method'])
        self._widgets['lbl_checkbox_empty_slope'] = QLabel('')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_checkbox_empty_slope'])
        self._widgets['lbl_slope'] = QLabel(f'Slope (Rise/Run {self._units_str}):')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_slope'])
        self._widgets['lbl_max_slope_dist'] = QLabel(f'Maximum slope distance {self._units_str}:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_max_slope_dist'])
        self._widgets['lbl_max_transition'] = QLabel('Maximum transition distance:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_max_transition'])
        self._widgets['lbl_quadtree_refinement'] = QLabel(f'Quadtree target cell size {self._units_str}:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_quadtree_refinement'])

    def _add_polygon_data_widgets(self):
        """Add widgets for the polygon properties."""
        self._widgets['edt_name'] = QLineEdit()
        self._widgets['edt_name'].setText(self._data.polygon_name.item())
        self._widgets['right_widget_layout'].addWidget(self._widgets['edt_name'])

        self._widgets['cbx_classification'] = QComboBox()
        self._classifications = ['Unassigned'] + self._meta_data['Feature type/Region'].tolist()
        for idx, text in enumerate(self._classifications):
            self._widgets['cbx_classification'].addItem(text, idx)
        self._widgets['cbx_classification'].setCurrentIndex(self._data.classification.item())
        self._current_classification = self._widgets['cbx_classification'].currentText()
        self._widgets['right_widget_layout'].addWidget(self._widgets['cbx_classification'])

        self._widgets['tog_insert'] = QCheckBox('Insert feature')
        self._widgets['right_widget_layout'].addWidget(self._widgets['tog_insert'])
        if int(self._data.insert_feature.item()) == 1:
            self._widgets['tog_insert'].setChecked(True)

        self._widgets['edt_manning_n'] = QLineEdit()
        self._widgets['edt_manning_n'].setText(str(self._data.manning_n.item()))
        self._widgets['right_widget_layout'].addWidget(self._widgets['edt_manning_n'])

        self._widgets['horiz_layout_elevation'] = QHBoxLayout()
        self._widgets['right_widget_layout'].addLayout(self._widgets['horiz_layout_elevation'])

        self._widgets['cbx_elevation_method'] = QComboBox()
        for idx, text in consts.EWN_ELEVATION_METHOD.items():
            self._widgets['cbx_elevation_method'].addItem(text, idx)
        self._widgets['cbx_elevation_method'].setCurrentIndex(self._data.elevation_method.item())
        self._widgets['horiz_layout_elevation'].addWidget(self._widgets['cbx_elevation_method'])

        self._widgets['edt_elevation'] = QLineEdit()
        self._widgets['edt_elevation'].setText(str(self._data.elevation.item()))
        self._widgets['horiz_layout_elevation'].addWidget(self._widgets['edt_elevation'])

        self._widgets['tog_slope'] = QCheckBox('Specify slope')
        self._widgets['right_widget_layout'].addWidget(self._widgets['tog_slope'])
        if int(self._data.specify_slope.item()) == 1:
            self._widgets['tog_slope'].setChecked(True)

        self._widgets['edt_slope'] = QLineEdit()
        self._widgets['edt_slope'].setText(str(self._data.slope.item()))
        self._widgets['right_widget_layout'].addWidget(self._widgets['edt_slope'])

        self._widgets['edt_max_slope_dist'] = QLineEdit()
        self._widgets['edt_max_slope_dist'].setText(str(self._data.maximum_slope_distance.item()))
        self._widgets['right_widget_layout'].addWidget(self._widgets['edt_max_slope_dist'])

        self._widgets['horiz_layout_transition'] = QHBoxLayout()
        self._widgets['right_widget_layout'].addLayout(self._widgets['horiz_layout_transition'])

        self._widgets['cbx_transition_method'] = QComboBox()
        for idx, text in consts.EWN_TRANSITION_METHOD.items():
            if text == 'Specified distance':
                text = f'{text} {self._units_str}'
            self._widgets['cbx_transition_method'].addItem(text, idx)
        self._widgets['cbx_transition_method'].setCurrentIndex(self._data.transition_method.item())
        self._widgets['horiz_layout_transition'].addWidget(self._widgets['cbx_transition_method'])

        self._widgets['edt_transition'] = QLineEdit()
        self._widgets['edt_transition'].setText(str(self._data.transition_distance.item()))
        self._widgets['horiz_layout_transition'].addWidget(self._widgets['edt_transition'])

        self._widgets['edt_quadtree_refinement'] = QLineEdit()

        try:
            self._widgets['edt_quadtree_refinement'].setText(str(self._data.quadtree_refinement_length.item()))
        except AttributeError:
            self._widgets['edt_quadtree_refinement'].setText('10')

        self._widgets['right_widget_layout'].addWidget(self._widgets['edt_quadtree_refinement'])

        # Tab Widget
        self._widgets['tab'] = QTabWidget()
        self._widgets['grp_main_vert_layout'].addWidget(self._widgets['tab'])

        # Tab Metadata
        self._widgets['tab_metadata_widget'] = QWidget()
        self._widgets['tab'].addTab(self._widgets['tab_metadata_widget'], 'Metadata')
        self._widgets['metadata_v_layout'] = QVBoxLayout(self._widgets['tab_metadata_widget'])
        self._widgets['txt_edt_meta_data'] = QTextEdit()
        self._widgets['txt_edt_meta_data'].setReadOnly(True)
        self._widgets['metadata_v_layout'].addWidget(self._widgets['txt_edt_meta_data'])

        # Tab Preview
        self._widgets['tab_preview_widget'] = QWidget()
        self._widgets['tab'].addTab(self._widgets['tab_preview_widget'], 'Preview')
        self._widgets['plot_v_layout'] = QVBoxLayout(self._widgets['tab_preview_widget'])
        add_preview_widget(self._widgets)

        self._setup_display_options_table()
        self._widgets['plot_v_layout'].addStretch()

        if self._multi_select_warning:
            msg = 'The selected polygons are assigned different EWN feature properties. ' \
                  'Changes to the EWN feature properties will be applied to all selected polygons.'
            self._widgets['lbl_multi_select'] = QLabel(msg)
            self._widgets['lbl_multi_select'].setStyleSheet('font-weight: bold; color: red')
            self._widgets['grp_main_vert_layout'].addWidget(self._widgets['lbl_multi_select'])

    def _background_check_change(self):
        self._display_map = self._widgets['background_map_check'].checkState() == Qt.Checked
        self._draw_polygon_plot()

    def _setup_display_options_table(self):
        # Display Options
        self._widgets['grp_display'] = QGroupBox('Display Options')
        self._widgets['v_layout_display_options'] = QVBoxLayout()
        self._widgets['h_layout_display_options'] = QHBoxLayout()
        self._widgets['grp_display'].setLayout(self._widgets['v_layout_display_options'])

        # Background Map Checkbox
        self._widgets['background_map_check'] = QCheckBox('Background Map')
        self._widgets['background_map_check'].setCheckState(Qt.Checked if self._display_map else Qt.Unchecked)
        if self._is_local:
            self._widgets['background_map_check'].setCheckState(Qt.Unchecked)
            self._widgets['background_map_check'].setEnabled(False)
        self._widgets['v_layout_display_options'].addWidget(self._widgets['background_map_check'])
        self._widgets['background_map_check'].stateChanged.connect(self._background_check_change)

        # Create the Table
        self._widgets['display_options_table'] = QTableWidget()
        self._widgets['display_options_table'].setMinimumHeight(250)
        self._widgets['display_options_table'].setColumnCount(5)
        self._widgets['display_options_table'].setRowCount(len(self._order_and_color))

        # Set the Headers
        self._widgets['display_options_table'].setHorizontalHeaderLabels(["Priority", "", "Name", "Color", "Weight"])

        for row, display_option in enumerate(self._order_and_color):
            self._add_display_option_to_display_option_table(display_option, row)
        self._widgets['display_options_table'].setColumnHidden(0, True)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._widgets['display_options_table'].resizeColumnsToContents()

        # Add to Dialog
        self._widgets['h_layout_display_options'].addWidget(self._widgets['display_options_table'])
        self._widgets['v_layout_display_options'].addLayout(self._widgets['h_layout_display_options'])

        # Priority Buttons
        self._widgets['h_layout_display_options_arrows'] = QHBoxLayout()
        self._widgets['display_up_arrow'] = QPushButton()
        pixmapi_up = QStyle.SP_ArrowUp
        self._widgets['display_up_arrow'].setIcon(self.style().standardIcon(pixmapi_up))
        self._widgets['h_layout_display_options_arrows'].addWidget(self._widgets['display_up_arrow'])
        self._widgets['display_down_arrow'] = QPushButton()
        pixmapi_down = QStyle.SP_ArrowDown
        self._widgets['display_down_arrow'].setIcon(self.style().standardIcon(pixmapi_down))
        self._widgets['h_layout_display_options_arrows'].addWidget(self._widgets['display_down_arrow'])
        self._widgets['h_layout_display_options_arrows'].addStretch()
        self._widgets['v_layout_display_options'].addLayout(self._widgets['h_layout_display_options_arrows'])
        self._widgets['plot_v_layout'].addWidget(self._widgets['grp_display'])

    def _add_display_option_to_display_option_table(self, display_option, row):
        # Priority
        self._widgets['display_options_table'].setItem(
            row, Columns.PRIORITY, QTableWidgetItem(str(display_option.priority))
        )
        # Check
        check_item_check = QCheckBox()
        # check_item.setFlags(Qt.ItemIsUserCheckable | Qt.ItemIsEnabled)
        check_item_check.setCheckState(Qt.Checked if display_option.visible else Qt.Unchecked)
        check_item_check.clicked.connect(self._display_visible_change)
        self._widgets['display_options_table'].setCellWidget(row, Columns.CHECK, check_item_check)
        self._widgets['display_options_table'].cellWidget(row, Columns.CHECK).stateChanged.connect(
            self._display_options_changed
        )
        # Name
        name_item = QTableWidgetItem(str(display_option.display_name))
        self._widgets['display_options_table'].setItem(row, Columns.NAME, name_item)
        # Color
        color_button = QPushButton()
        color_button.setStyleSheet(f'background-color: {display_option.color}')
        color_button.clicked.connect(self._display_color_change)
        self._widgets['display_options_table'].setCellWidget(row, Columns.COLOR, color_button)
        self._widgets['display_options_table'].setItem(row, Columns.COLOR, QTableWidgetItem(str(display_option.color)))
        # Weight
        weight_spinner = QDoubleSpinBox()
        weight_spinner.setMaximum(5.0)
        weight_spinner.setMinimum(0.1)
        weight_spinner.setSingleStep(0.1)
        weight_spinner.setValue(display_option.weight)
        weight_spinner.valueChanged.connect(self._display_weight_changed)
        self._widgets['display_options_table'].setCellWidget(row, Columns.WEIGHT, weight_spinner)
        self._widgets['display_options_table'].setItem(
            row, Columns.WEIGHT, QTableWidgetItem(str(display_option.weight))
        )

    def _display_visible_change(self):
        current_row = self._widgets['display_options_table'].currentRow()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        checkbox = self._widgets['display_options_table'].cellWidget(current_row, Columns.CHECK)
        display_option.visible = checkbox.checkState() == Qt.Checked
        self._display_options_changed()

    def _display_weight_changed(self):
        current_row = self._widgets['display_options_table'].currentRow()
        spinner = self._widgets['display_options_table'].cellWidget(current_row, Columns.WEIGHT)
        new_weight = spinner.value()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        display_option.weight = float(new_weight)
        self._widgets['display_options_table'].item(current_row, Columns.WEIGHT).setText(str(new_weight))
        self._display_options_changed()

    def _display_color_change(self):
        color = QColorDialog.getColor()
        current_row = self._widgets['display_options_table'].currentRow()
        current_name = self._widgets['display_options_table'].item(current_row, Columns.NAME).text()
        display_option = None
        for item in self._order_and_color:
            if item.display_name == current_name:
                display_option = item
                break
        display_option.color = color.name()
        button = self._widgets['display_options_table'].cellWidget(current_row, Columns.COLOR)
        button.setStyleSheet(f'background-color: {display_option.color}')
        self._widgets['display_options_table'].item(current_row, Columns.COLOR).setText(display_option.color)
        self._display_options_changed()

    def _display_options_changed(self):
        self._draw_polygon_plot()

    def _raise_display_priority(self):
        current_row = self._widgets['display_options_table'].currentRow()
        if current_row <= 0:
            return
        new_row = current_row - 1
        current_priority = self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).text()
        new_priority = self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).text()
        current_item = self._order_and_color[len(self._order_and_color) - 1 - current_row]
        other_item = self._order_and_color[len(self._order_and_color) - 1 - new_row]
        current_item.priority = int(new_priority)
        other_item.priority = int(current_priority)
        self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).setText(new_priority)
        self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).setText(current_priority)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._display_options_changed()

    def _lower_display_priority(self):
        current_row = self._widgets['display_options_table'].currentRow()
        if current_row == -1 or current_row == len(self._order_and_color) - 1:
            return
        new_row = current_row + 1
        current_priority = self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).text()
        new_priority = self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).text()
        current_item = self._order_and_color[len(self._order_and_color) - 1 - current_row]
        other_item = self._order_and_color[len(self._order_and_color) - 1 - new_row]
        current_item.priority = int(new_priority)
        other_item.priority = int(current_priority)
        self._widgets['display_options_table'].item(current_row, Columns.PRIORITY).setText(new_priority)
        self._widgets['display_options_table'].item(new_row, Columns.PRIORITY).setText(current_priority)
        self._widgets['display_options_table'].sortItems(Columns.PRIORITY, Qt.DescendingOrder)
        self._display_options_changed()

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def _setup_validators(self):
        """Add numeric validators to edit field widgets."""
        # Create validators
        self._dbl_validator = QxDoubleValidator(parent=self)
        self._pos_validator = QxDoubleValidator(parent=self)
        self._pos_validator.setBottom(0.0)
        self._greater_than_zero_validator = QxDoubleValidator(parent=self)
        self._greater_than_zero_validator.setBottom(0.1)
        self._change_limit_validator = QxDoubleValidator(parent=self)
        self._change_limit_validator.setBottom(0.0)
        self._change_limit_validator.setTop(1.0)

        # Connect validators to edit field widgets
        self._widgets['edt_elevation'].setValidator(self._dbl_validator)
        self._widgets['edt_slope'].setValidator(self._pos_validator)
        self._widgets['edt_max_slope_dist'].setValidator(self._pos_validator)
        self._widgets['edt_manning_n'].setValidator(self._pos_validator)
        self._widgets['edt_transition'].setValidator(self._pos_validator)
        self._widgets['edt_quadtree_refinement'].setValidator(self._pos_validator)
        self._widgets['edt_change_limit'].setValidator(self._change_limit_validator)

    def _setup_connections(self):
        """Setup slot/signal connections."""
        self._widgets['cbx_classification'].currentIndexChanged.connect(self._on_classification_changed)
        self._widgets['cbx_transition_method'].currentIndexChanged.connect(self._on_transition_changed)
        self._widgets['edt_transition'].editingFinished.connect(self._on_edit_finished_transition)
        self._widgets['btn_select_lock_dataset'].clicked.connect(self._on_select_lock_dataset)
        self._widgets['btn_select_geom'].clicked.connect(self._on_select_geom)
        self._widgets['btn_insert_feature'].clicked.connect(self._on_insert_feature)
        self._widgets['tog_slope'].stateChanged.connect(self._on_slope_toggle)
        self._widgets['display_up_arrow'].clicked.connect(self._raise_display_priority)
        self._widgets['display_down_arrow'].clicked.connect(self._lower_display_priority)

    def _select_color_for_display(self):
        if len(self._widgets['display_list'].selectedItems()) == 0:
            return
        color = QColorDialog.getColor()
        for item in self._order_and_color:
            if item.name == self._widgets['display_list'].selectedItems()[0].text():
                item.color = color.name()
        self._draw_polygon_plot()

    def _weight_change_for_display(self):
        if len(self._widgets['display_list'].selectedItems()) == 0:
            return
        weight = self._widgets['weight_spin'].value()
        for item in self._order_and_color:
            if item.name == self._widgets['display_list'].selectedItems()[0].text():
                item.weight = weight
        self._draw_polygon_plot()

    def _display_list_select_change(self):
        for item in self._order_and_color:
            if item.name == self._widgets['display_list'].selectedItems()[0].text():
                self._widgets['weight_spin'].setValue(item.weight)

    def _set_meta_data_text(self):
        """Sets text in the meta data text control."""
        te = self._widgets['txt_edt_meta_data']
        te.setText('')
        txt = '<h2>Classification meta data</h2>\n '
        if self._current_classification != 'Unassigned':
            skip_cols = ['Feature type/Region', 'deprecated']
            row = self._meta_data[self._meta_data['Feature type/Region'] == self._current_classification]
            cols = row.columns
            for col in cols:
                if col not in skip_cols:
                    txt = (f'{txt}<h3>{col}</h3>\n')
                    txt = f'{txt}<p>{row[col].item()}</p>\n\n'
        te.insertHtml(txt)

    def _dim_stuff(self):
        """Set the enabled flag on the widgets."""
        adcirc_levee = 'Internal ADCIRC levee'
        dim_types = ['Unassigned', 'Mesh transition zone', adcirc_levee]
        flag = True
        if self._current_classification in dim_types:
            flag = False
        self._widgets['edt_manning_n'].setEnabled(flag)
        self._widgets['cbx_elevation_method'].setEnabled(flag)
        self._widgets['edt_elevation'].setEnabled(flag)
        self._widgets['tog_slope'].setEnabled(flag)
        self._widgets['edt_slope'].setEnabled(flag)
        self._widgets['edt_max_slope_dist'].setEnabled(flag)
        if self._current_classification == adcirc_levee:
            flag = True
        self._widgets['cbx_transition_method'].setEnabled(flag)
        self._widgets['edt_transition'].setEnabled(flag)
        if flag:
            transition = self._widgets['cbx_transition_method'].currentText()
            if transition == 'Transition polygon':
                self._widgets['edt_transition'].setEnabled(False)
            else:
                self._widgets['edt_transition'].setEnabled(True)
        # For the slope toggle
        widgets = [
            self._widgets['lbl_slope'], self._widgets['lbl_max_slope_dist'], self._widgets['edt_slope'],
            self._widgets['edt_max_slope_dist']
        ]
        for widget in widgets:
            if self._widgets['tog_slope'].isChecked():
                widget.show()
            else:
                widget.hide()

    def _on_classification_changed(self, index):
        """Called when the classification changes.

        Args:
            index (int): the new index
        """
        old_class = self._current_classification
        self._current_classification = self._widgets['cbx_classification'].currentText()
        manning_n = 0.0
        if self._widgets['edt_manning_n'].text():
            manning_n = float(self._widgets['edt_manning_n'].text())
        if old_class == 'Unassigned' and 0.0 == manning_n:
            row = self._meta_data[self._meta_data['Feature type/Region'] == self._current_classification]
            self._widgets['edt_manning_n'].setText(str(row['Default n'].item()))
        self._set_meta_data_text()
        self._dim_stuff()

    def _on_edit_finished_transition(self):
        """Finish editing transition distance."""
        self._dim_stuff()
        self._get_polys_for_plot()
        self._draw_polygon_plot()

    def _on_transition_changed(self, index):
        """Called when the classification changes.

        Args:
            index (:obj:`int`): the new index
        """
        if self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_POLYGON:
            if 'transition_poly_pts' not in self._dlg_data:
                self._widgets['cbx_transition_method'].setCurrentIndex(0)
                txt = consts.EWN_TRANSITION_METHOD[consts.TRANSITION_METHOD_POLYGON]
                msg = f'Unable to change "Maximum transition distance" option to {txt!r}.\n' \
                      f'No transition polygon exists that contains the current polygon.'
                msg_box = QMessageBox(QMessageBox.Warning, 'SMS', msg, QMessageBox.Ok, self)
                msg_box.exec()
        if self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_FACTOR:
            self._widgets['edt_transition'].setText(str(self._initial_transition_factor))
        elif self._widgets['cbx_transition_method'].currentIndex() == consts.TRANSITION_METHOD_DISTANCE:
            self._widgets['edt_transition'].setText(str(self._initial_transition_distance))
        self._on_edit_finished_transition()

    def _on_select_geom(self):
        """Select a ugrid or mesh."""
        problem = self._check_project_explorer()
        if problem:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS',
                'No mesh or ugrid defined. Preview is only supported for meshes and ugrids.', QMessageBox.Ok, self
            )
            msg_box.exec()
            return
        selector_dlg = TreeItemSelectorDlg(
            title='Select Source Geometry',
            target_type='',
            pe_tree=self._pe_tree,
            parent=self,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS', 'TI_CGRID2D']
        )

        dlg_ran = selector_dlg.exec()
        if not dlg_ran:
            return

        source_geom_uuid = selector_dlg.get_selected_item_uuid()
        if not source_geom_uuid:  # No geometry selected
            self._reset_selected_geom()
            self._reset_lock_dataset()
            return

        self._geom_uuid = source_geom_uuid
        if source_geom_uuid != self._geom_uuid:
            self._reset_lock_dataset()
        tree_path = tree_util.build_tree_path(self._pe_tree, source_geom_uuid)

        self._widgets['txt_geom'].setText(tree_path)
        self._widgets['txt_lock_dataset'].setEnabled(True)
        self._widgets['btn_select_lock_dataset'].setEnabled(True)
        self.setCursor(Qt.WaitCursor)

        if source_geom_uuid not in self._ugrids:
            self._dlg_data['geom_uuid'] = self._geom_uuid
            preview_get_geometry_feedback(self)
        self._dlg_data['ugrid'] = self._ugrids[source_geom_uuid]
        self.is_cartesian = self.cogrid.grid_type in [GridType.rectilinear_2d, GridType.rectilinear_3d]
        self.is_quadtree = self.cogrid.grid_type in [GridType.quadtree_2d, GridType.quadtree_3d]

        if self.is_cartesian or self.is_quadtree:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Preview only supports meshes and non quadtree ugrids.', QMessageBox.Ok,
                self
            )
            msg_box.exec()
            self._reset_selected_geom()
            self.unsetCursor()
            return

        self._subset = self._subsets[source_geom_uuid]
        self._clipped_ugrid = None
        if 'ugrid' in self._dlg_data:
            if self._dlg_data['ugrid'].dimension_counts[3] > 0:
                self._reset_selected_geom()
                msg_box = QMessageBox(
                    QMessageBox.Warning, 'SMS', 'Selected Mesh/UGrid is 3D. Only 2D is supported.', QMessageBox.Ok, self
                )
                msg_box.exec()
                self.unsetCursor()
                return
            self._draw_polygon_plot()

    def _get_applicable_geom_selector_tree(self):
        creator = tree_util.ProjectExplorerTreeCreator()
        geom_tree = creator.copy(self._pe_tree)
        tree_util.filter_project_explorer(geom_tree, is_applicable_geom)
        return geom_tree

    def _get_dataset_selector_tree(self):
        """Get a dataset selector project explorer tree."""
        geom_item = tree_util.find_tree_node_by_uuid(self._pe_tree, self._geom_uuid)
        creator = tree_util.ProjectExplorerTreeCreator()
        dataset_tree = creator.copy(geom_item)
        if dataset_tree:
            tree_util.filter_project_explorer(dataset_tree, is_scalar_nodes_if_dset)
        return dataset_tree

    def _reset_selected_geom(self):
        self._widgets['txt_geom'].setText('(none selected)')
        self._dlg_data['ugrid'] = None
        self._geom_uuid = None
        self._dlg_data['geom_uuid'] = None
        self._clipped_ugrid = None
        self._subset = None
        self._widgets['txt_lock_dataset'].setEnabled(False)
        self._widgets['btn_select_lock_dataset'].setEnabled(False)

    def _reset_lock_dataset(self):
        """Reset the lock dataset."""
        self._widgets['txt_lock_dataset'].setText('(none selected)')
        self._dlg_data['lock_dataset'] = None
        self._lock_dataset_uuid = None
        self._locked_node_locations = None
        self._clipped_lock_dataset = None
        self._dlg_data['lock_dataset_uuid'] = self._lock_dataset_uuid

    def _on_select_lock_dataset(self):
        """Select a ugrid or mesh."""
        selector_dlg = TreeItemSelectorDlg(
            title='Select Lock Dataset',
            target_type='',
            pe_tree=self._get_dataset_selector_tree(),
            parent=self,
            show_root=True,
            selectable_xms_types=['TI_SFUNC', 'TI_SCALAR_DSET']
        )

        if selector_dlg.exec():
            lock_dataset_uuid = selector_dlg.get_selected_item_uuid()
            if not lock_dataset_uuid:  # No lock dataset selected
                self._reset_lock_dataset()
                self._draw_polygon_plot()
                return
            tree_path = tree_util.build_tree_path(self._pe_tree, lock_dataset_uuid)
            self._widgets['txt_lock_dataset'].setText(tree_path)
            self._lock_dataset_uuid = lock_dataset_uuid
            self._dlg_data['lock_dataset_uuid'] = self._lock_dataset_uuid

        # Get lock datset
        self._set_clipped_lock_dataset()

        if 'ugrid' in self._dlg_data:
            self._draw_polygon_plot()

    def _set_clipped_lock_dataset(self):
        query = self._dlg_data['query']
        lock_dataset = None
        if self._lock_dataset_uuid:
            lock_dataset = query.item_with_uuid(self._lock_dataset_uuid)
        else:
            return
        locations = [list(x) for x in self._ugrids[self._geom_uuid].locations]

        if lock_dataset:
            self._clipped_lock_dataset = [
                lock_dataset.values[0, locations.index(list(x))] for x in self._clipped_ugrid.locations
            ]

        self._locked_node_locations = [
            (x[0], x[1]) for x in self._clipped_ugrid.locations if lock_dataset.values[0, locations.index(list(x))] != 0
        ]

    def _on_insert_feature(self):
        """Inserts a feature and creates a new mesh."""
        if self._clipped_ugrid is None:
            msg_box = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Must select a mesh/ugrid to insert a feature.', QMessageBox.Ok, self
            )
            msg_box.exec()
            return
        self._insert_ugrid = None
        poly_pts = self._dlg_data['poly_pts']
        if self._widgets['cbx_elevation_method'].currentIndex() == 0:
            elev = float(self._widgets['edt_elevation'].text())
            poly_pts = [[p[0], p[1], elev] for p in poly_pts]
        poly_data = {
            'polygon_outside_pts_ccw': poly_pts,
            'classification': self._widgets['cbx_classification'].currentIndex(),
            'elevation': float(self._widgets['edt_elevation'].text()),
            'specify_slope': True if self._widgets['tog_slope'].isChecked() else False,
            'slope': float(self._widgets['edt_slope'].text()),
            'max_distance': float(self._widgets['edt_max_slope_dist'].text()),
            'elevation_method': self._widgets['cbx_elevation_method'].currentIndex(),
            'polygon_id': self._dlg_data['polygon_id'],
            'transition_method': self._widgets['cbx_transition_method'].currentIndex(),
            'transition_distance': float(self._widgets['edt_transition'].text())
        }
        if poly_data['transition_method'] == consts.TRANSITION_METHOD_POLYGON:
            poly_data['transition_poly_pts'] = self._dlg_data['transition_poly_pts']

        if poly_data['classification'] < 2:
            message = 'Invalid classification type.  Classification must not be Unassigned or Mesh transition zone'
            msg_box = QMessageBox(QMessageBox.Warning, 'SMS', message, QMessageBox.Ok, self)
            msg_box.exec()
            return
        bias = 1.0 - float(self._widgets['edt_change_limit'].text())
        is_levee = poly_data['classification'] == 2
        if is_levee:
            arcs = self._dlg_data['polygon'].arcs
            if len(arcs) != 4:
                poly = self._dlg_data['polygon']
                msg = f'Polygon id: {poly.id} invalid. Levees must be defined using 4 arcs.'
                msg_box = QMessageBox(QMessageBox.Warning, 'SMS', msg, QMessageBox.Ok, self)
                msg_box.exec()
                return

        ewn = EwnProcess(
            self._clipped_ugrid,
            self._is_geographic,
            is_levee,
            bias,
            lock_dataset=self._clipped_lock_dataset,
            is_cartesian=self.is_cartesian
        )
        ewn.show_transition_warning = False
        ewn.set_polygon_data(poly_data)
        preview_ewn_insert_feedback(self, ewn)
        if ewn.stitched_ugrid is not None:
            self._insert_ugrid = ewn.stitched_ugrid
            self._insert_transition_polygon = ewn.transition_polygon
            self._draw_polygon_plot()

    def _on_slope_toggle(self):
        """Called when the slope toggle box state changes."""
        self._dim_stuff()

    def _get_polys_for_plot(self):
        """Sets member variables for polygons to draw in the plot."""
        if 'poly_pts' not in self._dlg_data:
            return
        poly_pts = self._dlg_data['poly_pts']
        self._plot_transition_pts = []
        method = self._widgets['cbx_transition_method'].currentIndex()
        if method == consts.TRANSITION_METHOD_POLYGON and 'transition_poly_pts' in self._dlg_data:
            trans_poly = self._dlg_data['transition_poly_pts']
            xcoord, ycoord, _ = zip(*trans_poly)
            self._plot_transition_pts = [(pt[0], pt[1]) for pt in trans_poly]
        else:
            xcoord, ycoord, _ = zip(*poly_pts)
        self._box_min = [min(xcoord), min(ycoord)]
        self._box_max = [max(xcoord), max(ycoord)]

        dist = 0.0
        if method == consts.TRANSITION_METHOD_FACTOR:
            dist = float(self._widgets['edt_transition'].text())
            length = math.sqrt((self._box_max[0] - self._box_min[0])**2 + (self._box_max[1] - self._box_min[1])**2)
            dist = dist * length
        elif method == consts.TRANSITION_METHOD_DISTANCE:
            dist = float(self._widgets['edt_transition'].text())
            if self._is_geographic:
                _, _, center_y = self._length_and_center_from_box()
                dist = meters_to_decimal_degrees(dist, center_y)
        self._box_min = [self._box_min[0] - dist, self._box_min[1] - dist]
        self._box_max = [self._box_max[0] + dist, self._box_max[1] + dist]

        if not self._plot_transition_pts:
            self._plot_transition_pts = [
                (self._box_min[0], self._box_min[1]), (self._box_max[0], self._box_min[1]),
                (self._box_max[0], self._box_max[1]), (self._box_min[0], self._box_max[1]),
                (self._box_min[0], self._box_min[1])
            ]
        # must be in longitude, latitude
        self._plot_poly_pts = [(pt[0], pt[1]) for pt in poly_pts]
        self._clipped_ugrid = None

    def _draw_polygon_plot(self):
        """Draw the input polygon on the map."""
        if self._plot_poly_pts is None:
            self._get_polys_for_plot()
            if self._plot_poly_pts is None:
                return
        if 'ugrid' in self._dlg_data and self._dlg_data['ugrid'] is not None and self._clipped_ugrid is None:
            box = [(self._box_min[0], self._box_min[1]), (self._box_max[0], self._box_max[1])]
            self._clipped_ugrid = self._subset.subset_from_box(box)
            self._set_clipped_lock_dataset()

        folium_map = folium.Map(tiles='')
        if not self._is_local and self._display_map:
            url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}'
            tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI Street Map', attr='ESRI')
            tl.add_to(folium_map)
            url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
            tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI World Imagery', attr='ESRI')
            tl.add_to(folium_map)
            folium.LayerControl().add_to(folium_map)
        plot_pts = self._transform_points(self._plot_poly_pts)

        self._order_and_color.sort(key=lambda x: x.priority)

        for display_item in self._order_and_color:
            # Polygon
            if display_item.name == 'ewn_polygon' and display_item.visible:
                folium.vector_layers.PolyLine(
                    locations=flip_long_lat(plot_pts),
                    weight=display_item.weight,
                    color=display_item.color,
                    tooltip='polygon'
                ).add_to(folium_map)
            # Transition Boundary
            elif display_item.name == 'transition_boundary':
                transition_plot_pts = self._transform_points(self._plot_transition_pts)
                trans_poly = folium.vector_layers.PolyLine(
                    locations=flip_long_lat(transition_plot_pts),
                    weight=display_item.weight,
                    color=display_item.color,
                    tooltip='Maximum transition polygon'
                )
                if display_item.visible:  # Do this here so we can get the bounds.
                    trans_poly.add_to(folium_map)
                bounds = trans_poly.get_bounds()
                folium_map.fit_bounds(bounds)
            # Input Mesh
            elif display_item.name == 'input' and display_item.visible:
                self._add_ugrid_edges_to_plot(
                    folium_map, 'Input mesh/ugrid', self._clipped_ugrid, display_item.color, weight=display_item.weight
                )
            # Transition Polygon
            elif display_item.name == 'transition_polygon' and display_item.visible:
                self._add_insert_transition_poly_to_plot(folium_map, display_item.color, weight=display_item.weight)
            # Lock Nodes
            elif display_item.name == 'lock_nodes' and display_item.visible:
                self._add_locked_nodes_to_plot(folium_map, display_item.color, weight=display_item.weight)
            # Output Mesh
            elif display_item.name == 'output' and display_item.visible:
                self._add_ugrid_edges_to_plot(
                    folium_map, 'Output mesh/ugrid', self._insert_ugrid, display_item.color, weight=display_item.weight
                )
        # Use io, rather than a file on disk, to store the map as html in memory
        folium_map.save(self._url_file)

        if self._mesh_viewer is None:
            self._mesh_viewer = MeshViewerDialog(parent=self, map_url=self._url_file, folium_map=folium_map)
            self._mesh_viewer.show()
        self._mesh_viewer.update_display()
        self.unsetCursor()

    def _add_locked_nodes_to_plot(self, folium_map, color, weight):
        """Add locked nodes to plot."""
        if self._locked_node_locations is None:
            return
        lock_nodes = self._transform_points(self._locked_node_locations)
        tip = 'Locked Polygon'
        for pt in flip_long_lat(lock_nodes):
            folium.vector_layers.PolyLine(locations=[pt, pt], weight=weight, color=color,
                                          tooltip=tip).add_to(folium_map)

    def _add_insert_transition_poly_to_plot(self, folium_map, color, weight):
        """Draw the insert transition polygon on the map.

        Args:
            folium_map (:obj:`folium.Map`): the map control
            color (:obj:`str`): color for poly
            weight (:obj:`float`): weight of poly line
        """
        if self._insert_transition_polygon is None:
            return
        poly_pts = [(pt[0], pt[1]) for pt in self._insert_transition_polygon]
        tip = 'Mesh transition polygon'
        plot_pts = self._transform_points(poly_pts)
        folium.vector_layers.PolyLine(locations=flip_long_lat(plot_pts), weight=weight, color=color,
                                      tooltip=tip).add_to(folium_map)

    def _add_ugrid_edges_to_plot(self, folium_map, name, ugrid, color, weight=1):
        """Draw the ugrid on the map.

        Args:
            folium_map (:obj:`folium.Map`): the map control
            name (:obj:`str`): name of this layer
            ugrid (:obj:`xms.grid.UGrid`): ugrid
            color (:obj:`str`): color to draw the grid
            weight (:obj:`int`): weight of line being drawn
        """
        if ugrid is None:
            return
        # Get the bounds of the grid points passed in
        grid_points = ugrid.locations

        # Hash the edges
        cell_count = ugrid.cell_count
        edge_set = set()  # Use this set to remove duplicate edges
        for cell_idx in range(cell_count):
            cell_edges = ugrid.get_cell_edges(cell_idx)
            for edge in cell_edges:
                if edge[0] > edge[1]:
                    edge_set.add((edge[1], edge[0]))
                else:
                    edge_set.add((edge[0], edge[1]))

        # fg = folium.map.FeatureGroup(name).add_to(folium_map)
        ml_list = []
        for edge in edge_set:
            pts = [
                (grid_points[edge[0]][0], grid_points[edge[0]][1]), (grid_points[edge[1]][0], grid_points[edge[1]][1])
            ]
            pts = self._transform_points(pts)
            pts = flip_long_lat(pts)
            ml_list.append([pts[0], pts[1]])
        if len(ml_list) > 0:
            folium.vector_layers.PolyLine(locations=ml_list, weight=weight, color=f'{color}').add_to(folium_map)

    def _check_project_explorer(self):
        """Check if the project explorer is available."""
        if self._pe_tree is None:
            return 'Unable to retrieve the SMS project explorer tree.'

        for root_child in self._pe_tree.children:  # Loop through children of the project tree root
            if root_child.children and root_child.name in ['Mesh Data', 'UGrid Data']:
                # Found either the 2D Mesh root or the UGrid root and it has children. Assume there is at least
                # one valid geometry.
                return ''
        return 'A 2D Mesh or UGrid is required to run this command.'

    def _create_transition_coverage(self):
        new_transition_cov = Coverage(name='EWN Transition Polygon', uuid=str(uuid.uuid4()))
        pts = []
        cur_pt_id = 1
        for pt in self._insert_transition_polygon[:-1]:
            d_pt = Point(pt[0], pt[1], 0.0)
            d_pt.id = cur_pt_id
            cur_pt_id += 1
            pts.append(d_pt)
        new_transition_arc = Arc(start_node=pts[0], end_node=pts[0], vertices=pts[1:], feature_id=1)
        new_transition_polygon = Poly()
        new_transition_polygon.id = 1
        new_transition_polygon.set_arcs([new_transition_arc])
        new_transition_cov.arcs = [new_transition_arc]
        new_transition_cov.polygons = [new_transition_polygon]
        new_transition_cov.complete()
        self.transition_poly_cov = new_transition_cov

    def _save_data(self):
        """Save widget values."""
        self._data['polygon_name'] = self._widgets['edt_name'].text()
        self._data['classification'] = self._widgets['cbx_classification'].currentIndex()
        self._data['insert_feature'] = 1 if self._widgets['tog_insert'].isChecked() else 0
        self._data['elevation_method'] = self._widgets['cbx_elevation_method'].currentIndex()
        self._data['elevation'] = float(self._widgets['edt_elevation'].text())
        self._data['specify_slope'] = True if self._widgets['tog_slope'].isChecked() else False
        self._data['slope'] = float(self._widgets['edt_slope'].text())
        self._data['maximum_slope_distance'] = float(self._widgets['edt_max_slope_dist'].text())
        self._data['manning_n'] = float(self._widgets['edt_manning_n'].text())
        self._data['transition_method'] = self._widgets['cbx_transition_method'].currentIndex()
        self._data['transition_distance'] = float(self._widgets['edt_transition'].text())
        self._data['quadtree_refinement_length'] = float(self._widgets['edt_quadtree_refinement'].text())
        if self._widgets['tog_add_to_xms'].isChecked() and self._insert_ugrid is not None:
            self.new_ugrid = self._subset.stitch_grid(self._insert_ugrid)
        if self._widgets['tog_add_transition_cov_to_xms'].isChecked() and self._insert_ugrid is not None:
            self._create_transition_coverage()

    def accept(self):
        """Save data to persistent storage on OK."""
        self._save_data()
        if self._mesh_viewer is not None:
            self._mesh_viewer.close()
        super().accept()

    def reject(self):
        """Cancel command."""
        if self._mesh_viewer is not None:
            self._mesh_viewer.close()
        super().reject()

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def _transform_points(self, points):
        """Transforms the points from their current CRS (coordinate reference system) to the new one.

        See https://gis.stackexchange.com/questions/226200/how-to-call-gdaltransform-with-its-input

        Args:
            points (:obj:`list`): List of points.

        Returns:
            new_points (:obj:`list`): The transformed points.
        """
        if not self._transform and not self._is_local:
            return points

        # Transform each point
        new_points = []
        for point in points:
            if self._transform:
                # new_points = gdal_utils.transform_points([(p[0], p[1]) for p in points], self._transform)
                coords = gdal_utils.transform_point(point[0], point[1], self._transform)
                new_points.append([coords[0], coords[1]])
            else:
                x = meters_to_decimal_degrees(point[0], 0.0)
                y = meters_to_decimal_degrees(point[1], 0.0)
                new_points.append([x, y])

        return new_points


def flip_long_lat(points):
    """Function to flip the lat and long for x and y."""
    return [(p[1], p[0]) for p in points]
