"""The solution output options table."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt
from PySide2.QtGui import QColor

# 3. Aquaveo modules

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts


class ComputationTableFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states in computation table."""
    def __init__(self, parent):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        col = index.column()

        disable = False
        sediment_type = int(float(index.model().index(index.row(), consts.COL_SEDIMENT_TYPE).data(Qt.EditRole)))
        if col > consts.COL_SEDIMENT_TYPE:
            if sediment_type == consts.SEDIMENT_TYPE_NONE:
                disable = True  # If sediment is disabled, only allow user to edit polygon name and sediment type.
            elif col in [consts.COL_PRIORITY, consts.COL_PRIORITY_PERCENT] \
                    and sediment_type in [consts.SEDIMENT_TYPE_SPECIFY_FILL, consts.SEDIMENT_TYPE_SPECIFY_CUT]:
                disable = True  # No priority or priority percent if specified type
        if disable or col >= consts.COL_TOTAL_VOLUME:
            ret_flags = ret_flags & (~Qt.ItemIsEnabled)
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
        else:
            ret_flags |= Qt.ItemIsEnabled
            ret_flags |= Qt.ItemIsEditable
        return ret_flags

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (:obj:`int`): The role.

        Returns:
            The data at index, or None.
        """
        if not index.isValid():
            return None

        if role == Qt.BackgroundColorRole:
            sediment_type = int(float(index.model().index(index.row(), consts.COL_SEDIMENT_TYPE).data(Qt.EditRole)))
            if sediment_type in [consts.SEDIMENT_TYPE_AVAILABLE_FILL, consts.SEDIMENT_TYPE_AVAILABLE_CUT]:
                return QColor('lightBlue')
            elif sediment_type in [consts.SEDIMENT_TYPE_SPECIFY_FILL, consts.SEDIMENT_TYPE_SPECIFY_CUT]:
                return QColor('lightYellow')
        return self.sourceModel().data(index, role)
