"""Dialog for the insert levee tool."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import (
    QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QMessageBox, QPushButton, QSizePolicy, QSpacerItem,
    QVBoxLayout
)

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules


class InsertLeveeDialog(XmsDlg):
    """Dialog for the insert levee tool."""
    def __init__(self, parent, pe_tree):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent Qt dialog
            pe_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): The SMS project explorer tree
        """
        super().__init__(parent, 'xms.ewn.gui.insert_levee_dialog')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:EWN'
        self._pe_tree = pe_tree
        self._widgets = {}
        self._source_geom_uuid = ''
        self._adcirc_bc_coverage_uuid = ''

        self._bc_pe_tree = None
        self._filter_tree_to_adcirc_bc_coverages()

        self._setup_ui()

    def _setup_ui(self):
        """Setup dialog widgets."""
        self.setWindowTitle('Insert ADCIRC Levee')
        self._setup_source_geometry()
        self._setup_coverage()
        self._setup_transition()
        self._setup_main_layout()

    def _setup_source_geometry(self):
        """Set up the source geometry widgets."""
        self._widgets['lbl_source_geom'] = QLabel('Source geometry:')
        self._widgets['btn_source_geom'] = QPushButton('Select...')
        self._widgets['btn_source_geom'].clicked.connect(self.select_source_geometry)
        self._widgets['lbl_selected_geom'] = QLabel('(none selected)')
        self._widgets['layout_horiz_source_geom'] = QHBoxLayout()
        self._widgets['layout_horiz_source_geom'].addWidget(self._widgets['btn_source_geom'])
        self._widgets['layout_horiz_source_geom'].addWidget(self._widgets['lbl_selected_geom'])
        self._widgets['spcr_horiz_source_geom'] = QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        self._widgets['layout_horiz_source_geom'].addItem(self._widgets['spcr_horiz_source_geom'])

    def _setup_coverage(self):
        """Set up the adcirc bc coverage widgets."""
        self._widgets['lbl_bc_cov'] = QLabel('ADCIRC BC coverage:')
        self._widgets['btn_bc_cov'] = QPushButton('Select...')
        self._widgets['btn_bc_cov'].clicked.connect(self.select_bc_cov)
        self._widgets['lbl_selected_bc_cov'] = QLabel('(none selected)')
        self._widgets['layout_horiz_bc_cov'] = QHBoxLayout()
        self._widgets['layout_horiz_bc_cov'].addWidget(self._widgets['btn_bc_cov'])
        self._widgets['layout_horiz_bc_cov'].addWidget(self._widgets['lbl_selected_bc_cov'])
        self._widgets['spcr_horiz_bc_cov'] = QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
        self._widgets['layout_horiz_bc_cov'].addItem(self._widgets['spcr_horiz_bc_cov'])

    def _setup_transition(self):
        """Set up the adcirc bc coverage widgets."""
        float_validator = QxDoubleValidator(parent=self)
        float_validator.setBottom(0.0)
        float_validator.setTop(1.0)
        self._widgets['lbl_transition'] = QLabel('Element area change limit (fast 0.0 <---> 1.0 slow):')
        self._widgets['edt_transition'] = QLineEdit('0.5')
        self._widgets['edt_transition'].setValidator(float_validator)
        size_policy = self._widgets['edt_transition'].sizePolicy()
        size_policy.setHorizontalPolicy(QSizePolicy.MinimumExpanding)
        self._widgets['edt_transition'].setSizePolicy(size_policy)

    def _setup_main_layout(self):
        """Adds the dialog button bar and sets the top-level dialog layout."""
        # Add the dialog buttons layout
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])

        # Add a main layout.
        self._widgets['grp_main'] = QGroupBox('')
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['main_vert_layout'].addWidget(self._widgets['grp_main'])
        self._widgets['grp_main_vert_layout'] = QVBoxLayout()
        self._widgets['grp_main'].setLayout(self._widgets['grp_main_vert_layout'])
        self._widgets['grp_main_vert_layout'].addWidget(self._widgets['lbl_source_geom'])
        self._widgets['grp_main_vert_layout'].addLayout(self._widgets['layout_horiz_source_geom'])
        self._widgets['grp_main_vert_layout'].addWidget(self._widgets['lbl_bc_cov'])
        self._widgets['grp_main_vert_layout'].addLayout(self._widgets['layout_horiz_bc_cov'])
        self._widgets['grp_main_vert_layout'].addWidget(self._widgets['lbl_transition'])
        self._widgets['grp_main_vert_layout'].addWidget(self._widgets['edt_transition'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])
        self.setLayout(self._widgets['main_vert_layout'])

    def select_source_geometry(self):
        """Slot to display the source geometry tree item selector."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select Source Geometry',
            target_type='',
            pe_tree=self._pe_tree,
            previous_selection=self._source_geom_uuid,
            parent=self,
            selectable_xms_types=['TI_MESH2D', 'TI_UGRID_SMS']
        )

        if selector_dlg.exec():
            self._source_geom_uuid = selector_dlg.get_selected_item_uuid()
            if self._source_geom_uuid:
                self._widgets['lbl_selected_geom'].setText(
                    tree_util.build_tree_path(self._pe_tree, self._source_geom_uuid)
                )
            else:  # No target geometry selected, clear and disable the initial dataset selection
                self._widgets['lbl_selected_geom'].setText('(none selected)')

    def select_bc_cov(self):
        """Slot to display the source geometry tree item selector."""
        # Display a tree item selector dialog.
        selector_dlg = TreeItemSelectorDlg(
            title='Select ADCIRC Boundary Conditon Coverage',
            target_type='',
            pe_tree=self._bc_pe_tree,
            previous_selection=self._adcirc_bc_coverage_uuid,
            parent=self,
            selectable_xms_types=['TI_COVER']
        )

        if selector_dlg.exec():
            self._adcirc_bc_coverage_uuid = selector_dlg.get_selected_item_uuid()
            if self._adcirc_bc_coverage_uuid:
                self._widgets['lbl_selected_bc_cov'].setText(
                    tree_util.build_tree_path(self._bc_pe_tree, self._adcirc_bc_coverage_uuid)
                )
            else:  # No target geometry selected, clear and disable the initial dataset selection
                self._widgets['lbl_selected_bc_cov'].setText('(none selected)')

    def _filter_tree_to_adcirc_bc_coverages(self):
        """Creates a copy of the project explorer tree and filters out non-EWN coverages."""
        creator = tree_util.ProjectExplorerTreeCreator()
        self._bc_pe_tree = creator.copy(self._pe_tree)
        tree_util.filter_project_explorer(self._bc_pe_tree, is_adcirc_bc_if_coverage)

    def accept(self):
        """Ensure a geometry and output file has been selected."""
        if not self._source_geom_uuid:
            msgbox = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Select a source Mesh 2D or UGrid geometry.', QMessageBox.Ok, self
            )
            msgbox.exec()
            return
        if not self._adcirc_bc_coverage_uuid:
            msgbox = QMessageBox(
                QMessageBox.Warning, 'SMS', 'Select an ADCIRC Boundary Conditions coverage.', QMessageBox.Ok, self
            )
            msgbox.exec()
            return
        super().accept()

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def tool_inputs(self):
        """Returns a dict of the user inputs."""
        return {
            'ugrid': self._source_geom_uuid,
            'adcirc_coverage': self._adcirc_bc_coverage_uuid,
            'bias': float(self._widgets['edt_transition'].text())
        }


def is_adcirc_bc_if_coverage(item):
    """Check if a tree item is an ADCIRC BC coverage, but only if it is a coverage.

    Args:
        item (:obj:`TreeNode`): The item to check

    Returns:
        (:obj:`bool`): True if the tree item is an EWN Features coverage type or is not a coverage.
    """
    if item.item_typename == 'TI_COVER':
        if item.model_name == 'ADCIRC' and item.coverage_type == 'Boundary Conditions':
            return True
        return False
    return True
