"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.gencade.components.gencade_component import GenCadeComponent
from xms.gencade.data.sim_data import SimData
from xms.gencade.feedback.grid_mapping_worker_thread import GridMappingWorkerThread
from xms.gencade.gui.model_control_dlg import ModelControlDlg
from xms.gencade.gui.view_solution_dialog import ViewSolutionDialog


class SimComponent(GenCadeComponent):
    """A hidden Dynamic Model Interface (DMI) component for the GenCade model simulation."""

    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = SimData(self.main_file)
        self.class_name = 'SimComponent'
        self.module_name = 'xms.gencade.components.sim_component'
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            ('View Solution...', 'view_solution'),
        ]  # [(menu_text, menu_method)...]

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (:obj:`dict`): A dictionary with keys being UUIDs as strings representing the objects being
                linked into this component. The values of this dictionary are a list of strings of the parameter names
                of the "takes" from the XML that this is a part of.
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        for _, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param == 'gridCov':
                    action = ActionRequest(main_file=self.main_file, modality='MODAL', class_name=self.class_name,
                                           module_name=self.module_name, method_name='apply_to_sim',
                                           comp_uuid=self.uuid)
                    actions.append(action)
        self.data.commit()
        return messages, actions

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        prj_horiz_units = query.display_projection.horizontal_units.lower()
        dlg = ModelControlDlg(self.data, prj_horiz_units, win_cont)
        if dlg.exec():
            self.data.commit()
        return [], []

    def view_solution(self, query, params, win_cont):
        """Opens the View Solution dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        dlg = ViewSolutionDialog(self.data, win_cont)
        dlg.exec()
        return [], []

    def apply_to_sim(self, query, params, win_cont):
        """Action Request method for mapping the GenCade 1D grid.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): The ActionRequest parameter map, unused
            win_cont (:obj:`QWidget`): The parent window, unused

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        worker = GridMappingWorkerThread(query, self, win_cont)
        display_text = {
            'title': 'Applying 1D Grid',
            'working_prompt': 'Applying 1D grid to GenCade simulation. Please wait...',
            'error_prompt': 'Error(s) encountered applying 1D grid to simulation. Review log output for '
                            'more details.',
            'warning_prompt': 'Warning(s) encountered applying 1D grid to simulation. Review log output '
                              'for more details.',
            'success_prompt': 'Successfully applied 1D grid to simulation. Close this dialog to load the '
                              'applied data into SMS.',
            'note': '',
            'auto_load': 'Auto load applied data into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.gencade', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        if not feedback_dlg.exec():  # User aborted, clear any added data and unlink the tidal simulation.
            query.clear()
            if worker.grid_data['sim_uuid'] and worker.grid_data['grid_cov']:
                query.unlink_item(worker.grid_data['sim_uuid'], worker.grid_data['grid_cov'].uuid)
        return [], []
