"""MappedGridData class."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
# import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as xfs

# 4. Local modules


MAPPED_BC_MAIN_FILE = 'mapped_grid_comp.nc'


class MappedGridData(XarrayBase):
    """Manages data file for the mapped grid component."""

    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        # Initialize member variables before calling super so they are available for commit() call
        self._filename = data_file
        self._info = None
        self._locations = None
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets()
        super().__init__(self._filename)

    def _get_default_datasets(self):
        """Create default datasets if needed."""
        if not os.path.isfile(self._filename):
            info = {
                'FILE_TYPE': 'GENCADE_MAPPED_GRID',
                # 'VERSION': pkg_resources.get_distribution('xmsgencade').version,
                'display_uuid': '',
                'wkt': '',  # Display projection at time of mapping.
                'x0': 0.0,
                'y0': 0.0,
                'xend': 0.0,
                'yend': 0.0,
                'theta': 0.0,
                'num_cells': 0,
            }
            locations_data = {
                'dx': xr.DataArray(data=np.array([], dtype=np.float64)),
            }
            self._info = xr.Dataset(attrs=info)
            self._locations = xr.Dataset(data_vars=locations_data)
            self.commit()

    @property
    def locations(self):
        """Load the locations dataset from disk.

        Returns:
            (:obj:`xarray.Dataset`): Dataset interface to the locations dataset in the main file
        """
        if self._locations is None:
            self._locations = self.get_dataset('locations', False)
        return self._locations

    @locations.setter
    def locations(self, dset):
        """Setter for the locations dataset."""
        if dset:
            self._locations = dset

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._locations is not None:
            self._locations.close()
            self._drop_h5_groups(['locations'])
            self._locations.to_netcdf(self._filename, group='locations', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.
        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._locations is None:
            self._locations = self.get_dataset('locations', False)
        xfs.removefile(self._filename)  # Delete the existing NetCDF file
        self.commit()  # Rewrite all datasets
