"""A writer for GenCade variable resolution grid files."""

# 1. Standard Python modules
from io import StringIO
import logging
import os.path
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs.dialog_util import ensure_qapplication_exists
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.gencade.dmi.sim_queries import SimQueries


class ShdxExporter:
    """A class for writing out a Gencade .shdx file."""

    def __init__(self, main_file: str, query: Query, out_dir: Path, sim_node):
        """Constructor."""
        self._main_file: str = main_file
        self._shdxfile = None
        self._query = query
        self._out_dir = out_dir
        self._sim_node = sim_node
        self._log = logging.getLogger('xms.gencade')
        self.ss = StringIO()
        self.simulation_name = ''
        self.proj_name = ''
        self._sim_data = None
        self._sim_comp = query.item_with_uuid(
            item_uuid=sim_node.uuid, model_name="GenCade", unique_name="SimComponent"
        )
        self._sms_version = ''

    @staticmethod
    def run_script(testing: bool = False) -> None:
        """Runs the simulation exporter when called as a script.

        Args:
            testing (:obj:`bool`): Passing true closes the dialog automatically.
        """
        query = ShdxExporter._create_query()
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        sim_comp = query.item_with_uuid(
            item_uuid=sim_node.uuid, model_name="GenCade", unique_name="SimComponent"
        )
        main_file = sim_comp.main_file

        out_dir = Path.cwd()
        ShdxExporter._run(
            main_file=main_file, query=query, out_dir=out_dir, sim_node=sim_node, testing=testing
        )

    @staticmethod
    def _create_query() -> Query:  # pragma no cover - can't test this without an XMS running
        """Creates and returns a query.

        Returns:
            See description.
        """
        query = Query(timeout=300000)
        return query

    @staticmethod
    def _run(
        main_file: str,
        query: Query,
        out_dir: Path,
        sim_node: TreeNode,
        testing: bool = False
    ) -> None:
        """Runs the simulation exporter.

        Args:
            main_file (:obj:`str`): Simulation component main file.
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with GMS
            out_dir (:obj:`str`): Path where simulation is to be exported.
            sim_node (:obj:`TreeNode`): Simulation tree node.
            testing (:obj:`bool`): Passing true closes the dialog automatically.
        """
        feedback = True
        # To debug
        # feedback = False  # Note: this causes the feedback dialog to close automatically even if there were errors
        # from xms.guipy import debug_pause
        # debug_pause()

        exporter = ShdxExporter(main_file=main_file, query=query, out_dir=out_dir, sim_node=sim_node)
        from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread
        worker = ProcessFeedbackThread(do_work=exporter.do_work, parent=None)
        display_text = {
            'title': 'Saving GenCade Simulation',
            'working_prompt': f'Saving GenCade simulation to \"{out_dir}\".',  # noqa: B028
            'error_prompt': 'Error(s) encountered while saving.',
            'warning_prompt': 'Warning(s) encountered while saving.',
            'success_prompt': f'Successfully exported \"{out_dir}\".',  # noqa: B028
            'note': '',
            # 'auto_load': 'Close this dialog automatically when exporting is finished.',
            'log_format': '%(asctime)s - %(message)s',
            'use_colors': True
        }
        ensure_qapplication_exists()
        feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.gencade', worker=worker,
                                          parent=None)
        feedback_dlg.testing = not feedback or testing
        if feedback_dlg.exec():
            if feedback_dlg.logged_error():
                XmsEnvironment.report_export_error()
        else:
            XmsEnvironment.report_export_aborted()

    def do_work(self) -> None:
        """Exports this HydroGeoSphere simulation."""
        # self._clear_directory()  This probably isn't a good idea.
        shdx_filename = self._write()
        if shdx_filename:
            self._log = logging.getLogger('xms.gencade')
            self._log.info('Simulation export complete.\n')

    def _write(self) -> str:
        """Writes the simulation."""
        self._log.info('Writing simulation...')
        self._log.info('Retrieving source grid data from SMS.')
        self.proj_name = os.path.splitext(os.path.basename(self._query.xms_project_path))[0]
        # Build file names from project name for references in .gen file
        self._shdxfile = f'{self.proj_name}.shdx'

        query_helper = SimQueries(self._sim_comp, self._query)
        # Request the required source data from XMS
        grid_data = query_helper.get_grid_mapping_data()

        if grid_data['error']:  # Had a problem, abort and try to unlink the source coverage
            self._log.info('Problem retriving grid information.')
            return ''
        # self._sim_data = SimData(self._main_file)
        dxs = grid_data['mapped_grids'].data.locations.values
        with open(self._shdxfile, 'w') as f:
            f.write('************************************************\n')
            f.write(' DX SHORELINE DATA FOR VARIABLE GRID FOR GENCADE:\n')
            f.write(f' {len(dxs)} Sizes - TITLE: Fix this')
            f.write('************************************************\n')
            for i, a in enumerate(dxs):
                f.write(f' {a}')
                if i % 10 == 0:
                    f.write('\n')

        return self._shdxfile
