"""Code to copy a GMI coverage."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import copy
from pathlib import Path
import shutil
from typing import Sequence, Type, TypeVar
import uuid

# 2. Third party modules
from geopandas import GeoDataFrame

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.display.display_options_io import write_display_option_ids
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component, Coverage
from xms.gmi.components.coverage_component import (
    ARC_JSON, CoverageComponent, INITIAL_ARC_ATT_ID_FILE, INITIAL_ARC_COMP_ID_FILE, INITIAL_POINT_ATT_ID_FILE,
    INITIAL_POINT_COMP_ID_FILE, INITIAL_POLYGON_ATT_ID_FILE, INITIAL_POLYGON_COMP_ID_FILE, PT_JSON
)
from xms.guipy.data.target_type import TargetType
from xms.tool_core.coverage_writer import CoverageWriter

# 4. Local modules

# Type aliases
# See https://stackoverflow.com/a/71441339/5666265.
CcType = TypeVar('CcType', bound=CoverageComponent)  # To stop PyCharm's type checking from complaining (but doesn't)
"""Alias for a type derived from GMI CoverageComponent."""

DisplayIds = dict[TargetType, tuple[Sequence[int], Sequence[int]]]
"""Alias for a dict of target type -> tuple[feature ids, component ids]."""


def copy_coverage_and_component(coverage: GeoDataFrame, cov_comp: Type[CcType],
                                query: Query) -> tuple[GeoDataFrame, CcType]:
    """Copies the coverage and GMI coverage component.

    Args:
        coverage: The coverage.
        cov_comp: The coverage component.
        query: Object for communicating with XMS

    Returns:
        The new coverage and the new coverage component.
    """
    new_coverage = copy_coverage(coverage)
    new_cov_comp = copy_coverage_component(cov_comp, new_coverage.attrs['uuid'], query)
    return new_coverage, new_cov_comp


def copy_coverage(coverage: GeoDataFrame) -> GeoDataFrame:
    """Returns a copy of the new coverage.

    Args:
        coverage: The coverage.

    Returns:
        See description.
    """
    new_coverage = copy.deepcopy(coverage)
    new_coverage.attrs['uuid'] = str(uuid.uuid4())
    return new_coverage


def copy_coverage_component(cov_comp: Type[CcType], new_cov_uuid: str, query: Query) -> CcType:
    """Copies the coverage and the GMI coverage component.

    Args:
        cov_comp: The coverage component.
        new_cov_uuid: Uuid of the new coverage.
        query: Object for communicating with XMS

    Returns:
        The new coverage component.
    """
    new_cov_comp = _copy_component(cov_comp, new_cov_uuid, query)
    _set_update_ids(new_cov_comp, new_cov_uuid)
    return new_cov_comp


def add_to_query(
    coverage: GeoDataFrame, cov_comp: Type[CcType], unique_name: str, coverage_type: str, model_name: str, query: Query
) -> None:
    """Adds the coverage and component to the query.

    Args:
        coverage: The coverage.
        cov_comp: The coverage component.
        unique_name: Component unique_name from xml
        coverage_type: From <declare_coverage name=>
        model_name: Model name from xml
        query: Object for communicating with XMS
    """
    pt_json = str(Path(cov_comp.main_file).parent / PT_JSON)
    arc_json = str(Path(cov_comp.main_file).parent / ARC_JSON)
    cov_comp.display_option_list = [
        XmsDisplayMessage(file=pt_json, edit_uuid=coverage.attrs['uuid']),  # points
        XmsDisplayMessage(file=arc_json, edit_uuid=coverage.attrs['uuid']),  # arcs
    ]
    do_comp = make_do_cov_comp(cov_comp.main_file, unique_name, model_name, cov_comp.uuid)
    cov_filename = filesystem.temp_filename()
    writer = CoverageWriter(cov_filename)
    writer.write(coverage)
    do_coverage = Coverage(cov_filename)
    query.add_coverage(
        do_coverage=do_coverage,
        model_name=model_name,
        coverage_type=coverage_type,
        components=[do_comp],
        component_keywords=[
            {
                'component_coverage_ids': [cov_comp.uuid, cov_comp.update_ids],
                'display_options': cov_comp.get_display_options()
            }
        ],
    )


def make_do_cov_comp(main_file: str | Path, unique_name: str, model_name: str, coverage_uuid: str) -> Component:
    """Creates and returns a data_objects Component for a gmi CoverageComponent.

    Args:
        main_file: The main file associated with the component.
        unique_name: Component unique_name from xml.
        model_name: Model name from xml.
        coverage_uuid: Uuid of the coverage.

    Returns:
        See description.
    """
    return Component(main_file=str(main_file), unique_name=unique_name, model_name=model_name, comp_uuid=coverage_uuid)


def initialize_display(cov_comp: Type[CcType], display_ids: DisplayIds) -> None:
    """Initialize the component display.

    Args:
        cov_comp: The coverage component.
        display_ids: dict of target type -> tuple[feature IDs, component IDs].
    """
    cov_comp.update_id_files()
    bc_comp_dir = Path(cov_comp.main_file).parent
    for target_type, ids_tuple in display_ids.items():
        att_id_file, comp_id_file = initial_id_file_names_from_target_type(target_type)
        write_display_option_ids(str(bc_comp_dir / att_id_file), ids_tuple[0])
        write_display_option_ids(str(bc_comp_dir / comp_id_file), ids_tuple[1])


def initial_id_file_names_from_target_type(target_type: TargetType) -> tuple[str, str]:
    """Returns the tuple of initial id file names - feature, component - given the target type.

    Example: TargetType.point -> (INITIAL_POINT_ATT_ID_FILE, INITIAL_POINT_COMP_ID_FILE).

    Args:
        target_type: The feature type.

    Returns:
        See description.
    """
    if target_type == TargetType.point:
        return INITIAL_POINT_ATT_ID_FILE, INITIAL_POINT_COMP_ID_FILE
    elif target_type == TargetType.arc:
        return INITIAL_ARC_ATT_ID_FILE, INITIAL_ARC_COMP_ID_FILE
    elif target_type == TargetType.polygon:
        return INITIAL_POLYGON_ATT_ID_FILE, INITIAL_POLYGON_COMP_ID_FILE
    else:
        raise ValueError()


def _copy_component(cov_comp: Type[CcType], new_cov_uuid: str, query: Query) -> Type[CcType]:
    """Copies a GMI coverage component.

    Args:
        cov_comp: The coverage component.

    Returns:
        The new component.
    """
    # Copy component folder and save component there
    components_dir = Path(cov_comp.main_file).parent.parent
    new_comp_uuid = str(uuid.uuid4())
    new_dir = components_dir / new_comp_uuid
    shutil.copytree(Path(cov_comp.main_file).parent, new_dir)
    cov_comp.save_to_location(str(new_dir), 'DUPLICATE')

    # Create new component
    new_main_file = new_dir / Path(cov_comp.main_file).name
    klass = type(cov_comp)
    new_cov_comp = klass(str(new_main_file))

    # Update IDs
    query.load_component_ids(cov_comp, points=True, arcs=True)
    if cov_comp.cov_uuid in cov_comp.comp_to_xms:
        new_cov_comp.comp_to_xms[new_cov_uuid] = cov_comp.comp_to_xms[cov_comp.cov_uuid]
    new_cov_comp.cov_uuid = new_cov_uuid
    new_cov_comp.data.coverage_uuid = new_cov_uuid
    new_cov_comp.data.commit()  # save the coverage uuid
    return new_cov_comp


def _set_update_ids(new_cov_comp: Type[CcType], new_coverage_uuid: str) -> None:
    """Sets the update_ids for the new component.

    Args:
        new_cov_comp: The new component.
        new_coverage_uuid: New coverage uuid.

    Returns:
        The new component.
    """
    if new_coverage_uuid in new_cov_comp.comp_to_xms:
        cov_dict = new_cov_comp.comp_to_xms[new_coverage_uuid]
        new_cov_comp.update_ids[new_coverage_uuid] = {}
        for target_type, comp_id_dict in cov_dict.items():
            new_cov_comp.update_ids[new_coverage_uuid][target_type] = {}
            for comp_id, att_ids in comp_id_dict.items():
                for att_id in att_ids:
                    new_cov_comp.update_ids[new_coverage_uuid][target_type][att_id] = comp_id

        if TargetType.point not in new_cov_comp.update_ids[new_coverage_uuid]:
            new_cov_comp.update_ids[new_coverage_uuid][TargetType.point] = {}  # pragma no cover - dunno how to test
        if TargetType.arc not in new_cov_comp.update_ids[new_coverage_uuid]:
            new_cov_comp.update_ids[new_coverage_uuid][TargetType.arc] = {}  # pragma no cover - dunno how to test
