"""GsshaSimData class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from packaging.version import parse

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.gmi.data.sim_data import SimData

# 4. Local modules
from xms.gssha.data import sim_generic_model


class GsshaSimData(SimData):
    """Manages the data file for a simulation component."""
    GSSHA_DATA_VERSION = 'GSSHA_DATA_VERSION'  # To track changes to the generic model
    LATEST_VERSION = '0.0.2'  # The latest version

    def __init__(self, data_file: str | Path):
        """
        Initialize the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file.
        """
        # _migrate() will be called by base class.
        data_file_path = Path(data_file)
        self._file_exists = data_file_path.is_file()
        self._can_edit = not self._file_exists or filesystem.is_somewhere_below_system_temp(data_file_path)
        super().__init__(data_file)

        if self._can_edit:
            # Always add the GSSHA_DATA_VERSION string, set to the latest version
            self.info.attrs[self.GSSHA_DATA_VERSION] = self.LATEST_VERSION
            self.commit()

    def _migrate(self):
        """Method to migrate data from different versions of the file."""
        # Don't need to migrate if file didn't exist because it will be created from scratch with latest model
        if not self._file_exists or not self._can_edit:
            return

        version_str = self.info.attrs.get(self.GSSHA_DATA_VERSION, '0.0.0')
        version = parse(version_str)
        if version == parse('0.0.0'):
            gm = sim_generic_model.create(default_values=True)
            gm.global_parameters.restore_values(self.global_values)  # Restore old values into new model
            if gm.global_parameters.has_group('precipitation'):
                parameter = gm.global_parameters.group('precipitation').parameter('hyetograph_xy')
                if parameter:
                    val = parameter.value
                    if not isinstance(val, int):
                        parameter.value = val[1]  # This is the xy series ID
                        self.global_values = gm.global_parameters.extract_values()
        if version == parse('0.0.1'):
            # 'map_tables' was renamed to 'mapping_tables'
            if "'map_tables'" in self.model_values:
                self.model_values = self.model_values.replace("'map_tables'", "'mapping_tables'")
