"""TimeSeriesFileWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.xy.xy_util import XySeriesDict

# 4. Local modules
from xms.gssha.file_io import io_util


def write(gssha_file_path: Path, xy_dict: XySeriesDict, start_date_time: datetime) -> Path | None:
    """Writes the TIME_SERIES_FILE (.xys) file and returns the file path.

    See https://www.gsshawiki.com/File_Formats:Time_and_Elevation_Series_Files

    Args:
        gssha_file_path: .gssha file path.
        xy_dict: The xy series data.
        start_date_time: The starting date/time.
    """
    if not xy_dict:
        return None

    logger = logging.getLogger('xms.gssha')
    logger.info('Writing time series (.xys) file...')
    xys_file_path: Path = gssha_file_path.with_suffix('.xys')
    with open(xys_file_path, 'w') as file:
        for _id, xy_series in xy_dict.items():
            file.write('GSSHA_TS\n')
            file.write(f'"{xy_series.name}"\n')
            file.write('ABSOLUTE\n')
            for x, y in zip(xy_series.x, xy_series.y):
                file.write(f'{io_util.get_time_string(start_date_time, x)} {y}\n')
            file.write('END_TS\n\n')
    return xys_file_path
