"""Creates a link number free draw component."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.api.tree import tree_util
import xms.components.display.display_options_io as dio
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.guipy.testing import testing_tools

# 4. Local modules
from xms.gssha.components import dmi_util
from xms.gssha.components.link_numbers_component import LinkNumbersComponent
from xms.gssha.data import bc_util


class LinkNumberPreviewCreator(FeedbackThread):
    """Creates a free-draw component showing stream arc link numbers.

    Adapted from BcSnapRunner in xmsrsm.
    """
    def __init__(self, query):
        """Constructor.

        Args:
            query (Query): The XMS interprocess communication object
        """
        super().__init__(query)
        self._disp_wkt = query.display_projection.well_known_text
        self._sim_uuid = query.parent_item_uuid()
        self._sim_node = tree_util.find_tree_node_by_uuid(self._query.project_tree, self._sim_uuid)
        rv = dmi_util.find_bc_coverage_and_component(self._sim_node, query)
        self._bc_cov = rv[0]
        self._bc_comp = rv[1]
        self._do_comp = None
        self._arc_link_data = {}
        self._actions = []
        self._arc_files = {}
        self._comp_main_file: Path | None = None
        self._log = logging.getLogger('xms.gssha')
        self.display_text = {
            'title': 'GSSHA Generate Link Numbers View',
            'working_prompt': 'Generating GSSHA link numbers view. Please wait...',
            'warning_prompt': 'Warning(s) encountered while generating link numbers view.',
            'error_prompt': 'Error(s) encountered while generating link numbers view.',
            'success_prompt': 'Successfully generated link numbers view',
            'note': '',
            'auto_load': 'Close this dialog automatically when generation is finished.'
        }

    def _run(self) -> None:
        """Create the free-draw component."""
        self._log.info('Establishing communication with SMS.')
        try:
            self._remove_existing_free_draw_components()
            self._error_check()
            self._compute_link_number_locations()
            self._create_component_folder_and_copy_display_options()
            self._create_drawing()
            self._create_new_component()
            self._add_comp_to_xms()
        except Exception as error:
            self._log.error(str(error))

    @property
    def new_main_file_path(self) -> Path:
        """Returns the new component main file."""
        return self._comp_main_file

    def _remove_existing_free_draw_components(self):
        """Remove any existing free-draw components from the simulation."""
        items = tree_util.descendants_of_type(self._sim_node, xms_types=['TI_COMPONENT'])
        for item in items:
            self._query.delete_item(item.uuid)

    def _error_check(self):
        """Makes sure there is a bc coverages."""
        if self._bc_cov is None:
            self._log.error('No BC Coverages to map.')
            raise RuntimeError

    def _compute_link_number_locations(self):
        """Computes the link number locations."""
        self._log.info(f'Processing coverage: {self._bc_cov.attrs["name"]}')
        self._arc_link_data[self._bc_cov.attrs['uuid']] = {}
        stream_data = bc_util.get_bc_data(self._bc_cov, self._bc_comp, {'channel'})
        arc_links = bc_util.compute_link_numbers(stream_data)  # rv will be ArcLinks or an error string
        if isinstance(arc_links, str):
            raise RuntimeError(arc_links)
        if len(arc_links) != len(stream_data.feature_bc):
            raise RuntimeError('Link number assignment failed. Check that arcs are pointing downstream.')

        for arc, _group in stream_data.feature_bc.items():
            # label_point = arc_util.point_on_arc(arc, 0.5)
            # locs = [*label_point, *label_point]  # Flattened list. Both points are the same
            feature_arc = stream_data.feature_from_id(arc[0], arc[1])
            points = [[point[0], point[1], point[2]] for point in feature_arc.geometry.coords]
            locs = [coord for point in points for coord in point]  # Flattened list.
            self._arc_link_data[self._bc_cov.attrs['uuid']][arc[0]] = {
                'label': str(arc_links.get(arc[0], '-1')),
                'locations': locs
            }

    def _create_component_folder_and_copy_display_options(self):
        """Creates the folder for the mapped bc component and copies the display options from the bc coverage."""
        self._log.info('Creating component folder')
        bc_comp = self._bc_comp
        bc_comp_dir = Path(bc_comp.main_file).parent
        new_comp_uuid = testing_tools.new_uuid()
        new_comp_dir = bc_comp_dir.parent / new_comp_uuid
        new_comp_dir.mkdir()
        disp_path = bc_comp_dir / 'arc_display.json'
        new_path = new_comp_dir / 'arc_display.json'

        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        json_dict = dio.read_display_options_from_json(str(disp_path))
        json_dict['uuid'] = testing_tools.new_uuid()
        json_dict['comp_uuid'] = new_comp_uuid
        json_dict['is_ids'] = 0
        categories.from_dict(json_dict)
        categories.projection = {'wkt': self._disp_wkt}  # This causes trouble?
        for c in categories.categories:
            self._arc_files[c.description] = c.file

        dio.write_display_options_to_json(str(new_path), categories)

        self._comp_main_file = new_comp_dir / 'arc_display.json'

    def _create_drawing(self):
        """Creates the drawing elements for the component."""
        arc_display = {}
        labels = []
        for _, data in self._arc_link_data[self._bc_cov.attrs['uuid']].items():
            arc_display.setdefault('channel', []).append(data['locations'])
            labels.append(data['label'])

        comp_path = self._comp_main_file.parent
        for att_type, locs in arc_display.items():
            att_type = att_type.title()  # Files use title case ('Channel', not 'channel')
            file_path = comp_path / self._arc_files[att_type]
            dio.write_display_option_line_locations(str(file_path), locs, labels)

    def _create_new_component(self):
        """Create the new component for the link numbers."""
        comp_name = 'Link Numbers'
        comp_uuid = self._comp_main_file.parent.name
        self._do_comp = Component(
            main_file=str(self._comp_main_file),
            name=comp_name,
            model_name='GSSHA',
            unique_name='LinkNumbersComponent',
            comp_uuid=comp_uuid
        )
        comp = LinkNumbersComponent(str(self._comp_main_file))
        self._actions = [
            ActionRequest(
                modality='NO_DIALOG',
                main_file=str(self._comp_main_file),
                class_name='LinkNumbersComponent',
                module_name=comp.module_name,
                method_name='get_initial_display_options'
            )
        ]

    def _add_comp_to_xms(self):
        """Adds the mapped component to xms."""
        self._query.add_component(do_component=self._do_comp, actions=self._actions)
        self._query.link_item(taker_uuid=self._sim_uuid, taken_uuid=self._do_comp.uuid)


def create_link_number_preview(query: Query, win_cont: QWidget) -> Path:
    """Export a simulation with a feedback dialog.

    Args:
        query: XMS interprocess communicator
        win_cont: The window container.

    Returns:
        Path to the new component main file.
    """
    thread = LinkNumberPreviewCreator(query)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)
    return thread.new_main_file_path
