"""Dialog for choosing an external text editor application."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import collections
import os
import subprocess
import winreg

# 2. Third party modules
from PySide2.QtWidgets import QFileDialog
import win32api

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.choose_text_editor_dialog_ui import Ui_FileEditorSelectorDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.settings import SettingsManager


def run_text_editor(file_path, win_cont, icon):
    """Opens the file in the application selected, or asks user for the application to use.

    Args:
        file_path (str): Filepath of file to open.
        win_cont (PySide2.QtWidgets.QWidget): The window container.
        icon (PySide2.QtGui.QIcon): Icon to show in the dialog title

    """
    settings = SettingsManager(python_path=False)
    ask = settings.get_setting('Generals', 'AskEd', 1)
    editor = settings.get_setting('Generals', 'DefEd', '')
    valid_editor_selected = os.path.isfile(editor) if editor else False
    if ask or not valid_editor_selected:
        dlg = ChooseTextEditorDialog(parent=win_cont, file_path=file_path, selected_app_path=editor)
        if dlg.exec():
            editor = dlg.get_selected_app()
            valid_editor_selected = os.path.isfile(editor) if editor else False
            if valid_editor_selected:
                settings.save_setting('Generals', 'DefEd', editor)
                settings.save_setting('Generals', 'AskEd', dlg.get_ask_again(), reg_format=winreg.REG_DWORD)
            else:
                app_name = os.environ.get('XMS_PYTHON_APP_NAME')
                message_box.message_with_ok(
                    win_cont, 'No valid text editor was selected.', app_name=app_name, icon='Stop', win_icon=icon
                )
    if valid_editor_selected:
        subprocess.call([editor, file_path])  # Launch the text editor


def get_file_properties(fname):
    """Read all properties of the given file and return them as a dictionary.

    See https://stackoverflow.com/questions/580924/how-to-access-a-files-properties-on-windows
    """
    prop_names = (
        'Comments', 'InternalName', 'ProductName', 'CompanyName', 'LegalCopyright', 'ProductVersion', 'FileDescription',
        'LegalTrademarks', 'PrivateBuild', 'FileVersion', 'OriginalFilename', 'SpecialBuild'
    )

    props = {'FixedFileInfo': None, 'StringFileInfo': None, 'FileVersion': None}

    try:
        # backslash as parm returns dictionary of numeric info corresponding to VS_FIXEDFILEINFO struc
        fixed_info = win32api.GetFileVersionInfo(fname, '\\')
        props['FixedFileInfo'] = fixed_info
        props['FileVersion'] = "%d.%d.%d.%d" % (
            fixed_info['FileVersionMS'] / 65536, fixed_info['FileVersionMS'] % 65536,
            fixed_info['FileVersionLS'] / 65536, fixed_info['FileVersionLS'] % 65536
        )

        # \VarFileInfo\Translation returns list of available (language, codepage)
        # pairs that can be used to retreive string info. We are using only the first pair.
        lang, codepage = win32api.GetFileVersionInfo(fname, '\\VarFileInfo\\Translation')[0]

        # any other must be of the form \StringfileInfo\%04X%04X\parm_name, middle
        # two are language/codepage pair returned from above

        str_info = {}
        for prop_name in prop_names:
            str_info_path = u'\\StringFileInfo\\%04X%04X\\%s' % (lang, codepage, prop_name)
            str_info[prop_name] = win32api.GetFileVersionInfo(fname, str_info_path)

        props['StringFileInfo'] = str_info
    except Exception:
        pass

    return props


class ChooseTextEditorDialog(XmsDlg):
    """A dialog to let the user select which app to open a file with."""
    def __init__(self, parent=None, file_path='', selected_app_path=''):
        """Initializes the dialog.

        Args:
            parent (QObject): The dialog's Qt parent
            file_path (str): Filepath of file being opened for editing/viewing.
            selected_app_path (str): Filepath to application selected to open the file with.
        """
        super().__init__(parent, 'xmsmf6.gui.file_editor_selector_dialog')
        self._file_path = file_path
        self._selected_app_path = selected_app_path
        self._app_paths = []  # List of paths to executables, same size as and parallel to num combo_box items

        self.AppInfo = collections.namedtuple('AppInfo', 'file_type app_description app_path')

        # Init ui from .ui file
        self.ui = Ui_FileEditorSelectorDialog()
        self.ui.setupUi(self)

        # Set up widgets
        self.ui.txtFilePath.setText(self._file_path)
        self.setup_combo_box()

        # Signals
        self.ui.btnFindOther.clicked.connect(self.on_btn_find_other)

    def setup_combo_box(self):
        """Sets up the combo box of apps."""
        self.add_system_apps()
        self.add_selected_app()

    def add_system_apps(self):
        """Add system apps to the combo box."""
        extensions = {".txt", ".wri", ".rtf", ".doc", ".wpd", ".htm", ".html"}
        file_types = self.file_types_from_extensions(extensions)
        app_info = self.app_info_from_file_types(file_types)
        for item in app_info:
            self._app_paths.append(item.app_path)
            self.ui.cbxEditors.addItem(item.app_description)

    def add_selected_app(self):
        """Adds current selected app if it isn't already in the list."""
        if self._selected_app_path:
            props = get_file_properties(self._selected_app_path)
            if 'StringFileInfo' in props and props['StringFileInfo'] and 'FileDescription' in props['StringFileInfo']:
                app_description = props['StringFileInfo']['FileDescription']
                if self.ui.cbxEditors.findText(app_description) == -1:
                    self._app_paths.append(self._selected_app_path)
                    self.ui.cbxEditors.addItem(app_description)
                    self.ui.cbxEditors.setCurrentText(app_description)

    def app_info_from_file_types(self, file_types):
        """Returns a list of self.AppInfo named tuples containing the app path and description for each file type.

        Args:
            file_types

        Returns:
            See description.

        """
        ftype_output = subprocess.check_output('ftype', shell=True)  # Runs system command 'ftype'
        app_info = []
        for line in ftype_output.splitlines():
            words = line.decode('utf-8').split('=', 2)
            if words and words[0] in file_types and len(words) > 1:
                app_path = self.extract_path(words[1])
                full_app_path = self.expand_environment_variables(app_path)
                props = get_file_properties(full_app_path)
                if 'StringFileInfo' in props and props['StringFileInfo'] and\
                        'FileDescription' in props['StringFileInfo']:
                    app_info.append(
                        self.AppInfo(
                            file_type=words[0],
                            app_description=props['StringFileInfo']['FileDescription'],
                            app_path=full_app_path
                        )
                    )
                    if len(app_info) >= len(file_types):
                        break
        return app_info

    def file_types_from_extensions(self, extensions):
        """Return a set of file types associated with the given extensions.

         If an extension has not associated file type, there will be nothing for it in the set.

        Args:
            extensions (set{str}): Set of extensions, including the '.', e.g. ['.txt', '.doc'].

        Returns:
            See description.

        """
        file_types = set()
        assoc_output = subprocess.check_output('assoc', shell=True)  # Runs system command 'assoc'
        for line in assoc_output.splitlines():
            words = line.decode('utf-8').split('=', 2)
            if words and words[0] in extensions and len(words) > 1:
                file_types.add(words[1])
        return file_types

    def extract_path(self, word):
        """Returns the path to the executable from the command line.

        If there are spaces in the path, the path will be quoted and we can use that to find the path. If

        Args:
            word (str): The command line.

        Returns:
            See description.
        """
        if word.startswith('"'):
            app_path = word[0:word.find('"', 1)]  # Remove everything after quoted path
        else:
            app_path = word.split(' ')[0]
        return app_path

    def expand_environment_variables(self, app_path):
        """Replace %SystemRoot% etc with the actual path.

        Args:
            app_path:

        Returns:
            The app_path with the environment variable replaced with the actual path.

        """
        if app_path.startswith('%'):
            env_var = app_path[0:app_path.find('%', 1) + 1]
            if env_var:
                env_var_expanded = os.path.expandvars(env_var)
                full_app_path = app_path.replace(env_var, env_var_expanded)
                return full_app_path
        return app_path

    def on_btn_find_other(self):
        """Open the Open File dialog so user can select an executable."""
        self._selected_app_path, _ = QFileDialog.getOpenFileName(
            self, 'Select File Editor', self._selected_app_path, 'Executable Files (*.exe)'
        )
        if self._selected_app_path:
            props = get_file_properties(self._selected_app_path)
            if 'FileDescription' in props['StringFileInfo']:
                description = props['StringFileInfo']['FileDescription']
                self._app_paths.append(self._selected_app_path)
                self.ui.cbxEditors.addItem(description)
                self.ui.cbxEditors.setCurrentText(description)

    def get_selected_app(self):
        """Returns the path of the currently selected application in the combo box.

        Returns:
            See description.

        """
        if self.ui and self.ui.cbxEditors and self._app_paths:
            current_index = self.ui.cbxEditors.currentIndex()
            if current_index < len(self._app_paths):
                return self._app_paths[current_index]

    def get_ask_again(self):
        """Returns the state of the 'Don't ask again' toggle.

        Returns:
            See description.

        """
        if self.ui and self.ui.togNeverAskAgain:
            return 0 if self.ui.togNeverAskAgain.isChecked() else 1

    def accept(self):
        """Called when the OK button is clicked."""
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        super().reject()
