"""Event filter for formatting numbers in edit fields for XMS Python dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import re

# 2. Third party modules
from PySide2.QtCore import QEvent, QObject, Signal
from PySide2.QtGui import QValidator

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.validators.qx_locale import QxLocale


class NumberCorrector(QObject):
    """Event filter for formatting numbers in edit fields for XMS Python dialogs."""
    MERICA_LOCALE = QxLocale
    OLD_DEFAULT_PRECISION = 10
    DEFAULT_PRECISION = -1
    text_corrected = Signal()

    def __init__(self, parent=None):
        """Construct the event filter."""
        super().__init__(parent)

    def eventFilter(self, obj, event):  # noqa: N802
        """Validate text as it is being inputted.

        Args:
            obj (QObject): Qt object to handle event for
            event (QEvent): The Qt event to handle

        Returns:
            (bool): True if the event was handled
        """
        try:
            event_type = event.type()
            if event_type in [QEvent.FocusOut, QEvent.Close] and obj:
                valid = obj.validator()
                is_dbl = valid.decimals() != 0
                curr_text = obj.text()
                if not curr_text:
                    curr_text = '0.0' if is_dbl else '0'
                    obj.setText(curr_text)
                    obj.setModified(True)
                state, val, ok = obj.validator().validate(obj.text(), 0)
                if is_dbl and (state == QValidator.Intermediate or state == QValidator.Invalid):
                    # Change the text to be something valid
                    prec = valid.decimals()
                    if prec < 0:
                        prec = NumberCorrector.OLD_DEFAULT_PRECISION
                    current, ok = NumberCorrector.MERICA_LOCALE.toDouble(obj.text())
                    if ok:  # Is a valid double, but out of range
                        if current < valid.bottom():
                            obj.setText(f'{valid.bottom():.{prec}f}')
                        elif prec > valid.top():
                            obj.setText(f'{valid.top():.{prec}f}')
                        else:
                            obj.setText(f'{current:f}')
                    else:
                        obj.undo()
                    obj.setModified(True)
                if is_dbl:
                    # add/trim trailing zeros as needed
                    current, ok = NumberCorrector.MERICA_LOCALE.toDouble(obj.text())
                    s = self.format_double(current)
                    obj.setText(s)
                    obj.setModified(True)
                    self.text_corrected.emit()
                if not is_dbl and (state == QValidator.Intermediate or state == QValidator.Invalid):  # Integers
                    current, ok = NumberCorrector.MERICA_LOCALE.toInt(obj.text())
                    if ok:  # Is a valid double, but out of range
                        if current <= valid.bottom():
                            obj.setText(NumberCorrector.MERICA_LOCALE.toString(valid.bottom()))
                        else:
                            obj.setText(NumberCorrector.MERICA_LOCALE.toString(valid.top()))
                    else:
                        obj.setText(NumberCorrector.MERICA_LOCALE.toString(valid.bottom()))
                    obj.setModified(True)
        except AttributeError:
            # this can get called on a QWidget that doesn't have the validator(), text() and other methods
            # for these objects just pass the event on
            pass
        return super().eventFilter(obj, event)

    @staticmethod
    def format_double(value: float, prec: int = -1, version: int = 3) -> str:
        """Returns a double as a formatted string.

        Args:
            value: Number to format
            prec: Maximum number of significant figures to include in the string (switch to scientific notation if
             needed).
            version: Version to use (1, or 2 are the options right now. See test_format_double())

        Returns:
            (str): See description
        """
        if version == 1:
            # This makes numbers like 1e25 appear like '10000000000000000905969664.0'
            prec = NumberCorrector.OLD_DEFAULT_PRECISION if prec == -1 else prec
            display_text = f'{value:.{prec}f}'
            display_text = re.sub('0+$', '', display_text)
            display_text = normalize_trailing_zeros(display_text)

        elif version == 2:
            # Like using 'g' but switches between e and f better and better formatting
            # Idea from https://stackoverflow.com/questions/4626338
            prec = NumberCorrector.OLD_DEFAULT_PRECISION if prec == -1 else prec
            if abs(value) == 0.0:
                display_text = '0.0'
            else:
                log_value = math.log10(abs(value))
                if log_value < -prec or log_value > prec:
                    # Use scientific notation
                    display_text = f'{value:.{prec}e}'
                    display_text = re.sub('0(0*)e', '0e', display_text)  # '1.000000000e+41 -> 1.0e+41
                else:
                    # Use floating point notation
                    display_text = f'{value:.{prec}f}'.rstrip('0')
                    display_text = normalize_trailing_zeros(display_text)
        elif version == 3:
            return format_float(value, prec)
        else:
            raise ValueError(f'Invalid version: {version}')
        return display_text


def normalize_trailing_zeros(number_string: str) -> str:
    """Ensures the number string ends with a decimal point and a zero (".0").

    Args:
        number_string: A number formatted as a floating point string (not scientific notation).

    Returns:
        The string.
    """
    if '.' not in number_string:
        number_string += '.0'
    elif number_string.endswith('.'):
        number_string += '0'
    else:
        # We have a '.'. Check for trailing zeros.
        while number_string.endswith('0') and not number_string.endswith('.0'):
            number_string = number_string[:-1]
    return number_string


def format_float(x: float, decimals: int = -1, max_width: int = 15) -> str:
    """
    Formats a number to a string which can be no longer than max width.

    Does not care about precision. Tries to match the C++ code in xmscore/misc/StringUtil.cpp.

    Args:
        x: The number to format.
        decimals: Maximum number of digits after the decimal point. -1 means any number of decimal places is allowed.
        max_width: The maximum width of the string. Default is 15.

    Returns:
        The string.
    """
    if abs(x) == 0.0:
        return '0.0'

    # determine if scientific notation is needed
    mx = 10.0**(max_width - 3)
    mn = 1e-5
    use_sci_not = abs(x) >= mx or (0.0 < abs(x) < mn)

    if not use_sci_not:  # floating point notation
        p = max_width if decimals == -1 else decimals
        o = f'{x:#{max_width - 1}.{p}f}'.lstrip()
        o = normalize_trailing_zeros(o)

        # float longer than max_width will need rounding to fit max_width
        if len(o) > max_width:
            o = str(round(x, max_width - 1 - str(x).index('.')))
    else:  # scientific notation
        o = f'{x:#{max_width}e}'.lstrip()
        o = re.sub('0(0*)e', '0e', o)  # '1.000000000e+41 -> 1.0e+41
    return o
