"""QPushButton widget for selecting a color."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QPoint, QSize, Signal, Slot
from PySide2.QtGui import QColor, QIcon, QPixmap
from PySide2.QtWidgets import QColorDialog, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.widgets.color_button_ui import Ui_ColorButton
from xms.guipy.widgets.color_popout import ColorPopout


class ColorButton(QWidget):
    """QPushButton implementation for selecting a color."""
    color_changed = Signal()
    current_color_changed = Signal(QColor)
    color_pushed = Signal()

    def __init__(self, parent=None, transparency=False):
        """Initializes the widget.

        Args:
            parent (Something derived from QWidget): The parent window.
            transparency (bool): True is transparency is allowed.
        """
        super().__init__(parent)

        self.ui = Ui_ColorButton()
        self.ui.setupUi(self)
        self.color = QColor()
        self.transparency = transparency

        self.ui.btn_color.clicked.connect(self.color_clicked)
        self.customContextMenuRequested[QPoint].connect(self.draw_color_popup)
        self.ui.btn_arrow.clicked.connect(self.draw_color_popup_arrow)
        self.current_color_changed[QColor].connect(self.color_changed)

    @Slot()
    def color_clicked(self):
        """Called when a button that is invokes the color dialog is clicked."""
        self.color_pushed.emit()
        if self.transparency:
            new_color = QColorDialog.getColor(self.color, self, options=QColorDialog.ShowAlphaChannel)
        else:
            new_color = QColorDialog.getColor(self.color, self)

        if new_color.isValid():
            self.set_color(new_color)

    @Slot()
    def draw_color_popup_arrow(self):
        """Creates a ColorPopout by the arrow button."""
        self.color_pushed.emit()
        pos = self.ui.btn_arrow.pos()
        pos.setY(pos.y() + self.ui.btn_arrow.height())
        self.draw_color_popup(pos)

    @Slot(QPoint)
    def draw_color_popup(self, pos):
        """Creates a ColorPopout at the position.

        Args:
            pos (QPoint): Position where the ColorPopout is to be drawn.
        """
        popup = ColorPopout(self.mapToGlobal(pos), self)
        popup.color_changed[QColor].connect(self.set_color)
        popup.color_changed[QColor].connect(popup.close)
        popup.show()
        popup.more_colors_clicked.connect(self.color_clicked)

    @Slot(QColor)
    def set_color(self, new_color):
        """Sets the current color. This is called when the user chooses a new color.

        Args:
            new_color (QColor): The new current color.
        """
        self.color = new_color
        self.ui.btn_color.setIcon(self.get_color_icon())
        self.current_color_changed.emit(self.color)

    def get_color_icon(self):
        """Draws an icon of the current color.

        Returns:
            symbol (QIcon): The square color icon.
        """
        icon = QIcon()
        size = QSize(16, 16)
        pix = QPixmap(size)
        pix_disabled = QPixmap(size)
        color = QColor(self.color)
        disabled_color = QColor(0, 0, 0, 0)
        color.setAlpha(255)
        pix.fill(color)
        pix_disabled.fill(disabled_color)
        icon.addPixmap(pix, QIcon.Active)
        icon.addPixmap(pix, QIcon.Normal)
        icon.addPixmap(pix, QIcon.Selected)
        icon.addPixmap(pix_disabled, QIcon.Disabled)
        return icon
