"""QTableView implementation."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import re

# 2. Third party modules
from PySide2.QtCore import (
    QAbstractProxyModel,
    QEvent,
    QItemSelection,
    QItemSelectionModel,
    QModelIndex,
    QPoint,
    QRect,
    QSize,
    Qt,
    Signal,
)
from PySide2.QtGui import QKeyEvent, QKeySequence, QPainter, QPen
from PySide2.QtSql import QSqlTableModel
from PySide2.QtWidgets import QApplication, QTableView

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs import dialog_util
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.widgets.check_box_header import CheckBoxHeader, update_data_from_header, update_header_from_data


class QxTableView(QTableView):
    """QTableView implementation for use in XMS packages."""
    pasted = Signal()

    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent)
        self.pasting = False
        self.size_to_contents = False
        self.paste_delimiter = '\t'  # Overwrite if want to support pasting text that is not tab delimited.

        # Set this to False if you don't want to set columns with combobox delegates as combobox columns in the model.
        self.set_cbx_columns_in_model = True
        self.paste_errors = []
        self.use_header_checkboxes: bool = False
        self._checkbox_header: CheckBoxHeader | None = None

        # Drag-fill schtuff
        self.allow_drag_fill = False  # Set this to True to allow for drag-fill of cells like Excel (kind of).
        self.dragging = False
        self.start_drag_index = None
        self.end_drag_index = None
        self.initial_selection = []

    """Drag-fill overrides"""

    def mousePressEvent(self, event: QEvent) -> None:  # noqa: N802
        """Override to draw drag outline.

        Args:
            event: The event
        """
        selected_indexes = self._gripper_indexes(event)
        # Only enter drag-fill mode if a box is selected
        if selected_indexes and self._selection_forms_box(selected_indexes):
            self.initial_selection = selected_indexes
            self.start_drag_index = self.initial_selection[0]
            self.dragging = True
        super().mousePressEvent(event)

    def mouseMoveEvent(self, event: QEvent) -> None:  # noqa: N802
        """Override to draw drag outline.

        Args:
            event: The event
        """
        super().mouseMoveEvent(event)  # Call base first, or it will mess up the selection when dragging.
        if self.dragging:
            index = self.indexAt(event.pos())
            if index.isValid():
                self.end_drag_index = index  # Last cell we have selected with the drag
                item_selection = QItemSelection(self.start_drag_index, self.end_drag_index)
                self.clearSelection()  # If dragging, fix the current selection to include the initial selection.
                self.selectionModel().select(item_selection, QItemSelectionModel.Select)

    def mouseReleaseEvent(self, event: QEvent) -> None:  # noqa: N802
        """Override to update data after a drag-fill operation.

        Args:
            event: The event
        """
        valid_start = self.start_drag_index and self.start_drag_index.isValid()
        valid_end = self.end_drag_index and self.end_drag_index.isValid()
        if self.dragging and self.initial_selection and valid_start and valid_end:
            self._fill_data()  # Update the data for the cells selected in the drag.
        self._reset_dragging()
        super().mouseReleaseEvent(event)

    def mouseDoubleClickEvent(self, event: QEvent) -> None:  # noqa: N802
        """Override to fill a column when double-clicking in a cell's drag-fill gripper.

        Args:
            event: The event
        """
        selected_indexes = self._gripper_indexes(event)
        if selected_indexes:
            self._fill_column(self.indexAt(event.pos()))
        else:
            super().mouseDoubleClickEvent(event)

    def selectionChanged(self, selected, deselected):  # noqa: N802
        """Override to clean up after drag-fill.

        Args:
            selected (QModelIndex): The selected items
            deselected (QModelIndex): The deselected items
        """
        super().selectionChanged(selected, deselected)
        if self.allow_drag_fill:  # If we allow drag-fill, update the viewport to clean out artifacts.
            self.viewport().update()

    def paintEvent(self, event):  # noqa: N802
        """Override to paint a border around selected cells when drag-fill is enabled.

        Args:
            event (QPaintEvent): The event
        """
        super().paintEvent(event)

        selected_indexes = self.selectedIndexes()
        if not self.allow_drag_fill or len(selected_indexes) == 0:
            return  # Don't need to do anything special.

        if not self.dragging and not self._selection_forms_box(selected_indexes):
            return  # Don't draw the drag-fill border if the selection is not a box. If already dragging, it should be.

        # Draw a border around the selected cells.
        last_selected_rect = self.visualRect(selected_indexes[-1])
        rect = self.visualRect(selected_indexes[0]) | last_selected_rect
        painter = QPainter(self.viewport())
        pen = QPen(Qt.black, 2)
        painter.setPen(pen)
        painter.drawRect(rect)
        # Draw the gripper on the last selected cell.
        gripper_rect = self._drag_fill_gripper(last_selected_rect)
        painter.fillRect(gripper_rect, Qt.black)

    def _reset_dragging(self):
        """Reset the variables used for drag-fill."""
        self.dragging = False
        self.start_drag_index = None
        self.end_drag_index = None
        self.initial_selection = []

    def _drag_fill_gripper(self, rect):
        """Adjust a QRect of a cell to the dimensions of the gripper for that cell.

        Args:
            rect (QRect): The cell's model index
        """
        # This gives you a square in the lower right corner of the cell with sides GRIPPER_FACTOR * height of the font.
        gripper_factor = 0.3
        shift = self.fontMetrics().height() * gripper_factor
        br = rect.bottomRight()
        return QRect(QPoint(br.x() - shift, br.y() - shift), QPoint(br.x(), br.y()))

    def _gripper_indexes(self, event):
        """Get the indexes of the selected cells when a mouse is clicked in the drag-fill gripper.

        Args:
            event (QEvent): The event

        Returns:
            (list[QModelIndex]): The selected indexes, or empty list if the click is outside the drag-fill gripper.
        """
        if self.allow_drag_fill and event.button() == Qt.LeftButton:
            index = self.indexAt(event.pos())
            if index.isValid():
                cell_rect = self.visualRect(index)
                gripper_rect = self._drag_fill_gripper(cell_rect)
                if gripper_rect.contains(event.pos()):  # Only enter drag-fill mode if the cursor is in the gripper.
                    return self.selectedIndexes()
        return []

    def _fill_data(self):
        """Fill values with after a drag."""
        start_value = self.model().data(self.initial_selection[-1], role=Qt.UserRole)
        # If more than one cell is selected when starting drag, use the difference between the last two
        # to increment the value assigned to subsequent cells selected in the drag.
        try:
            delta = float(self.initial_selection[-1].data()) - float(self.initial_selection[-2].data())
        except Exception:
            delta = 0.0  # At least one of last two selected cells is not a numerical item or only one cell selected.

        start_row = self.start_drag_index.row()
        end_row = self.end_drag_index.row()
        start_col = self.start_drag_index.column()
        end_col = self.end_drag_index.column()
        for row in range(min(start_row, end_row), max(start_row, end_row) + 1):
            for col in range(min(start_col, end_col), max(start_col, end_col) + 1):
                index = self.model().index(row, col)
                if index in self.initial_selection:
                    continue  # Don't change values of the initially selected cells.

                try:  # Try to append the delta and cast back to the original data type.
                    start_value = type(start_value)(float(start_value) + delta)
                except Exception:
                    pass  # start_value must not be a number

                orig_value = self.model().data(index)
                try:  # Try to update the value of the cell in the model.
                    self.model().setData(index, start_value)
                except Exception:
                    self.model().setData(index, orig_value)  # Restore the original value.

    def _fetch_all(self) -> None:
        """Fetch all the data.

        At least for SQL models, which don't load all the rows into memory, we sometimes need to load all the rows into
        memory. canFetchMore() and fetchMore() have different signatures for other model types, but it might also be a
        problem there.
        """
        while self.model().canFetchMore():
            self.model().fetchMore()

    def _fill_column(self, index):
        """Fill values in a column after a double click in the drag-fill gripper.

        Args:
            index (QModelIndex): The index of the cell that was double-clicked
        """
        if not index.isValid():
            return
        with dialog_util.wait_cursor_context():
            # At least for SQL models, we need to load all the rows into memory or we won't be able to change them.
            # canFetchMore() and fetchMore() have different signatures for other model types, but it might also
            # be a problem there.
            if isinstance(self.model(), QSqlTableModel):
                self._fetch_all()

            value = self.model().data(index)
            for row in range(index.row() + 1, self.model().rowCount()):
                this_idx = self.model().index(row, index.column())
                orig_value = self.model().data(this_idx)
                try:  # Try to update the value of the cell in the model.
                    self.model().setData(this_idx, value)
                except Exception:
                    self.model().setData(index, orig_value)  # Restore the original value.

    def _selection_forms_box(self, indexes: list[QModelIndex]) -> bool:
        """Checks if a list of QModelIndex objects forms a rectangular box in a QTreeView.

        Args:
            indexes: A list of QModelIndex objects to check.

        Returns:
            (bool): True if the indexes form a box, False otherwise.
        """
        if not indexes:
            return False
        elif len(indexes) == 1:
            return True  # A single index is considered a box

        # Calculate the expected number of indexes in a box
        rows = set(index.row() for index in indexes)
        columns = set(index.column() for index in indexes)
        expected_count = len(rows) * len(columns)
        if len(indexes) != expected_count:
            return False

        # Check for gaps in the selected rows and columns
        if len(set(range(min(rows), max(rows) + 1)).difference(rows)) > 0:
            return False
        if len(set(range(min(columns), max(columns) + 1)).difference(columns)) > 0:
            return False

        return True

    """end Drag-fill overrides"""

    def sizeHint(self):  # noqa: N802
        """Returns the size hint. Overridden to size width to contents.

        From https://stackoverflow.com/questions/6337589/qlistwidget-adjust-size-to-content

        Returns:
            See description.
        """
        if not self.size_to_contents:
            return super().sizeHint()
        else:
            s = QSize()

            # Height
            s.setHeight(super().sizeHint().height())

            # Width
            buffer = 5  # Looks better with a buffer
            s.setWidth(self._compute_width() + buffer)
            return s

    def _compute_width(self) -> int:
        """Compute and return the total width of the table."""
        horz_header = self.horizontalHeader()
        count = horz_header.count()
        row_width = 0
        for i in range(count):
            if not horz_header.isSectionHidden(i):
                row_width += horz_header.sectionSize(i)
        return row_width

    def _move_and_select(self, row: int, column: int | None) -> None:
        """Move to the given row and column in the table.

        Args:
            row: The row.
            column: The column. If None, column is obtained from the current selection.
        """
        if column is None:
            # Get column from selection
            selection = self.selectionModel().selection()
            column = selection.first().left() if selection else 0

        # Set new current index
        index = self.model().index(row, column)
        self.setCurrentIndex(index)

        # Select the cell
        self.clearSelection()
        item_selection = QItemSelection(index, index)
        self.selectionModel().select(item_selection, QItemSelectionModel.Select)

    def _move_to_first_row(self) -> None:
        """Move to the first row in the table."""
        row_count = self.model().rowCount()
        if row_count > 0:
            self._move_and_select(0, None)

    def _move_to_last_row(self) -> None:
        """Move to the last row in the table."""
        row_count = self.model().rowCount()
        if row_count > 0:
            self._move_and_select(row_count - 1, None)

    def _move_to_start(self) -> None:
        """Move to the first row and column in the table."""
        row_count = self.model().rowCount()
        if row_count > 0:
            self._move_and_select(0, 0)

    def _move_to_end(self) -> None:
        """Move to the last row and column in the table."""
        row_count = self.model().rowCount()
        if row_count > 0:
            self._move_and_select(row_count - 1, self.model().columnCount() - 1)

    def keyPressEvent(self, event: QKeyEvent):  # noqa: N802
        """To handle key press events like: copy, paste, insert, delete, etc.

         From https://www.walletfox.com/course/qtableviewcopypaste.php

        Args:
            event (QKeyEvent): The event.
        """
        selected_rows = self.selectionModel().selectedRows()
        # at least one entire row selected
        handled = False
        if selected_rows:
            if event.key() == Qt.Key_Insert:
                self.model().insertRows(selected_rows[0].row(), len(selected_rows))
                handled = True
            elif event.key() == Qt.Key_Delete:
                self.model().removeRows(selected_rows[0].row(), len(selected_rows))
                handled = True

        # at least one cell selected
        if not handled and self.selectedIndexes():
            if event.key() == Qt.Key_Delete:
                selected_indexes = self.selectedIndexes()
                for index in selected_indexes:
                    self.model().setData(index, '')
            elif event.matches(QKeySequence.Copy):
                self.on_copy()
            elif event.matches(QKeySequence.Paste):
                self.on_paste()
            elif _ctrl_up(event):
                self._move_to_first_row()
            elif _ctrl_down(event):
                if isinstance(self.model(), QSqlTableModel):
                    self._fetch_all()
                self._move_to_last_row()
            elif event.matches(QKeySequence.MoveToStartOfDocument):
                self._move_to_start()
            elif event.matches(QKeySequence.MoveToEndOfDocument):
                if isinstance(self.model(), QSqlTableModel):
                    self._fetch_all()
                self._move_to_end()
            else:
                QTableView.keyPressEvent(self, event)

    def _can_paste(self) -> bool:
        """Returns True if pasting is allowed."""
        # See if all columns are marked as read only
        model = self.model()
        if hasattr(model, 'read_only_columns') and model.read_only_columns:
            for column_index in range(model.columnCount()):
                if column_index not in model.read_only_columns:
                    return True
            return False
        return True

    def on_paste(self):
        """Pastes data from the clipboard into the selected cells."""
        if not self._can_paste():
            return

        self.pasting = True
        self.paste_errors = []
        text = QApplication.clipboard().text()
        clipboard_rows = list(filter(None, text.split("\n")))
        init_index = self.selectedIndexes()[0]
        init_row = init_index.row()
        init_col = init_index.column()

        # Insert rows if necessary
        if self.model().rowCount() < init_row + len(clipboard_rows):
            count = init_row + len(clipboard_rows) - self.model().rowCount()
            self.model().insertRows(self.model().rowCount(), count)

        selected_count = len(self.selectedIndexes())
        row_offset = 0
        if len(clipboard_rows) == 1 and selected_count > 1:
            # Paste one row into multiple selected rows by repeatedly pasting the one row
            last_index = self.selectedIndexes()[-1]
            last_row = last_index.row()
            for row in range(init_row, last_row + 1):
                self.paste_row(
                    clipboard_index=0,
                    clipboard_rows=clipboard_rows,
                    init_row=row,
                    init_col=init_col,
                    row_offset=row_offset
                )
        else:
            # Paste one or more rows into the table (doesn't matter how many are selected)
            for i in range(len(clipboard_rows)):
                self.paste_row(i, clipboard_rows, init_row, init_col, row_offset)

        self.pasting = False
        if self.paste_errors:
            msg = 'Errors occurred when pasting data.'
            details = '\n'.join(self.paste_errors)
            self.paste_errors = []
            app_name = os.environ.get('XMS_PYTHON_APP_NAME', '')
            message_with_ok(
                parent=self.window(), message=msg, app_name=app_name, icon='Error', win_icon=None, details=details
            )
        self.model().dataChanged.emit(QModelIndex(), QModelIndex())  # Needed to update the table view
        self.pasted.emit()

    def paste_row(self, clipboard_index, clipboard_rows, init_row, init_col, row_offset):
        """Paste a row to the table.

        Args:
            clipboard_index (int): Index of row being pasted.
            clipboard_rows (list[str]): Rows being pasted.
            init_row (int): Upper left index of table row where we're pasting.
            init_col (int): Upper left index of table column where we're pasting.
            row_offset (int): Increases as hidden rows are skipped.
        """
        # Skip hidden rows
        row = init_row + clipboard_index + row_offset
        while self.isRowHidden(row):
            row_offset += 1
            row += 1

        column_contents = re.split(self.paste_delimiter, clipboard_rows[clipboard_index])
        column_offset = 0
        for j in range(len(column_contents)):

            # Skip hidden columns
            col = init_col + j + column_offset
            while self.isColumnHidden(col):
                column_offset += 1
                col += 1

            if row < self.model().rowCount() and col < self.model().columnCount():
                if not self.model().setData(self.model().index(row, col), column_contents[j]):
                    self.paste_errors.append(f'Error setting data in row: {row + 1}, column: {col + 1}')

    def on_copy(self):
        """Copies data from the selected cells to the clipboard."""
        text = ''
        tab = '\t'
        # For some reason the following crashes so we do it one at a time
        # selection_range = self.selectionModel().selection().first()
        selection_model = self.selectionModel()
        selection = selection_model.selection()
        selection_range = selection.first()
        for i in range(selection_range.top(), selection_range.bottom() + 1):
            row_contents = []
            if not self.isRowHidden(i):
                for j in range(selection_range.left(), selection_range.right() + 1):
                    if not self.isColumnHidden(j):
                        row_contents.append(self.model().index(i, j).data())
            text = text + tab.join(str(cell_contents) for cell_contents in row_contents) + '\n'
        QApplication.clipboard().setText(text)

    def setItemDelegateForColumn(self, column, delegate):  # noqa: N802
        """Override of base class version so we can handle delegates on paste.

        Args:
            column (int): The column.
            delegate: The delegate.

        """
        if self.set_cbx_columns_in_model and self.model():
            if isinstance(delegate, QxCbxDelegate):
                col = column
                local_model = self.model()
                while isinstance(local_model, QAbstractProxyModel):
                    source_idx = local_model.mapToSource(local_model.index(0, col))
                    col = source_idx.column()
                    local_model = local_model.sourceModel()
                local_model.set_combobox_column(col, delegate.get_choices())

        # Call the base class
        super().setItemDelegateForColumn(column, delegate)

    def resize_height_to_contents(self):
        """Resize the table view height based on the number of rows."""
        vert_header = self.verticalHeader()
        count = vert_header.count()
        scrollbar_height = self.horizontalScrollBar().height()
        header_height = self.horizontalHeader().height()
        row_height = 0
        for i in range(count):
            if not vert_header.isSectionHidden(i):
                row_height += vert_header.sectionSize(i)
        self.setMinimumHeight(scrollbar_height + header_height + row_height)

    def setModel(self, model) -> None:  # noqa: N802
        """Overrides QTableView setModel so that we can do some extra stuff.

        Args:
            model (QAbstractItemModel): The model.
        """
        super().setModel(model)
        self._connect_show_combo_box_popup()
        if self.use_header_checkboxes:
            self._add_header_checkboxes()
            self.model().dataChanged.connect(self._on_data_changed)

    def _on_data_changed(self, top_left: QModelIndex, bottom_right: QModelIndex) -> None:
        """Called when the model sends the dataChanged signal.

        Args:
            top_left: Top left of range modified.
            bottom_right: Bottom right of range modified.
        """
        if self.use_header_checkboxes:
            update_header_from_data(self.model(), self._checkbox_header)

    def _add_header_checkboxes(self) -> None:
        """Adds checkboxes in the header for checkbox columns."""
        if hasattr(self.model(), 'get_checkbox_columns') and hasattr(self.model(), 'read_only_columns'):
            check_box_columns = self.model().get_checkbox_columns()
            self._checkbox_header = CheckBoxHeader(list(check_box_columns), parent=self)
            self._checkbox_header.set_read_only_columns(self.model().read_only_columns)
            self._checkbox_header.clicked.connect(self._on_horizontal_header_clicked)
            self.setHorizontalHeader(self._checkbox_header)
            update_header_from_data(self.model(), self._checkbox_header)

    def _on_horizontal_header_clicked(self, logical_index: int, state: Qt.CheckState) -> None:
        """Called if self.use_header_checkboxes is True and mouse is clicked in the header.

        Slot connected to 'clicked' signal emitted by CheckBoxHeader.

        Args:
            logical_index: The logical index of the section clicked.
            state: The state of the checkbox in the header.
        """
        update_data_from_header(logical_index, state, self.model())

    def _connect_show_combo_box_popup(self):
        """Connects the selectionChanged signal to the _show_combo_box_popup() slot."""
        if self.selectionModel():
            self.selectionModel().selectionChanged.connect(self._show_combo_box_popup)

    def _show_combo_box_popup(self, current, previous) -> None:
        """If attached to QTableView.selectionModel().selectionChanged signal, shows combo box delegate menu on click.

        We call QxTableView.edit() when a cell with a combo box is selected so the pop-up menu is shown immediately.
        See https://stackoverflow.com/questions/67431515/auto-expand-qcombobox-that-is-delegate-in-qtreeview. Otherwise
        it takes like 3 clicks to get it to show.

        You must connect the signal after setting up the model. Otherwise, QTableView.selectionModel() may be None.

        Args:
            current (QItemSelection): current index.
            previous (QItemSelection): previous index.
        """
        combo_box_delegate_columns = self._get_combo_box_delegate_columns()
        if combo_box_delegate_columns:
            indexes = current.indexes()
            if len(indexes) == 1:
                index = indexes[0]
                if index.column() in combo_box_delegate_columns:
                    self.edit(index)

    def _get_combo_box_delegate_columns(self) -> set[int] | None:
        """Returns a set of integers indicating the columns that have QxCbxDelegate delegates."""
        combo_box_delegate_columns = set()
        model = self.model()
        if model:
            for column in range(model.columnCount()):
                delegate = self.itemDelegateForColumn(column)
                if delegate and isinstance(delegate, QxCbxDelegate):
                    combo_box_delegate_columns.add(column)
        return combo_box_delegate_columns


def _ctrl_up(event: QKeyEvent) -> bool:
    """Return True if the key press event is CTRL+ up arrow key."""
    return event.key() == Qt.Key.Key_Up and event.modifiers() == Qt.KeyboardModifier.ControlModifier


def _ctrl_down(event: QKeyEvent) -> bool:
    """Return True if the key press event is CTRL+ down arrow key."""
    return event.key() == Qt.Key.Key_Down and event.modifiers() == Qt.KeyboardModifier.ControlModifier
