"""Worker thread for the solution importer."""
# 1. Standard python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.hecras._HECRAS_ImportSolution import HecrasSolutionReader


class SolutionLoadWorkerThread(QThread):
    """Worker thread for importing a single ADCIRC solution file."""
    processing_finished = Signal()

    def __init__(self):
        """Construct the worker."""
        super().__init__()
        self.reader = None
        self._logger = logging.getLogger('xms.hecras')

    def run(self):
        """Thread runner that exports the fort.14."""
        # Look for a mapped BC coverage first.
        self._logger.info('Loading HEC-RAS solution file...')
        try:
            # Read the solution file and send to XMS. Will be loaded on first compatible geometry found.
            self.reader = HecrasSolutionReader()
            self.reader.read()
        except Exception:
            self._logger.exception('Error(s) encountered while loading HEC-RAS solution files.')
        finally:
            self.processing_finished.emit()


def read_solution_file():  # pragma: no cover
    """Reads a HEC-RAS solution file.

    Triggered as a drag-n-drop read of a solution file into XMS.
    """
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)

    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope

    worker = SolutionLoadWorkerThread()
    display_text = {
        'title': 'Loading HEC-RAS Solution File',
        'working_prompt': 'Reading HEC-RAS solution file. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
        'success_prompt': 'Successfully read HEC-RAS solution and converted to XMDF format. Close this dialog to '
                          'load the data into SMS.',
        'note': '',
        'auto_load': 'Auto load solutions into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.hecras', worker, win_cont)
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:
        worker.reader.send()
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
