"""SimComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtCore import QDir, QProcess, Qt
from PySide2.QtWidgets import QApplication, QDialog, QFileDialog

# 3. Aquaveo modules
from xms.api.dmi import MenuItem, XmsEnvironment
from xms.guipy import file_io_util
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.hgs.components import dmi_util
from xms.hgs.components.hgs_component_base import HgsComponentBase
from xms.hgs.file_io import hgs_exporter
from xms.hgs.gui import run_progress_dialog
from xms.hgs.gui import solution_plots_window
from xms.hgs.gui.materials_dialog import MaterialsDialog
from xms.hgs.gui.sim_control_dialog import SimControlDialog
from xms.hgs.misc.settings import Settings
from xms.hgs.simulation_runner import sim_runner


class SimComponent(HgsComponentBase):
    """A Dynamic Model Interface (DMI) component for a HydroGeoSphere simulation."""

    last_used_export_dir = 'LAST_USED_EXPORT_DIR'

    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)

        self.main_filepath = Path(main_file) if main_file else None
        self._data = file_io_util.read_json_file(self.main_filepath)
        if main_file and not self.main_filepath.is_file():  # Create the file if it doesn't exist yet
            file_io_util.write_json_file({}, self.main_filepath)

    def save_to_location(self, new_path: str, save_type: str):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        rv = super().save_to_location(new_path, save_type)
        return rv

    def create_event(self, lock_state: bool):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        with open(self.main_filepath, 'w') as file:
            file.write('{}')

        return [], []

    @classmethod
    def get_last_used_export_directory(cls, main_filepath: Path) -> str:
        """Returns the last used directory where sim was exported (via 'Export' command) or '' if not found.

        Args:
            main_filepath (Path): This component's main file path.

        Returns:
            See description.
        """
        return Settings.get(main_filepath=main_filepath, variable=SimComponent.last_used_export_dir, default='')

    @classmethod
    def set_last_used_export_directory(cls, main_filepath: Path, last_used_dir: str) -> None:
        """Sets the last used directory where sim was exported (via 'Export' command).

        Args:
            main_filepath (Path): This component's main file path.
            last_used_dir (str): Path to directory that was used most recently.
        """
        Settings.set(main_filepath=main_filepath, variable=SimComponent.last_used_export_dir, value=last_used_dir)

    def get_project_explorer_menus(self, main_file_list: list[str]):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if not main_file_list or len(main_file_list) > 1:
            return []  # Multi-select or nothing selected

        menu_list: list['MenuItem | None'] = [None]  # None is interpreted as a spacer menu item
        self._add_tree_menu_item('Simulation Control...', '_open_dialog', menu_list)
        self._add_tree_menu_item('Materials...', '_materials_dialog', menu_list, 'materials.svg')
        self._add_tree_menu_item('Export...', '_export_dialog', menu_list, 'save.svg')
        self._add_tree_menu_item('Open Containing Folder', '_open_containing_folder', menu_list)
        menu_list.append(None)
        self._add_tree_menu_item('Run grok', '_run_grok', menu_list, 'model_run.svg')
        self._add_tree_menu_item('Run phgs', '_run_phgs', menu_list, 'model_run.svg')
        self._add_tree_menu_item('Run hsplot', '_run_hsplot', menu_list, 'model_run.svg')
        self._add_tree_menu_item('Solution Plots...', '_solution_plots', menu_list, 'plot.svg')

        menu_list.append(None)
        return menu_list

    def _run_grok(self, query, params, win_cont):
        """Runs grok.exe.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        run_progress_dialog.run(sim_runner.GROK_EXE, dmi_util.get_grok_filepath(query), win_cont)
        return [], []

    def _run_phgs(self, query, params, win_cont):
        """Runs phgs.exe.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        run_progress_dialog.run(sim_runner.PHGS_EXE, dmi_util.get_grok_filepath(query), win_cont)
        return [], []

    def _run_hsplot(self, query, params, win_cont):
        """Runs hsplot.exe.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        run_progress_dialog.run(sim_runner.HSPLOT_EXE, dmi_util.get_grok_filepath(query), win_cont)
        return [], []

    def _solution_plots(self, query, params, win_cont):
        """Displays the solution plots.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        xy_series_filepath = dmi_util.get_grok_filepath(query).with_name('xy_series.h5')
        solution_plots_window.run(xy_series_filepath, parent=None)  # parent=None fixes config dialog closing plot
        return [], []

    def _export_dialog(self, query, params, win_cont):
        """Opens a dialog with options on how and where to export the simulation.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (dict): Generic map of parameters. Unused in this case.
            win_cont (QWidget): The window container.
        """
        messages: list = []
        actions: list = []

        dir_ = self.get_last_used_export_directory(self.main_filepath)
        if not dir_:
            dir_ = QDir.homePath()
        dialog = QFileDialog(parent=win_cont, directory=dir_, caption='Export Simulation')
        if dialog.exec() == QDialog.Accepted:
            folder = Path(dialog.selectedFiles()[0]).parent
            SimComponent.set_last_used_export_directory(self.main_filepath, str(folder))
            hgs_exporter.HgsExporter.run_export(
                main_file=str(self.main_filepath), query=query, out_dir=folder, feedback=True
            )
        return messages, actions

    def _open_containing_folder(self, query, params, win_cont):
        """Opens the File Explorer to the .grok file directory, if it exists.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (dict): Generic map of parameters. Unused in this case.
            win_cont (QWidget): The window container.
        """
        messages: list = []
        actions: list = []
        grok_filepath = dmi_util.get_grok_filepath(query)
        if not XmsEnvironment.xms_environ_project_path():
            message = 'The project and simulation must first be saved.'
            message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        elif not grok_filepath.parent.is_dir():
            message = f'No simulation found at "{str(grok_filepath.parent)}". You must first save the simulation.'
            message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        else:
            # os.startfile(grok_filepath.parent) Causes Windows Explorer to appear behind GMS
            QProcess.startDetached('explorer', ['/select', ',', str(grok_filepath)])
        return messages, actions

    def _open_dialog(self, query, params, win_cont):
        """Opens the simulation model dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages: list = []
        actions: list = []

        dialog = SimControlDialog(self._data, query, win_cont)
        dialog.setModal(True)
        locked = False  # There's currently no way to lock / unlock the sim so I guess it's always unlocked
        if dialog.exec() == QDialog.Accepted and not locked:
            # Write the package
            QApplication.setOverrideCursor(Qt.WaitCursor)
            file_io_util.write_json_file(dialog.data, self.main_filepath)
            QApplication.restoreOverrideCursor()

        return messages, actions

    def _materials_dialog(self, query, params, win_cont):
        """Opens the materials dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        messages: list = []
        actions: list = []

        materials_filepath = self._materials_filepath()
        materials_data = file_io_util.read_json_file(materials_filepath)
        sim_data = file_io_util.read_json_file(self.main_filepath)
        dialog = MaterialsDialog(materials_data=materials_data, sim_data=sim_data, parent=win_cont)
        if dialog.exec() == QDialog.Accepted:
            file_io_util.write_json_file(dialog.data, materials_filepath)

        return messages, actions

    def _materials_filepath(self) -> Path:
        """Returns the path to the materials data file."""
        return self.main_filepath.with_name('materials.json')
