"""TecplotPointReader class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.xy.xy_series import XySeries

# 4. Local modules
from xms.hgs.file_io import file_io_util, tecplot_block_reader_base

# Type aliases
Locations = list[tuple[float, float, float]]  # list of xyz locations


class TecplotPointReader:
    """Reads a Tecplot ascii point file and makes XySeries.

    I assume the file format doesn't vary from the examples I've seen. If so, this will need to be made smarter.
    """
    def __init__(self, filepath: Path) -> None:
        """Initializes the file.

        Args:
            filepath (Path): File path.
        """
        self._filepath = filepath

        self._title = ''
        self._variable_names = []
        self._variable_units = {}
        self._columns = None

    def read(self) -> list[XySeries]:
        """Reads the file."""
        try:
            with self._filepath.open('r') as self._file:
                self._read_title()
                self._read_variables()
                self._variable_units, line = file_io_util.read_units(self._file)
                self._read_columns(line)
                return self._create_xy_series()
        except Exception:
            log = logging.getLogger('xms.hgs')
            log.warning(f'Could not read file: "{str(self._filepath)}".')
            return []

    def _read_title(self) -> None:
        """Reads the 'TITLE' line."""
        self._title, _ = tecplot_block_reader_base.read_title(['TITLE'], self._file)

    def _read_variables(self) -> None:
        """Reads the VARIABLES line."""
        self._variable_names, _ = tecplot_block_reader_base.read_variables(self._file)

    def _read_columns(self, line: str) -> None:
        """Reads the columns of data."""
        file_io_util.skip_to(self._file, ['ZONE'], line=line)
        line = next(self._file)
        line = file_io_util.skip_comments(self._file, line)
        column_count = len(self._variable_names)
        self._columns = [[] for _ in range(column_count)]  # list of columns
        try:
            while line:
                if line.upper().startswith('ZONE'):  # Must be a well file. We don't support reading this.
                    break
                values = [float(x) for x in line.split()]
                for i in range(column_count):
                    self._columns[i].append(values[i])
                line = next(self._file)  # FYI, don't mix use of next(file) with file.readline()
        except StopIteration:  # We've reached the end of the file (expected - not exceptional)
            return

    def _create_xy_series(self) -> list[XySeries]:
        """Returns XySeries for each variable."""
        xy_series = []
        for i in range(1, len(self._columns)):
            name = f'{self._filepath.name}|{self._title}-{self._variable_names[i]}'
            x = self._columns[0]
            y = self._columns[i]
            x_title = 'Time'
            y_title = self._variable_names[i]
            if self._variable_units:
                x_units = self._variable_units.get(1, '')
                if x_units:
                    x_title += f' {x_units}'
                y_units = self._variable_units.get(i + 1, '')
                if y_units:
                    y_title += f' {y_units}'
            xy_series.append(XySeries(x=x, y=y, name=name, x_title=x_title, y_title=y_title))
        return xy_series
