"""MaterialsDialog class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from enum import IntEnum
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QItemSelectionModel, QModelIndex
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QDialog, QFileDialog, QHeaderView, QToolBar

# 3. Aquaveo modules
from xms.guipy import file_io_util
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.resources import resources_util
from xms.guipy.settings import SettingsManager
from xms.guipy.testing import testing_tools
from xms.guipy.widgets import widget_builder
from xms.tool_core.table_definition import FloatColumnType, InputFileColumnType, TableDefinition

# 4. Local modules
from xms.hgs.data.domains import Domains
from xms.hgs.gui import gui_util
from xms.hgs.gui.gui_exchange import GuiExchange
from xms.hgs.gui.materials_dialog_ui import Ui_materials_dialog


class Columns(IntEnum):
    """An enumeration matching the columns in the widget and the XMS C++ enumeration."""
    NAME = 0
    DOMAIN = 1
    ZONES = 2
    UUID = 3


class ColumnNames:
    """An enumeration of the column names."""
    NAME = 'Name'
    DOMAIN = 'Domain'
    ZONES = 'Zones'
    UUID = 'Uuid'


def _tooltip_filepath() -> Path:
    """Returns the path to the tooltip file."""
    return Path(__file__).parent / 'materials_dialog_tooltips.json'


class MaterialsDialog(XmsDlg):
    """MaterialsDialog class."""

    # Shorter variable names for these icons
    INSERT_SVG = ':/resources/icons/row-insert.svg'
    ADD_SVG = ':/resources/icons/row-add.svg'
    DELETE_SVG = ':/resources/icons/row-delete.svg'
    MOVE_UP = ':/resources/icons/row-up.svg'
    MOVE_DOWN = ':/resources/icons/row-down.svg'

    def __init__(self, materials_data, sim_data, parent=None) -> None:
        """Initializes the class.

        Args:
            materials_data: Data dict of all material data.
            sim_data: Data dict of all material data.
            parent: Parent widget.
        """
        super().__init__(parent, 'xms.hgs.gui.materials_dialog')
        self.data = None  # materials_data
        self._sim_data = sim_data
        self.ui = Ui_materials_dialog()
        self.ui.setupUi(self)
        self._table_model: QxPandasTableModel | None = None
        self._toolbar: QToolBar | None = None
        self._actions: dict | None = None
        self._gx = GuiExchange(self.ui, data=self.data, tooltip_filepath=_tooltip_filepath())
        self._selected_uuid: str = ''
        self._domain_to_stacked_widget_index = {Domains.PM: 0, Domains.OLF: 1, Domains.ET: 2}
        self._opened_file: Path | None = None
        self._help_getter = gui_util.help_getter('xms.hgs.gui.materials_dialog')

        self._create_toolbar()
        self._setup_signals()
        self._setup_open_and_save_buttons()
        self._setup_with_materials_data(materials_data)

    def _setup_with_materials_data(self, materials_data):
        """Sets up the dialog given the materials data.

        Args:
            materials_data: Data dict of all material data.
        """
        self.data = materials_data
        self._gx.set_data(materials_data)

        self._setup_table()
        self._gx.enable_whats_this(self)
        self._gx.read_tooltips()
        self._select_initial_material()

    def _setup_signals(self):
        """Sets up the signals."""
        self.ui.btn_time_dependent_k_for_chosen_elements.clicked.connect(
            self._on_btn_time_dependent_k_for_chosen_elements
        )
        self.ui.btn_saturation_relative_k.clicked.connect(self._on_btn_saturation_relative_k)
        self.ui.btn_pressure_saturation.clicked.connect(self._on_btn_pressure_saturation)
        self.ui.btn_time_varying_friction.clicked.connect(self._on_btn_time_varying_friction)
        self.ui.btn_raster_filename.clicked.connect(self._on_btn_raster_filename)
        self.ui.btn_rdf_table.clicked.connect(self._on_btn_rdf_table)
        self.ui.btn_time_root_depth_table.clicked.connect(self._on_btn_time_root_depth_table)
        self.ui.btn_lai_tables.clicked.connect(self._on_btn_lai_tables)
        self.ui.btn_time_varying_lai_from_raster.clicked.connect(self._on_btn_time_varying_lai_from_raster)
        self.ui.cbx_unsaturated_tables_or_functions.currentTextChanged.connect(self._enable_unsaturated_widgets)
        self.ui.cbx_unsaturated_functions.currentTextChanged.connect(self._enable_unsaturated_widgets)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.btn_open.clicked.connect(self._on_btn_open)
        self.ui.btn_save.clicked.connect(self._on_btn_save)

    def _exchange_data(self, set_data: bool) -> None:
        """Either sets the data dict from the controls (set_data==True) or vice-versa (set_data==False).

        Args:
            set_data (bool): See description.
        """
        default_unknown = -999.0

        # Porous media
        # group = Domains.PM
        self._gx.do_rbt_edt(set_data, 'k isotropic', True, 'k isotropic', 7.438e-5)
        self._gx.do_rbt_edt(set_data, 'k anisotropic', False, 'kxx', 7.438e-5)
        self._gx.do_rbt_edt(set_data, 'k anisotropic', False, 'kyy', 7.438e-5)
        self._gx.do_rbt_edt(set_data, 'k anisotropic', False, 'kzz', 7.438e-5)
        self._gx.do_chk_btn(set_data, 'time dependent k for chosen elements', '')
        self._gx.do_chk_edt(set_data, 'specific storage', '', 1.0e-4)
        self._gx.do_chk_edt(set_data, 'porosity', '', 0.375)
        self._gx.do_cbx(set_data, 'unsaturated tables or functions')
        self._gx.do_chk_btn(set_data, 'saturation-relative k', '')
        self._gx.do_chk_btn(set_data, 'pressure-saturation', '')
        self._gx.do_cbx(set_data, 'unsaturated functions')
        self._gx.do_chk_edt(set_data, 'residual saturation', '', default_unknown)
        self._gx.do_chk_edt(set_data, 'alpha', '', 1.9)
        self._gx.do_chk_edt(set_data, 'beta', '', 6.0)
        self._gx.do_chk_edt(set_data, 'pore connectivity', '', 0.5)
        self._gx.do_chk_edt(set_data, 'air entry pressure', '', 1.9)
        self._gx.do_chk_edt(set_data, 'exponent', '', 2.83333)
        self._gx.do_chk_edt(set_data, 'minimum relative permeability', '', 0.01)
        self._gx.do_chk_edt(set_data, 'table smoothness factor', '', 0.001)
        self._gx.do_chk_edt(set_data, 'table minimum pressure', '', -1000.0)
        self._gx.do_chk_edt(set_data, 'table maximum s-k slope', '', 100.0)
        self._gx.do_chk(set_data, 'use tabulated unsaturated functions', True)

        # Surface flow
        # group = Domains.OLF
        self._gx.do_chk_edt(set_data, 'x friction', '', 0.0548)
        self._gx.do_chk_edt(set_data, 'x friction', '', 0.0548)
        self._gx.do_chk_edt(set_data, 'y friction', '', 0.0548)
        self._gx.do_chk_btn(set_data, 'time varying friction', '')
        self._gx.do_chk_edt(set_data, 'rill storage height', '', 1.0e-6)
        self._gx.do_chk_edt(set_data, 'obstruction storage height', '', 1.0e-6)
        self._gx.do_chk_edt(set_data, 'coupling length', '', 0.0001)
        self._gx.do_chk_edt(set_data, 'maximum flow depth', '', default_unknown)
        self._do_read_rill_storage_from_raster(set_data)

        # ET
        # group = Domains.ET
        self._gx.do_chk_edt(set_data, 'evaporation depth', '', 0.2)
        self._gx.do_chk(set_data, 'potential evaporation using transpiration')
        self._gx.do_chk(set_data, 'edf quadratic decay function')
        self._gx.do_chk_edt(set_data, 'root depth', '', default_unknown)
        self._gx.do_chk_btn(set_data, 'rdf table', '')
        self._gx.do_chk_btn(set_data, 'time-root depth table', '')
        self._gx.do_chk(set_data, 'rdf quadratic decay function')
        self._gx.do_chk_btn(set_data, 'lai tables', '')
        self._gx.do_chk_btn(set_data, 'time varying lai from raster', '')
        self._gx.do_chk_edt(set_data, 'transpiration fitting parameters', 'c_1', 0.5)
        self._gx.do_chk_edt(set_data, 'transpiration fitting parameters', 'c_2', 0.0)
        self._gx.do_chk_edt(set_data, 'transpiration fitting parameters', 'c_3', 1.0)
        self._gx.do_chk_edt(set_data, 'transpiration limiting pressure head', 'hwp_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'transpiration limiting pressure head', 'hfc_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'transpiration limiting pressure head', 'ho_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'transpiration limiting pressure head', 'han_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'evaporation limiting pressure head', 'he2_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'evaporation limiting pressure head', 'he1_et', default_unknown)
        self._gx.do_chk_edt(set_data, 'canopy storage parameter', '', 1.5)
        self._gx.do_chk_edt(set_data, 'canopy evaporation interval', '', 0.0)
        self._gx.do_chk_edt(set_data, 'initial interception storage', '', 1.5)

    def _setup_open_and_save_buttons(self):
        """Sets up the open button."""
        debug_file_exists = Path('C:/temp/debug_xmshgs_materials_dialog.dbg').is_file()
        self.ui.btn_open.setVisible(debug_file_exists)
        self.ui.btn_save.setVisible(debug_file_exists)
        self.ui.btn_save.setEnabled(False)

    def _on_btn_open(self):
        """Lets us open a materials.json file."""
        rv = QFileDialog.getOpenFileName(self, 'Materials json file', '', '')
        if rv and rv[0]:
            self._opened_file = Path(rv[0])
            materials_data = file_io_util.read_json_file(self._opened_file)
            self._setup_with_materials_data(materials_data)
            self.ui.btn_save.setEnabled(True)

    def _on_btn_save(self):
        """Saves to the opened materials.json file."""
        file_io_util.write_json_file(self.data, self._opened_file)

    def _do_read_rill_storage_from_raster(self, set_data: bool) -> None:
        """Handles data exchange for the 'read rill storage from raster' stuff."""
        self._gx.do_chk_edt(set_data, 'read rill storage from raster', 'raster_filename', '')
        self._gx.do_chk_edt(set_data, 'read rill storage from raster', 'scale_factor', 1.0)
        if not set_data:
            chk = self._gx.get_widget('chk', 'read rill storage from raster')
            btn = self._gx.get_widget('btn', 'raster_filename')
            lbl = self._gx.get_widget('lbl', 'scale_factor')
            edt = self._gx.get_widget('edt', 'scale factor')
            self._gx.make_enabling_lambda(chk, btn)
            self._gx.make_enabling_lambda(chk, lbl)
            self._gx.make_enabling_lambda(chk, edt)
            self._gx.add_tooltip(chk, 'read rill storage from raster')
            self._gx.add_tooltip(self._gx.get_widget('edt', 'raster filename'), 'raster filename')
            self._gx.add_tooltip(edt, 'scale factor')

    def _select_initial_material(self):
        """Selects the first material when opening the dialog."""
        if self._table_model.rowCount():
            idx = self._table_model.createIndex(0, 0)
            flags = QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows
            self.ui.tbl_materials.selectionModel().select(idx, flags)
            # That should trigger self._on_selection_changed() which will set the current material and load the data
        else:
            self._enable_widgets()

    def showEvent(self, event):  # noqa: N802 - function name should be lowercase
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        self._restore_splitter_geometry()

    def _save_splitter_geometry(self) -> None:
        """Save the current position of the splitter."""
        settings = SettingsManager()
        settings.save_setting('xms.hgs', f'{self._dlg_name}.splitter', self.ui.splitter.sizes())

    def _restore_splitter_geometry(self) -> None:
        """Restore the position of the splitter."""
        splitter = self._get_splitter_sizes()
        if not splitter:
            return
        splitter_sizes = [int(size) for size in splitter]
        self.ui.splitter.setSizes(splitter_sizes)

    def _get_splitter_sizes(self):
        """Returns a list of the splitter sizes that are saved in the registry."""
        settings = SettingsManager()
        splitter = settings.get_setting('xms.hgs', f'{self._dlg_name}.splitter')
        return splitter

    def _setup_table(self) -> None:
        """Initializes the table of materials."""
        widget_builder.style_table_view(self.ui.tbl_materials)
        gui_util.setup_table_context_menus(self.ui.tbl_materials, self._on_index_column_click, self._on_right_click)

        # Setup the model
        empty_table: dict[str, list[str]] = {
            ColumnNames.NAME: [],
            ColumnNames.DOMAIN: [],
            ColumnNames.ZONES: [],
            ColumnNames.UUID: []
        }
        table_dict = self.data.get('table', empty_table)
        df = pd.DataFrame(table_dict)
        df.index += 1
        self._table_model = QxPandasTableModel(df)
        self.ui.tbl_materials.setModel(self._table_model)
        self._table_model.dataChanged.connect(self._on_data_changed)
        self._table_model.set_default_values(
            {
                ColumnNames.NAME: 'material_1',
                ColumnNames.DOMAIN: Domains.PM,
                ColumnNames.ZONES: '1',
                ColumnNames.UUID: ''
            }
        )

        # Add tool tips for column headings
        tool_tips_dict = {Columns.ZONES: 'Space separated list of zones (e.g. "1 2 3")'}
        self._table_model.set_horizontal_header_tooltips(tool_tips_dict)

        # Appearance
        self.ui.tbl_materials.setColumnHidden(Columns.UUID, True)
        self._add_combo_box_delegates()
        self.ui.tbl_materials.horizontalHeader().setSectionResizeMode(Columns.NAME, QHeaderView.ResizeToContents)
        self.ui.tbl_materials.horizontalHeader().setSectionResizeMode(Columns.DOMAIN, QHeaderView.ResizeToContents)
        self.ui.tbl_materials.horizontalHeader().setStretchLastSection(True)

        # signal
        self.ui.tbl_materials.selectionModel().selectionChanged.connect(self._on_selection_changed)

    def _add_combo_box_delegates(self) -> None:
        """Adds a combo box to the Domain column."""
        delegate = QxCbxDelegate(self)
        # delegate = ComboDelegate(self)
        delegate.set_strings(self._get_domain_combo_box_strings())
        self.ui.tbl_materials.setItemDelegateForColumn(Columns.DOMAIN, delegate)

    def _get_domain_combo_box_strings(self) -> list[str]:
        """Returns the list of strings that will be in the domain combo box."""
        domain_strings = []
        domain_strings.append(Domains.PM)  # This one should always exist
        if self._sim_data.get('chk_surface_flow', False):
            domain_strings.append(Domains.OLF)
        if self._sim_data.get('chk_et', False):
            domain_strings.append(Domains.ET)
        return domain_strings

    def _on_index_column_click(self, point) -> None:
        """Called on a right-click event in the index column (vertical header).

        Args:
            point (QPoint): The point clicked
        """
        row = self.ui.tbl_materials.verticalHeader().logicalIndexAt(point)
        self.ui.tbl_materials.selectRow(row)
        menu_list = [
            ['row-insert', 'Insert', self._on_btn_insert],
            ['row-delete', 'Delete', self._on_btn_delete],
            ['copy', 'Copy', self.ui.tbl_materials.on_copy],
            ['paste', 'Paste', self.ui.tbl_materials.on_paste],
        ]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_materials.viewport().mapToGlobal(point))

    def _on_right_click(self, point) -> None:
        """Slot called when user right-clicks in the table.

        Args:
            point(QPoint): The point clicked.
        """
        # row = self.ui.table_view.logicalIndexAt(point)
        menu_list = [
            ['copy', 'Copy', self.ui.tbl_materials.on_copy], ['paste', 'Paste', self.ui.tbl_materials.on_paste]
        ]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_materials.viewport().mapToGlobal(point))

    def _get_unique_sorted_selected_rows(self) -> list[int]:
        """Returns the set of selected row numbers (0-based), in order from least to greatest."""
        selected_list = self.ui.tbl_materials.selectedIndexes()
        return gui_util.get_unique_sorted_selected_rows(selected_list)

    def _reselect_rows(self, selected_rows: list[int]) -> None:
        """Selects the rows in the table that were selected before as indicated by selected_rows.

        Args:
            selected_rows: List of rows.
        """
        for row in selected_rows:
            if row >= self._table_model.rowCount():
                row = self._table_model.rowCount() - 1
            idx = self._table_model.createIndex(row, 0)
            flags = QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows
            self.ui.tbl_materials.selectionModel().select(idx, flags)

    def _create_toolbar(self) -> None:
        """Creates the toolbar."""
        self._toolbar = QToolBar(self)
        button_list = [
            [MaterialsDialog.INSERT_SVG, 'Insert Row', self._on_btn_insert],
            [MaterialsDialog.ADD_SVG, 'Add Row', self._on_btn_add],
            [MaterialsDialog.DELETE_SVG, 'Delete Row', self._on_btn_delete],
            [MaterialsDialog.MOVE_UP, 'Move Up', self._on_btn_up],
            [MaterialsDialog.MOVE_DOWN, 'Move Down', self._on_btn_down]
        ]
        self._actions = widget_builder.setup_toolbar(self._toolbar, button_list)
        self.ui.hlay_tbl_buttons.insertWidget(0, self._toolbar)

    def _find_unique_name(self, old_names) -> str:
        """Returns a unique name like 'material_5' and adds it to old_names."""
        i = 1
        while True:
            name = f'material_{i}'
            if name not in old_names:
                old_names.add(name)
                return name
            i += 1

    def _initialize_new_materials(self):
        """Make sure the new materials have unique uuids and names.

        Do this right after adding new materials before syncing the data dict with the table.
        """
        # Get all the old names
        old_names = set()
        for row in range(self._table_model.rowCount()):
            uuid = self._table_model.data(self._table_model.index(row, Columns.UUID))
            if uuid:
                old_names.add(self._table_model.data(self._table_model.index(row, Columns.NAME)))

        # Assign new unique uuids and unique names where needed
        for row in range(self._table_model.rowCount()):
            uuid = self._table_model.data(self._table_model.index(row, Columns.UUID))
            if not uuid:  # New material
                self._table_model.setData(self._table_model.index(row, Columns.UUID), testing_tools.new_uuid())
                name = self._table_model.data(self._table_model.index(row, Columns.NAME))
                if name in old_names:
                    unique_name = self._find_unique_name(old_names)
                    self._table_model.setData(self._table_model.index(row, Columns.NAME), unique_name)

    def _sync_data_with_model(self):
        # Get the table.
        new_dict = {'table': self._table_model.data_frame.to_dict('list')}

        # Add the materials dicts
        if ColumnNames.UUID in new_dict['table']:
            for uuid_ in new_dict['table'][ColumnNames.UUID]:
                if uuid_ in self.data:
                    new_dict[uuid_] = self.data[uuid_]
                else:
                    new_dict[uuid_] = {}
        self.data = new_dict

    def _move_row(self, up, selected_row, selected_column):
        """Moves a row up or down.

        Note: This implementation only works with DataFrames that have a 1-based sequential integer Index. If your
        DataFrame is not structured this way, you will need to override this method.

        Args:
            up (bool): True if moving up, else False
            selected_row (int): Selected row.
            selected_column (int): Selected column.
        """
        source_row = selected_row
        source_idx = source_row + 1  # Assuming a 1-based sequential integer pandas.Index
        dest_row = source_row - 1 if up else source_row + 1
        dest_idx = dest_row + 1  # Assuming a 1-based sequential integer pandas.Index

        self._table_model.swap_rows(source_idx, dest_idx, source_row, dest_row)

        # Update the selection
        new_index = self._table_model.index(dest_row, selected_column)
        self.ui.tbl_materials.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )

    def _on_btn_up(self) -> None:
        """Moves the row up."""
        selected_rows = self._get_unique_sorted_selected_rows()
        if len(selected_rows) == 1 and selected_rows[0] > 0:
            selected_list = self.ui.tbl_materials.selectedIndexes()
            self._move_row(up=True, selected_row=selected_list[0].row(), selected_column=selected_list[0].column())

    def _on_btn_down(self) -> None:
        """Moves the row down."""
        selected_rows = self._get_unique_sorted_selected_rows()
        if len(selected_rows) == 1 and selected_rows[0] < self._table_model.rowCount() - 1:
            selected_list = self.ui.tbl_materials.selectedIndexes()
            self._move_row(up=False, selected_row=selected_list[0].row(), selected_column=selected_list[0].column())

    def _on_btn_insert(self) -> None:
        """Called when the Insert button is clicked. Inserts rows in the table."""
        selected_rows = self._get_unique_sorted_selected_rows()
        if gui_util.rows_are_contiguous(selected_rows):
            self._table_model.insertRows(row=selected_rows[0], count=len(selected_rows))
        else:
            for row in reversed(selected_rows):
                self._table_model.insertRows(row=row, count=1)
        self._initialize_new_materials()
        self._sync_data_with_model()
        self._reselect_rows(selected_rows)

    def _on_btn_add(self) -> None:
        """Called when the Add button is clicked."""
        selected_rows = self._get_unique_sorted_selected_rows()
        if not selected_rows:
            self._table_model.insertRows(row=self._table_model.rowCount(), count=1)
        else:
            for row in reversed(selected_rows):
                self._table_model.insertRows(row=row + 1, count=1)
        self._initialize_new_materials()
        self._sync_data_with_model()
        selected_rows = [0] if not selected_rows else [row + 1 for row in selected_rows]
        self._reselect_rows(selected_rows)
        self._enable_widgets()

    def _on_btn_delete(self) -> None:
        """Called when the Delete button is clicked."""
        selected_rows = self._get_unique_sorted_selected_rows()
        for row in reversed(selected_rows):
            self._table_model.removeRows(row=row, count=1)
        self._sync_data_with_model()
        self._reselect_rows(selected_rows)
        self._enable_widgets()

    def _enable_toolbar(self):
        """Enables and disables things."""
        selected_list = self.ui.tbl_materials.selectedIndexes()
        selections_exist = len(selected_list) > 0
        selected_rows = self._get_unique_sorted_selected_rows()
        self._toolbar.widgetForAction(self._actions[MaterialsDialog.INSERT_SVG]).setEnabled(selections_exist)
        self._toolbar.widgetForAction(self._actions[MaterialsDialog.DELETE_SVG]).setEnabled(selections_exist)
        self._toolbar.widgetForAction(self._actions[MaterialsDialog.MOVE_UP]).setEnabled(len(selected_rows) == 1)
        self._toolbar.widgetForAction(self._actions[MaterialsDialog.MOVE_DOWN]).setEnabled(len(selected_rows) == 1)

    def _enable_unsaturated_widgets(self):
        """Enables and disables things."""
        # Porous media - unsaturated tables stuff
        tables = (self.ui.cbx_unsaturated_tables_or_functions.currentText() == 'Tables')
        # Enabling/disabling the group box enables/disables everything in it, which is handy
        self.ui.grp_unsaturated_tables.setEnabled(tables)
        self.ui.grp_unsaturated_functions.setEnabled(not tables)

        brooks_corey = self.ui.cbx_unsaturated_functions.currentText() == 'brooks-corey'
        self.ui.chk_alpha.setEnabled(not tables and not brooks_corey)
        self.ui.chk_air_entry_pressure.setEnabled(not tables and brooks_corey)
        self.ui.chk_exponent.setEnabled(not tables and brooks_corey)

    def _enable_widgets(self):
        material_selected = bool(self._selected_uuid)
        self.ui.grp_porous_media.setEnabled(material_selected)
        self.ui.grp_surface_flow.setEnabled(material_selected)
        self.ui.grp_et.setEnabled(material_selected)
        self._enable_toolbar()
        self._enable_unsaturated_widgets()

    def _on_selection_changed(self, selected, deselected) -> None:
        """Called when the user clicks in the table and changes what cells are selected.

        Args:
            selected (QItemSelection): What is now selected.
            deselected (QItemSelection): What used to be selected.
        """
        del selected  # Unused parameter
        del deselected  # Unused parameter

        # Save and update the values
        self._save_values(self._selected_uuid)
        self._selected_uuid, name, domain = self._get_selected_material_info()
        if self._selected_uuid:
            self._load_values(self._selected_uuid)
            self._show_domain_widgets(domain)
        self._update_material_name_label(name)
        self._enable_widgets()

    def _update_material_name_label(self, material_name: str) -> None:
        """Updates the material name label.

        Args:
            material_name (str): Name of the material
        """
        self.ui.txt_material.setText(f'Material: {material_name}')

    def _run_table_dialog(self, title: str, table_definition: TableDefinition, button: str) -> None:
        """Runs the TableDialog.

        Args:
            title (str): Window title.
            table_definition (TableDefinition): Defines the table (column types, fixed row count or not).
            button (str): Name of the button widget.
        """
        json_str = self._gx.get_data().get(button, '')
        if json_str and 'schema' not in json_str:
            json_str = pd.read_json(json_str).to_json(orient='table')  # Switch string to new format
        json_str = gui_util.run_table_dialog(title, table_definition, json_str, self, button)
        if json_str is not None:
            self._gx.get_data()[button] = json_str

    def _on_btn_time_dependent_k_for_chosen_elements(self):
        """Called when the button is clicked."""
        table_def = TableDefinition(
            [
                FloatColumnType(header='Time', tool_tip='Time', default=0.0),
                FloatColumnType(header='Kxx', tool_tip='Kxx', default=0.0),
                FloatColumnType(header='Kyy', tool_tip='Kyy', default=0.0),
                FloatColumnType(header='Kzz', tool_tip='Kzz', default=0.0),
            ],
        )
        self._run_table_dialog('Time Dependent K', table_def, 'btn_time_dependent_k_for_chosen_elements')

    def _on_btn_raster_filename(self):
        """Called when the button is clicked."""
        starting_file = self._gx.get_data().get('edt_raster_filename', '')
        rv = QFileDialog.getOpenFileName(self, 'Raster File', starting_file, '')
        self._gx.get_data()['edt_raster_filename'] = rv[0]
        self.ui.edt_raster_filename.setText(rv[0])

    def _on_btn_saturation_relative_k(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_saturation_relative_k', 'saturation-relative k', 'saturation', 'rel_perm')

    def _on_btn_pressure_saturation(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_pressure_saturation', 'pressure-saturation', 'pressure', 'saturation')

    def _on_btn_time_varying_friction(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_time_varying_friction', 'time varying friction', 'time', 'nmann')

    def _on_btn_rdf_table(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_rdf_table', 'rdf table', 'depth', 'density')

    def _on_btn_time_root_depth_table(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_time_root_depth_table', 'time-root depth table', 'time', 'root_depth')

    def _on_btn_lai_tables(self):
        """Called when the button is clicked."""
        self._run_xy_series_dialog('btn_lai_tables', 'lai tables', 'time', 'lai')

    def _on_btn_time_varying_lai_from_raster(self):
        """Called when the button is clicked."""
        table_def = TableDefinition(
            [
                FloatColumnType(header='Time', default=0.0, tool_tip='Time'),
                InputFileColumnType(header='Raster', default='', file_filter='*.*', tool_tip='Raster file')
            ]
        )
        self._run_table_dialog('Time Varying Lai From Raster', table_def, 'btn_time_varying_lai_from_raster')

    def _run_xy_series_dialog(self, widget_name: str, title: str, column1: str, column2: str) -> None:
        """Runs the XySeriesEditor."""
        df = self._tuple_data_to_data_frame(widget_name, column1, column2)
        gms_icon = QIcon(resources_util.get_resource_path(':/resources/icons/gms.ico'))
        dialog = XySeriesEditor(
            data_frame=df,
            series_name=title.capitalize(),
            icon=gms_icon,
            parent=self,
            readonly_x=False,
            can_add_rows=True,
            stair_step=False
        )
        if dialog.exec() == QDialog.Accepted:
            self._save_xy_series_dialog_data_to_tuples(dialog, widget_name)

    def _tuple_data_to_data_frame(self, widget_name: str, column1: str, column2: str) -> pd.DataFrame:
        """Returns a dataframe made from the list of tuples identified by name."""
        list_of_tuples = self._gx.get_data().get(widget_name, [])
        df_dict = {column1: [v1 for v1, _ in list_of_tuples], column2: [v2 for _, v2 in list_of_tuples]}
        df = pd.DataFrame(data=df_dict)
        df.index += 1  # Start index column at 1, not 0
        return df

    def _save_xy_series_dialog_data_to_tuples(self, dialog: XySeriesEditor, widget_name: str) -> None:
        """Saves the xy data in the XySeriesEditor dialog to our data dict as a list of tuples."""
        column1 = dialog.model.data_frame.iloc[:, 0].tolist()
        column2 = dialog.model.data_frame.iloc[:, 1].tolist()
        list_of_tuples = [(v1, v2) for v1, v2 in zip(column1, column2)]
        self._gx.get_data()[widget_name] = list_of_tuples

    def _get_selected_material_info(self) -> tuple[str, str, str]:
        """Returns the uuid of the material currently selected."""
        selected_rows = self._get_unique_sorted_selected_rows()
        if not selected_rows:
            return '', '', ''
        uuid_ = self._get_table_data(selected_rows[0], Columns.UUID)
        name = self._get_table_data(selected_rows[0], Columns.NAME)
        domain = self._get_table_data(selected_rows[0], Columns.DOMAIN)
        return uuid_, name, domain

    def _get_table_data(self, row: int, column: int) -> str:
        """Returns the value in the cell at row, column."""
        idx = self._table_model.createIndex(row, column)
        return self._table_model.data(idx)

    def _save_values(self, selected_uuid: str) -> None:
        """Save the properties for the previously selected material."""
        if selected_uuid:
            self._exchange_data(set_data=True)

    def _load_values(self, selected_uuid: str) -> None:
        """Load the gui widgets with the properties for the newly selected material."""
        if selected_uuid:
            data = self.data.get(selected_uuid, {})
            self._gx.set_data(data)
            self._exchange_data(set_data=False)

    def _show_domain_widgets(self, domain: str) -> None:
        """Displays the widgets that go with the domain."""
        self.ui.stacked_widget.setCurrentIndex(self._domain_to_stacked_widget_index[domain])

    def _on_data_changed(self, top_left_index: QModelIndex, bottom_right_index: QModelIndex) -> None:
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        same_row = top_left_index.row() == bottom_right_index.row()
        same_column = top_left_index.column() == bottom_right_index.column()
        if same_row and same_column and top_left_index.column() == Columns.DOMAIN:
            self._show_domain_widgets(self._get_table_data(top_left_index.row(), Columns.DOMAIN))
            self._update_material_name_label(self._get_table_data(top_left_index.row(), Columns.NAME))

    def accept(self) -> None:
        """Called on OK."""
        self._save_values(self._selected_uuid)
        self._sync_data_with_model()
        self._save_splitter_geometry()
        super().accept()

    def reject(self) -> None:
        """Called when the Cancel button is clicked."""
        self._save_splitter_geometry()
        super().reject()
